﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/srepo/srepo_Result.h>
#include <nn/srepo/detail/srepo_ApiDetail.h>
#include <nn/srepo/detail/srepo_SystemReportGenerator.h>
#include <nn/srepo/detail/msgpack/srepo_MessagePack.h>
#include <nn/util/util_Endian.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace srepo { namespace detail {

void SystemReportGenerator::Initialize(size_t* outPosition, Bit8* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, ReportBufferSizeMin);

    msgpack::OutputStreamParam stream = {buffer, bufferSize, 0};
    msgpack::WriteMap16(&stream, 0);

    *outPosition = stream.position;
}

nn::Result SystemReportGenerator::AddKey(size_t* outPosition,
    const char* key, Bit8* buffer, size_t bufferSize, size_t position) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(key);

    size_t keyLength = 0;
    NN_RESULT_THROW_UNLESS(VerifyKey(&keyLength, key), ResultInvalidKey());

    NN_RESULT_THROW_UNLESS(!CheckKeyDuplication(buffer, position, key), ResultDuplicatedKey());

    NN_RESULT_THROW_UNLESS(GetKeyValueCount(buffer, bufferSize) < KeyValueCountMax, ResultKeyValueCountLimitReached());

    msgpack::OutputStreamParam stream = {buffer, bufferSize, position};

    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, key, static_cast<uint32_t>(keyLength)), ResultOutOfResource());

    *outPosition = stream.position;

    NN_RESULT_SUCCESS;
}

nn::Result SystemReportGenerator::AddKeyValue(size_t* outPosition,
    const char* key, int64_t value, Bit8* buffer, size_t bufferSize, size_t position) NN_NOEXCEPT
{
    size_t valuePosition = 0;
    NN_RESULT_DO(AddKey(&valuePosition, key, buffer, bufferSize, position));

    msgpack::OutputStreamParam stream = {buffer, bufferSize, valuePosition};

    NN_RESULT_THROW_UNLESS(msgpack::WriteSignedIntegerAutoSize(&stream, value), ResultOutOfResource());

    IncrementKeyValueCount(buffer, bufferSize);

    *outPosition = stream.position;

    NN_RESULT_SUCCESS;
}

nn::Result SystemReportGenerator::AddKeyValue(size_t* outPosition,
    const char* key, const Any64BitId& value, Bit8* buffer, size_t bufferSize, size_t position) NN_NOEXCEPT
{
    size_t valuePosition = 0;
    NN_RESULT_DO(AddKey(&valuePosition, key, buffer, bufferSize, position));

    msgpack::OutputStreamParam stream = {buffer, bufferSize, valuePosition};

    Bit64 id;
    nn::util::StoreBigEndian(&id, value.id);

    NN_RESULT_THROW_UNLESS(msgpack::WriteExtension(&stream, MsgpackExtensionType_Any64BitId, &id, 8), ResultOutOfResource());

    IncrementKeyValueCount(buffer, bufferSize);

    *outPosition = stream.position;

    NN_RESULT_SUCCESS;
}

nn::Result SystemReportGenerator::AddKeyValue(size_t* outPosition,
    const char* key, double value, Bit8* buffer, size_t bufferSize, size_t position) NN_NOEXCEPT
{
    size_t valuePosition = 0;
    NN_RESULT_DO(AddKey(&valuePosition, key, buffer, bufferSize, position));

    msgpack::OutputStreamParam stream = {buffer, bufferSize, valuePosition};

    NN_RESULT_THROW_UNLESS(msgpack::WriteFloat64(&stream, value), ResultOutOfResource());

    IncrementKeyValueCount(buffer, bufferSize);

    *outPosition = stream.position;

    NN_RESULT_SUCCESS;
}

nn::Result SystemReportGenerator::AddKeyValue(size_t* outPosition,
    const char* key, const char* value, Bit8* buffer, size_t bufferSize, size_t position) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(value);

    size_t valueLength = 0;
    NN_RESULT_THROW_UNLESS(VerifyStringValue(&valueLength, value), ResultInvalidValue());

    size_t valuePosition = 0;
    NN_RESULT_DO(AddKey(&valuePosition, key, buffer, bufferSize, position));

    msgpack::OutputStreamParam stream = {buffer, bufferSize, valuePosition};

    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, value, static_cast<uint32_t>(valueLength)), ResultOutOfResource());

    IncrementKeyValueCount(buffer, bufferSize);

    *outPosition = stream.position;

    NN_RESULT_SUCCESS;
}

nn::Result SystemReportGenerator::AddKeyValue(size_t* outPosition,
    const char* key, const void* value, size_t size, Bit8* buffer, size_t bufferSize, size_t position) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(VerifyBinaryValue(value, size), ResultInvalidValue());

    size_t valuePosition = 0;
    NN_RESULT_DO(AddKey(&valuePosition, key, buffer, bufferSize, position));

    msgpack::OutputStreamParam stream = {buffer, bufferSize, valuePosition};

    NN_RESULT_THROW_UNLESS(msgpack::WriteBinary(&stream, value, static_cast<uint32_t>(size)), ResultOutOfResource());

    IncrementKeyValueCount(buffer, bufferSize);

    *outPosition = stream.position;

    NN_RESULT_SUCCESS;
}

void SystemReportGenerator::IncrementKeyValueCount(Bit8* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(buffer[0], msgpack::DataType_Map16);
    NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, ReportBufferSizeMin);

    if (bufferSize < ReportBufferSizeMin)
    {
        return;
    }

    uint16_t count = nn::util::LoadBigEndian(reinterpret_cast<uint16_t*>(&buffer[1]));

    nn::util::StoreBigEndian(reinterpret_cast<uint16_t*>(&buffer[1]), ++count);
}

int SystemReportGenerator::GetKeyValueCount(const Bit8* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(buffer[0], msgpack::DataType_Map16);
    NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, ReportBufferSizeMin);

    if (bufferSize < ReportBufferSizeMin)
    {
        return 0;
    }

    return nn::util::LoadBigEndian(reinterpret_cast<const uint16_t*>(&buffer[1]));
}

bool SystemReportGenerator::CheckKeyDuplication(const Bit8* data, size_t dataSize, const char* key) NN_NOEXCEPT
{
    msgpack::InputStreamParam stream = {data, dataSize};
    msgpack::AnyData any = {};

    msgpack::ReadCurrent(&any, &stream);

    NN_SDK_ASSERT_EQUAL(any.type, msgpack::AnyDataType_Map);

    while (stream.GetRemainSize() > 0)
    {
        msgpack::ReadCurrent(&any, &stream);

        if (any.type != msgpack::AnyDataType_String)
        {
            NN_SDK_ASSERT(false, "No string data.");
            return true;
        }

        // 大文字小文字は区別しない。
        if (nn::util::Strnicmp(key, reinterpret_cast<const char*>(&stream.data[any.string.position]),
                static_cast<int>(any.string.length)) == 0 && key[any.string.length] == '\0')
        {
            return true;
        }

        // バリューを読み飛ばす。
        if (!msgpack::ReadCurrent(&any, &stream))
        {
            NN_SDK_ASSERT(false, "The next value has been corrupted.");
            return true;
        }
    }

    return false;
}

}}}
