﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/srepo/detail/service/core/srepo_DurationReportGenerator.h>
#include <nn/srepo/detail/srepo_SystemReportGenerator.h>
#include <nn/srepo/detail/service/core/srepo_ReportBuffer.h>

namespace nn { namespace srepo { namespace detail { namespace service { namespace core {

#if defined(NN_DETAIL_SREPO_DURATION_REPORT_TRACE_ENABLED)
#define NN_DETAIL_SREPO_DURATION_REPORT_TRACE NN_DETAIL_SREPO_TRACE
#else
#define NN_DETAIL_SREPO_DURATION_REPORT_TRACE(...) (void)0
#endif

const char* DurationReportGenerator::GetKindName(SystemDurationKind kind) const NN_NOEXCEPT
{
    static const char* Names [] =
    {
        "network",
        "notification",
        "fg_program",
        "operation_mode",
        "power",
        "controller",
    };

    return Names[static_cast<int>(kind)];
}

const char* DurationReportGenerator::GetKindName(UserDurationKind kind) const NN_NOEXCEPT
{
    static const char* Names [] =
    {
        "account",
        "friend_presence",
    };

    return Names[static_cast<int>(kind)];
}

const char* DurationReportGenerator::GetStateName(SystemDurationKind kind) const NN_NOEXCEPT
{
    static const char* Names [] =
    {
        "network_state",
        "notification_state",
        "fg_program_state",
        "operation_mode_state",
        "power_state",
        "controller_state",
    };

    return Names[static_cast<int>(kind)];
}

const char* DurationReportGenerator::GetStateName(UserDurationKind kind) const NN_NOEXCEPT
{
    static const char* Names [] =
    {
        "account_state",
        "friend_presence_state",
    };

    return Names[static_cast<int>(kind)];
}

const char* DurationReportGenerator::GetDurationName(SystemDurationKind kind) const NN_NOEXCEPT
{
    static const char* Names [] =
    {
        "network_sec",
        "notification_sec",
        "fg_program_sec",
        "operation_mode_sec",
        "power_sec",
        "controller_sec",
    };

    return Names[static_cast<int>(kind)];
}

const char* DurationReportGenerator::GetDurationName(UserDurationKind kind) const NN_NOEXCEPT
{
    static const char* Names [] =
    {
        "account_sec",
        "friend_presence_sec",
    };

    return Names[static_cast<int>(kind)];
}

void DurationReportGenerator::CreateHeader(
    Header* pOut,
    nn::TimeSpanType current,
    const TriggerString& trigger) const NN_NOEXCEPT
{
    pOut->version = m_Version;
    pOut->bootId = m_BootId;
    pOut->stableDuration = nn::TimeSpan(current) - m_PreviousReportedTime;
    pOut->upTime = current;
    pOut->trigger = trigger;
}

void DurationReportGenerator::GenerateTriggerString(
    TriggerString* pOut,
    const char* kindName,
    const SomeValueBase& newState,
    const SomeValueBase& oldState) const NN_NOEXCEPT
{
    pOut->type = kindName;
    pOut->oldState.Set(&oldState);
    pOut->newState.Set(&newState);
}

void DurationReportGenerator::GenerateTriggerString(
    TriggerString* pOut,
    const char* kindName,
    const SomeValue<ControllerStatus>& newState,
    const SomeValue<ControllerStatus>& oldState) const NN_NOEXCEPT
{
    // type     : "controller(total,rail)"
    // oldState : "1,0"
    // newState : "1,1"

    NN_UNUSED(kindName);

    pOut->type = "controller(total,rail)";
    pOut->oldState.Set(&oldState);
    pOut->newState.Set(&newState);
}

nn::Result DurationReportGenerator::BuildReportBuffer(
    ReportOutput* pOutReportOutput,
    const nn::TimeSpan& current,
    const Header& header,
    const nn::account::Uid* pUidList,
    int uidCount,
    const uint64_t* pSeqNoList,
    int seqNoCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pUidList);
    NN_SDK_ASSERT_EQUAL(uidCount, seqNoCount);
    NN_UNUSED(seqNoCount);

    if(pOutReportOutput)
    {
        pOutReportOutput->count = uidCount;
    }

    if(uidCount == 0)
    {
        NN_RESULT_SUCCESS;
    }

    size_t position;
    SystemReportGenerator::Initialize(&position, m_ReportBuffer, sizeof(m_ReportBuffer));

    // システム情報のレポート作成
    NN_RESULT_DO(BuildSystemReportBuffer(&position, m_ReportBuffer, sizeof(m_ReportBuffer), position, current, header));

    // システム情報レポートは各ユーザで同一なので、構築したバッファを使いまわす
    const size_t UserReportStartPosition = position;

    auto GetMapObjectsCount = [&](int* pOut, const nn::Bit8* pBuffer, size_t size) NN_NOEXCEPT -> nn::Result
    {
        msgpack::AnyData any;
        msgpack::InputStreamParam stream = {pBuffer, size, 0};

        NN_RESULT_THROW_UNLESS(msgpack::ReadCurrent(&any, &stream), nn::srepo::ResultInvalidReportData());
        NN_RESULT_THROW_UNLESS(any.type == msgpack::AnyDataType_Map, nn::srepo::ResultInvalidReportData());

        *pOut = static_cast<int>(any.num);
        NN_RESULT_SUCCESS;
    };

    int systemReportMapObjCount;
    NN_RESULT_DO(GetMapObjectsCount(&systemReportMapObjCount, m_ReportBuffer, sizeof(m_ReportBuffer)));

    for(int i = 0 ; i < uidCount ; i++)
    {
        // ユーザー向けレポートの Key,Value 追加のたびに
        // マップ内オブジェクト数が増えていくので systemReportMapObjCount で固定する.
        msgpack::OutputStreamParam stream = {m_ReportBuffer, sizeof(m_ReportBuffer), 0};
        msgpack::WriteMap16(&stream, static_cast<uint16_t>(systemReportMapObjCount));

        const auto& uid = pUidList[i];
        const uint64_t seqNo = pSeqNoList[i];

        // ユーザ1人分のレポート作成
        NN_RESULT_DO(BuildUserReportBuffer(&position, m_ReportBuffer, sizeof(m_ReportBuffer), UserReportStartPosition, current, uid, seqNo));

        NN_RESULT_DO(PushReport(uid, m_ReportBuffer, position));

        if(pOutReportOutput != nullptr) // 主にテスト向けにバッファを出力する
        {
            pOutReportOutput->values[i].Set(uid, m_ReportBuffer, position);
        }
    }

    NN_RESULT_SUCCESS;
}

namespace
{
    class ReportGeneratorImpl
    {
    private:
        nn::Bit8* m_pBuffer;
        const size_t m_Size;
        size_t m_Position;

    public:

        ReportGeneratorImpl(nn::Bit8* pBuffer, size_t size, size_t position) NN_NOEXCEPT
            : m_pBuffer(pBuffer)
            , m_Size(size)
            , m_Position(position)
        {
        }

        size_t GetPosition() const NN_NOEXCEPT
        {
            return m_Position;
        }

        nn::Result AddKeyValue(const char* key, int64_t value) NN_NOEXCEPT
        {
            NN_DETAIL_SREPO_DURATION_REPORT_TRACE("%30s | %lld\n", key, value);
            return SystemReportGenerator::AddKeyValue(&m_Position, key, value, m_pBuffer, m_Size, m_Position);
        }
        nn::Result AddKeyValue(const char* key, uint64_t value) NN_NOEXCEPT
        {
            NN_DETAIL_SREPO_DURATION_REPORT_TRACE("%30s | %016llx\n", key, value);
            nn::srepo::Any64BitId id = { static_cast<Bit64>(value) };
            return SystemReportGenerator::AddKeyValue(&m_Position, key, id, m_pBuffer, m_Size, m_Position);
        }
        nn::Result AddKeyValue(const char* key, const char* value) NN_NOEXCEPT
        {
            NN_DETAIL_SREPO_DURATION_REPORT_TRACE("%30s | \"%s\"\n", key, value);
            return SystemReportGenerator::AddKeyValue(&m_Position, key, value, m_pBuffer, m_Size, m_Position);
        }
        nn::Result AddKeyValue(const char* key, const nn::account::Uid& uid) NN_NOEXCEPT
        {
            NN_DETAIL_SREPO_DURATION_REPORT_TRACE("%30s | %016llx-%016llx\n", key, uid._data[0], uid._data[1]);

            // ここでレポートに載せなくても付加される

            NN_UNUSED(key);
            NN_UNUSED(uid);

            NN_RESULT_SUCCESS;
        }
        nn::Result AddSomeState(const char* stateName, const SomeValueBase& value) NN_NOEXCEPT
        {
            if(value.GetTag() == SomeValue<ControllerStatus>::Tag)
            {
                return AddSomeState(stateName, static_cast<const SomeValue<ControllerStatus>&>(value).GetRef());
            }
            else if(value.GetTag() == SomeValue<bool>::Tag)
            {
                return AddSomeState(stateName, static_cast<const SomeValue<bool>&>(value).GetRef());
            }

            SomeValueString stateString(&value);

            NN_DETAIL_SREPO_DURATION_REPORT_TRACE("%30s | \"%s\"\n", stateName, stateString.Get());
            return SystemReportGenerator::AddKeyValue(&m_Position, stateName, stateString.Get(), m_pBuffer, m_Size, m_Position);
        }
        nn::Result AddDuration(const char* durationName, const nn::TimeSpan& value) NN_NOEXCEPT
        {
            const int64_t seconds = nn::TimeSpan(value).GetSeconds();

            NN_DETAIL_SREPO_DURATION_REPORT_TRACE("%30s | %lld\n", durationName, seconds);
            return SystemReportGenerator::AddKeyValue(&m_Position, durationName, seconds, m_pBuffer, m_Size, m_Position);
        }

        // ControllerStatus専用
        nn::Result AddSomeState(const char* stateName, const ControllerStatus& value) NN_NOEXCEPT
        {
            // total と rail でそれぞれ key, value を追加する

            NN_UNUSED(stateName);

            NN_DETAIL_SREPO_DURATION_REPORT_TRACE("%30s | %d\n", "controller_total", value.totalNum);
            NN_RESULT_DO(SystemReportGenerator::AddKeyValue(&m_Position, "controller_total", static_cast<int64_t>(value.totalNum), m_pBuffer, m_Size, m_Position));

            NN_DETAIL_SREPO_DURATION_REPORT_TRACE("%30s | %d\n", "controller_rail", value.railNum);
            NN_RESULT_DO(SystemReportGenerator::AddKeyValue(&m_Position, "controller_rail", static_cast<int64_t>(value.railNum), m_pBuffer, m_Size, m_Position));

            NN_RESULT_SUCCESS;
        }

        // bool 専用
        nn::Result AddSomeState(const char* stateName, const bool& value) NN_NOEXCEPT
        {
            // true:1, false:0 として integer で書き込む

            const int64_t TrueValue = 1LL;
            const int64_t FalseValue = 0LL;

            NN_DETAIL_SREPO_DURATION_REPORT_TRACE("%30s | %d\n", stateName, value ? TrueValue : FalseValue);
            return SystemReportGenerator::AddKeyValue(&m_Position, stateName, value ? TrueValue : FalseValue, m_pBuffer, m_Size, m_Position);
        }
    };
}

nn::Result DurationReportGenerator::BuildSystemReportBuffer(
    size_t* pOutSize,
    nn::Bit8* pBuffer, size_t size, size_t startPosition,
    const nn::TimeSpan& current,
    const Header& header) const NN_NOEXCEPT
{
    NN_DETAIL_SREPO_DURATION_REPORT_TRACE("---- System ------------------------------------------\n");
    ReportGeneratorImpl impl(pBuffer, size, startPosition);

    // ヘッダのレポート作成
    NN_RESULT_DO( impl.AddKeyValue("version", header.version) );
    NN_RESULT_DO( impl.AddKeyValue("boot_id", header.bootId) );
    NN_RESULT_DO( impl.AddKeyValue("up_time_sec", header.upTime.GetSeconds()) );
    NN_RESULT_DO( impl.AddKeyValue("stable_duration_sec", header.stableDuration.GetSeconds()) );
    NN_RESULT_DO( impl.AddKeyValue("trigger_state_type", header.trigger.type) );
    NN_RESULT_DO( impl.AddKeyValue("trigger_state_old", header.trigger.oldState.Get()) );
    NN_RESULT_DO( impl.AddKeyValue("trigger_state_new", header.trigger.newState.Get()) );

    // システム情報のレポート作成
    StateDuration systemDurations[SystemStateChangeManager::StateDurationCount];
    m_SystemStateChangeManager.GenerateStateDurationArray(systemDurations, current);

    for(int i = 0 ; i < SystemStateChangeManager::StateDurationCount ; i++)
    {
        auto kind = static_cast<SystemDurationKind>(i);
        auto&& value = systemDurations[i];
        NN_RESULT_DO( impl.AddSomeState(GetStateName(kind), *value.pStateValue) );
        NN_RESULT_DO( impl.AddDuration(GetDurationName(kind), value.duration) );
    }

    *pOutSize = impl.GetPosition();
    NN_DETAIL_SREPO_DURATION_REPORT_TRACE("--------------------------------- total : %zu byte \n", impl.GetPosition());

    NN_RESULT_SUCCESS;
}

nn::Result DurationReportGenerator::BuildUserReportBuffer(
    size_t* pOutSize,
    nn::Bit8* pBuffer, size_t size, size_t startPosition,
    const nn::TimeSpan& current,
    const nn::account::Uid& uid,
    uint64_t seqNo) const NN_NOEXCEPT
{
    // ユーザー情報取得
    StateDuration userDurations[OneUserStateChangeManager::StateDurationCount];
    m_MultiUserDurationManager.GenerateStateDurationArray(userDurations, uid, current);

    // ユーザー情報のレポート作成
    NN_DETAIL_SREPO_DURATION_REPORT_TRACE("---- uid %016llx-%016llx ----\n", uid._data[0], uid._data[1]);

    ReportGeneratorImpl impl(pBuffer, size, startPosition);

    NN_RESULT_DO( impl.AddKeyValue("seq_no", seqNo) );

    for(int i = 0 ; i < OneUserStateChangeManager::StateDurationCount ; i++)
    {
        auto kind = static_cast<UserDurationKind>(i);
        auto&& value = userDurations[i];
        NN_RESULT_DO( impl.AddSomeState(GetStateName(kind), *value.pStateValue) );
        NN_RESULT_DO( impl.AddDuration(GetDurationName(kind), value.duration) );
    }
    *pOutSize = impl.GetPosition();
    NN_DETAIL_SREPO_DURATION_REPORT_TRACE("--------------------------------- total : %zu byte \n", impl.GetPosition());

    NN_RESULT_SUCCESS;
}

nn::Result DurationReportGenerator::PushReport(const nn::account::Uid& uid, nn::Bit8* pBuffer, size_t size) const NN_NOEXCEPT
{
    const nn::ApplicationId appId = { 0x0100000000001024 }; // SIGLO-84627 にて発行

    return ReportBuffer::GetInstance().Push(
        ReportCategory_Normal, uid, "state_change_event", appId, pBuffer, size);
}

}}}}}
