﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/bpc/bpc_BoardPowerControl.h>
#include <nn/bpc/bpc_WakeupConfigSystem.h>
#include <nn/gpio/gpio.h>
#include <nn/gpio/gpio_PadAccessorDev.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_SystemProcess.h>
#include <nn/xcd/xcd_Sleep.h>
#include <nn/spsm/detail/spsm_Log.h>
#include "spsm_WakeReason.h"
#include "spsm_SettingsHolder.h"
#include "spsm_Debug.h"

namespace nn { namespace spsm { namespace server {

namespace {
    nn::spsm::WakeReasonFlagSet g_LastWakeReasonFlagSet;
    bool g_IsWakeReasonEmulated = false;
    bool g_IsWakeReasonFlagSetUpdateRequired = false;
}

    void IntializeWakeReason() NN_NOEXCEPT
    {
        nn::gpio::Initialize();

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::bpc::InitializeBoardPowerControl();
        nn::psm::Initialize();

        if(SettingsHolder::GetInstance().GetWakePinDebugMode() != 0)
        {
            nn::gpio::SetWakePinDebugMode(static_cast<nn::gpio::WakePinDebugMode>(SettingsHolder::GetInstance().GetWakePinDebugMode()));
        }
#endif
    }

    // -------------------------------------------------------------------------------------
    // アグリゲートされたウェイク要因フラグの取得

    void RequireUpdateWakeReasonFlagSet(bool require) NN_NOEXCEPT
    {
        g_IsWakeReasonFlagSetUpdateRequired = require;
    }

    void UpdateWakeReasonFlagSet() NN_NOEXCEPT
    {
        if ( g_IsWakeReasonEmulated || !g_IsWakeReasonFlagSetUpdateRequired )
        {
            return;
        }
        g_IsWakeReasonFlagSetUpdateRequired = false;

        auto wakeReason = nn::spsm::WakeReasonFlagSet();

#if defined(NN_BUILD_CONFIG_SPEC_NX)
        if (nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_PmuIrq))
        {
            #if defined(NN_BUILD_CONFIG_OS_HORIZON)
            nn::bpc::WakeupReason bpcWakeReason;
            nn::bpc::GetWakeupReason(&bpcWakeReason);
            if (bpcWakeReason & nn::bpc::WakeupReasonPowerButton)
            {
                NN_DETAIL_SPSM_INFO("Wake reason: PowerButtonPressed\n");
                wakeReason |= WakeReasonFlag::PowerButtonPressed::Mask;
            }
            // XXX: SDEV では PMIC の ACOK 割り込みは立たず、しかも素早い挿抜で起きると ON/OFF 両方立つ場合やどちらも立たない場合がある
            // 現状信用できないので、GpioPadName_CradleIrq を代わりに使うことにしてこちらは無効化
            #if 0
            if (bpcWakeReason & nn::bpc::WakeupReasonAcOkOn)
            {
                NN_DETAIL_SPSM_INFO("Wake reason: AcOkOn\n");
                wakeReason |= WakeReasonFlag::AcOk::Mask;
            }
            if (bpcWakeReason & nn::bpc::WakeupReasonAcOkOff)
            {
                NN_DETAIL_SPSM_INFO("Wake reason: AcOkOff\n");
                wakeReason |= WakeReasonFlag::AcOk::Mask;
            }
            #endif
            if (bpcWakeReason & nn::bpc::WakeupReasonRtc)
            {
                NN_DETAIL_SPSM_INFO("Wake reason: Rtc\n");
                switch ( nn::bpc::GetLastEnabledWakeupTimerType() )
                {
                    case nn::bpc::WakeupTimerType_None:
                    {
                        // TORIAEZU
                        NN_DETAIL_SPSM_WARN("Invalid timer type. Falling to background task case\n");
                        wakeReason |= WakeReasonFlag::RtcForBackgroundTask::Mask;
                        break;
                    }
                    case nn::bpc::WakeupTimerType_FullWakeup:
                    {
                        wakeReason |= WakeReasonFlag::RtcForFullWakeup::Mask;
                        break;
                    }
                    case nn::bpc::WakeupTimerType_BackgroundTask:
                    {
                        wakeReason |= WakeReasonFlag::RtcForBackgroundTask::Mask;
                        break;
                    }
                } // NOLINT(style/switch_default)
            }
            #endif
        }
        if (nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_CradleIrq))
        {
            // PMIC 側の AcOk が前述の問題を抱えているので、こちらのシグナルで判定
            // TORIAEZU: 現時点で抜きで起きたのか差しで起きたのかまで知る必要はなく、フラグは共通
            NN_DETAIL_SPSM_INFO("Wake reason: GpioPadName_CradleIrq\n");
            wakeReason |= WakeReasonFlag::AcOk::Mask;
        }
        if (nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_BattMgicIrq))
        {
            NN_DETAIL_SPSM_INFO("Wake reason: GpioPadName_BattMgicIrq\n");

            auto state = nn::psm::GetBatteryVoltageState();
            NN_DETAIL_SPSM_INFO("Wake reason: Battery Voltage State %d\n", state);

            if ( state == nn::psm::BatteryVoltageState_ShutdownRequired )
            {
                NN_DETAIL_SPSM_INFO("Wake reason: BatteryLevelTooLow\n");
                wakeReason |= WakeReasonFlag::BatteryLevelTooLow::Mask;
            }
            else
            {
                NN_DETAIL_SPSM_INFO("Wake reason: BatteryChargerUpdateRequired (FGIC)\n");
                wakeReason |= WakeReasonFlag::BatteryChargerUpdateRequired::Mask;
            }
        }
        if (nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_Bq24190Irq))
        {
            NN_DETAIL_SPSM_INFO("Wake reason: BatteryChargerUpdateRequired (ChargerIC)\n");
            wakeReason |= WakeReasonFlag::BatteryChargerUpdateRequired::Mask;
        }
        if (nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_BtWakeAp))
        {
            NN_DETAIL_SPSM_INFO("Wake reason: BluetoothActivityDetect\n");
            wakeReason |= WakeReasonFlag::BluetoothActivityDetect::Mask;
        }
        if (nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_WifiWakeHost))
        {
            NN_DETAIL_SPSM_INFO("Wake reason: WiFiActivityDetect\n");
            wakeReason |= WakeReasonFlag::WiFiActivityDetect::Mask;
        }
        if (nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_SdCd))
        {
            NN_DETAIL_SPSM_INFO("Wake reason: SdCardDetect\n");
            wakeReason |= WakeReasonFlag::SdCardDetect::Mask;
        }
        if (nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_GameCardCd))
        {
            NN_DETAIL_SPSM_INFO("Wake reason: GameCardDetect\n");
            wakeReason |= WakeReasonFlag::GameCardDetect::Mask;
        }
        // Detect 時は CTS も同時に入るため、Detect が立っていた時は CTS は無視する
        if (nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_ExtconDetS))
        {
            NN_DETAIL_SPSM_INFO("Wake reason: ControllerDetectL\n");
            wakeReason |= WakeReasonFlag::ControllerDetectL::Mask;
        }
        else if (nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_ExtUart3Cts))
        {
            // ControllerL は CTS による要因は BatteryLevelChange のみ
            NN_DETAIL_SPSM_INFO("Wake reason: ControllerBatteryLevelChangeL\n");
            wakeReason |= WakeReasonFlag::ControllerBatteryLevelChangeL::Mask;
        }
        // Detect 時は CTS も同時に入るため、Detect が立っていた時は CTS は無視する
        if (nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_ExtconDetU))
        {
            NN_DETAIL_SPSM_INFO("Wake reason: ControllerDetectR\n");
            wakeReason |= WakeReasonFlag::ControllerDetectR::Mask;
        }
        else if (nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_ExtUart2Cts))
        {

            if(nn::xcd::GetAwakeTriggerReasonForRightRail() == nn::xcd::AwakeTriggerReason_HomePressed)
            {
                NN_DETAIL_SPSM_INFO("Wake reason: ControllerButtonR\n");
                wakeReason |= WakeReasonFlag::ControllerButtonR::Mask;
            }
            else
            {
                // 明示的に HOME ボタン押しの要因が取れなかったときは、BatteryLevelChange と判断
                NN_DETAIL_SPSM_INFO("Wake reason: ControllerBatteryLevelChangeR\n");
                wakeReason |= WakeReasonFlag::ControllerBatteryLevelChangeR::Mask;
            }
        }
#else // defined(NN_BUILD_CONFIG_SPEC_NX)
        wakeReason |= WakeReasonFlag::PowerButtonPressed::Mask; // Generic 等での暫定措置
#endif

        g_LastWakeReasonFlagSet = wakeReason;
    } // NOLINT(impl/function_size)

    WakeReasonFlagSet GetWakeReasonFlagSet() NN_NOEXCEPT
    {
        return g_LastWakeReasonFlagSet;
    }

    void SetWakeReasonFlagSet(const WakeReasonFlagSet& flagSet) NN_NOEXCEPT
    {
        g_LastWakeReasonFlagSet = flagSet;
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    void SetEmulatedWakeReasonFlagSet(const WakeReasonFlagSet& flagSet) NN_NOEXCEPT
    {
        g_LastWakeReasonFlagSet = flagSet;
        g_IsWakeReasonEmulated = flagSet.IsAnyOn() ? true : false;
    }
#endif

}}}
