﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/spsm/detail/spsm_Log.h>
#include "spsm_SettingsHolder.h"

namespace nn { namespace spsm { namespace server {

namespace {

    //! Firmware Debug Settings を読む。
    template <typename T>
    void ReadSetting(T* pOutValue, const char* pName, const char* pSettingsName) NN_NOEXCEPT
    {
        size_t targetSize = ::nn::settings::fwdbg::GetSettingsItemValueSize(pSettingsName, pName);
        if ( 0 == targetSize )
        {
            NN_DETAIL_SPSM_WARN("Failed to read %s.%s (Value is not written)\n", pSettingsName, pName);
            return;
        }
        else if ( sizeof(T) != targetSize )
        {
            NN_DETAIL_SPSM_WARN("Failed to read %s.%s (data size does not match). expected=%d, actual=%d\n", pSettingsName, pName, sizeof(T), targetSize);
            return;
        }

        T temporaryValue;
        if ( sizeof(T) != (::nn::settings::fwdbg::GetSettingsItemValue(&temporaryValue, sizeof(T), pSettingsName, pName)) )
        {
            NN_DETAIL_SPSM_WARN("Failed to read %s.%s.\n", pSettingsName, pName);
        }
        else
        {
            *pOutValue = temporaryValue;
        }
    }

}

    void SettingsHolder::LoadStaticConfiguration() NN_NOEXCEPT
    {
        ReadSetting(&m_HasBattery, "has_battery", "platformconfig");
        ReadSetting(&m_WakePinDebugMode, "wake_pin_debug_mode", "systemsleep");
    }

    void SettingsHolder::LoadTransitionBehaviorSettings() NN_NOEXCEPT
    {
        int transitionTimeoutSec = 0;
        ReadSetting(&transitionTimeoutSec, "transition_timeout_sec", "systemsleep");
        m_TransitionTimeoutTime = nn::TimeSpan::FromSeconds(transitionTimeoutSec);
    }

    void SettingsHolder::LoadSleepReadyBehaviorSettings() NN_NOEXCEPT
    {
        ReadSetting(&m_EnterSc7Flag, "enter_sc7", "systemsleep");
    }

    void SettingsHolder::LoadSleepModeSettings() NN_NOEXCEPT
    {
        ReadSetting(&m_EnterSleepFlag, "enter_sleep", "systemsleep");
    }

    void SettingsHolder::LoadFullAwakeBehaviorSettings() NN_NOEXCEPT
    {
        int hushTimeAfterBriefPowerButtonPressMs = 1000;
        ReadSetting(&hushTimeAfterBriefPowerButtonPressMs, "hush_time_after_brief_power_button_press_ms", "systemsleep");
        m_HushTimeAfterBriefPowerButtonPress = nn::TimeSpan::FromMilliSeconds(hushTimeAfterBriefPowerButtonPressMs);
    }

}}}

