﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/spsm/spsm_PowerStateTypes.h>
#include <nn/spsm/server/spsm_PowerStateMessageQueue.h>
#include <nn/spsm/server/spsm_ServerTypes.h>
#include "spsm_PowerStateHandlerContext.h"
#include "spsm_NotificationMessageQueue.h"

namespace nn { namespace spsm { namespace server {

    class PowerStateMachine
    {
        NN_DISALLOW_COPY(PowerStateMachine);
        NN_DISALLOW_MOVE(PowerStateMachine);

    private: // Singleton
        PowerStateMachine() NN_NOEXCEPT;

    public:
        static PowerStateMachine& GetInstance() NN_NOEXCEPT
        {
            static PowerStateMachine inst;
            return inst;
        }

        void Initialize(const InitializeMode mode) NN_NOEXCEPT;
        PowerStateMessageQueue* GetPowerStateMessageQueue() NN_NOEXCEPT;
        NotificationMessageQueue* GetNotificationMessageQueue() NN_NOEXCEPT;
        PowerState GetCurrentState() const NN_NOEXCEPT;
        void AddStateChangeObserver(StateChangeObserver* pObserver) NN_NOEXCEPT;
        void RemoveStateChangeObserver(StateChangeObserver* pObserver) NN_NOEXCEPT;
        Result EnterSleep(os::SystemEvent** ppOutAwakeEvent) NN_NOEXCEPT;
        Result Shutdown(bool reboot) NN_NOEXCEPT;
        Result PutErrorState() NN_NOEXCEPT;
        void LoopStateMachine() NN_NOEXCEPT;

    private:
        void ProcessMessage(PowerStateMessageWithMeta messageWithMeta) NN_NOEXCEPT;
        void ProcessStateTransition() NN_NOEXCEPT;
        void UpdateStateInfoOnReport() NN_NOEXCEPT;

    private:
        // ステート遷移中に確保されるミューテックス
        // 遷移していない状態での処理を行うには、これを取得する
        nn::os::Mutex               m_StateTransitionMutex;

        // ステートマシンを駆動するメッセージキュー
        PowerStateMessageQueue      m_PowerStateMessageQueue;

        // 上位レイヤに伝達するためのメッセージキュー
        NotificationMessageQueue    m_NotificationMessageQueue;

        // 現在のステートマシンのステート
        PowerState                  m_CurrentState;

        // ステートハンドラ向けの各種コンテキスト情報
        PowerStateHandlerContext    m_Context;

        // システムが復帰した際に通知するイベント（EnterSleep を呼び出したクライアントに NativeHandle が渡される）
        nn::os::SystemEvent         m_AwakeEvent;

        // 状態遷移オブザーバのリスト
        // Add / Remove 操作時には m_StateTransitionMutex の取得が必要
        StateChangeObserverList                m_ObserverList;

        // 上位からエラーステートに遷移指示が出たときに立てるフラグ。これが立つと強制的にステート遷移を停止する
        bool                        m_IsOnError;
    };

}}}
