﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/spsm/spsm_Debug.h>
#include <nn/spsm/detail/spsm_Log.h>
#include "spsm_Debug.h"
#include "spsm_EventLog.h"

namespace nn { namespace spsm { namespace server {

namespace {
    LogItemList        g_LogItems;
}

    const LogItemList& GetLog() NN_NOEXCEPT
    {
        return g_LogItems;
    }

    void ResetLog() NN_NOEXCEPT
    {
        g_LogItems.clear();
    }

    void LogPscControl(nn::psc::PmState pmState, nn::psc::PmTransitionOrder order, nn::psc::PmFlagSet flags) NN_NOEXCEPT
    {
        g_LogItems.emplace_back(pmState, order, flags, nn::os::GetSystemTick());
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        NN_DETAIL_SPSM_INFO("LogPscControl: pmState=%d, order=%s, flags=0x%x\n",
            pmState,
            (order == nn::psc::PmTransitionOrder_ToHigherPowerState) ? "ToHigher" : "ToLower",
            flags);
#endif
    }

    void LogPowerStateHandler(nn::spsm::PowerState state, nn::spsm::PowerState newDestState, PowerStateHandlerType handlerType, PowerStateMessage message) NN_NOEXCEPT
    {
        g_LogItems.emplace_back(state, newDestState, handlerType, message, nn::os::GetSystemTick());
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        NN_DETAIL_SPSM_INFO("LogPowerStateHandler: state=%s, newDestState=%s, handlerType=%s, message=%s\n",
            GetPowerStateNameString(state),
            GetPowerStateNameString(newDestState),
            GetPowerStateHandlerTypeNameString(handlerType),
            GetPowerStateMessageNameString(message));
#endif
    }

    void LogSc7Entry() NN_NOEXCEPT
    {
        g_LogItems.emplace_back(LogType_Sc7Entry, nn::os::GetSystemTick());
    }

    void LogSc7Exit() NN_NOEXCEPT
    {
        g_LogItems.emplace_back(LogType_Sc7Exit, nn::os::GetSystemTick());
    }

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    const char* GetPowerStateHandlerTypeNameString(nn::spsm::server::PowerStateHandlerType handlerType) NN_NOEXCEPT
    {
        #define NN_DETAIL_SPSM_RETURN_CASE_LABEL_STRING(name) case PowerStateHandlerType_##name: { return NN_MACRO_STRINGIZE(name); }

        switch (handlerType)
        {
            NN_DETAIL_SPSM_RETURN_CASE_LABEL_STRING(Entry)
            NN_DETAIL_SPSM_RETURN_CASE_LABEL_STRING(Exit)
            NN_DETAIL_SPSM_RETURN_CASE_LABEL_STRING(OnDestination)
            NN_DETAIL_SPSM_RETURN_CASE_LABEL_STRING(OnMessage)
            default: NN_UNEXPECTED_DEFAULT;
        }

        #undef NN_DETAIL_SPSM_RETURN_CASE_LABEL_STRING
    }
#endif

    void AnalyzeLogForLastSleepWakeSequence(SleepWakeSequenceAnalyzedData* pData) NN_NOEXCEPT
    {
        pData->timeStampSleepTrigger = 0LL;
        pData->timeStampFullAwakeExit = 0LL;
        pData->timeStampMinimumAwakeDownwardDone = 0LL;
        pData->timeStampSleepReadyDone = 0LL;
        pData->timeStampEssentialServicesSleepReadyDone = 0LL;
        pData->timeStampSc7Entry = 0LL;
        pData->timeStampSc7Exit = 0LL;
        pData->timeStampEssentialServicesAwakeDone = 0LL;
        pData->timeStampMinimumAwakeUpwardDone = 0LL;
        pData->timeStampFullAwakeDone = 0LL;

        for ( size_t i = 0; i < g_LogItems.size(); ++i )
        {
            const auto& log = g_LogItems[i];

            if ( log.type == LogType_PowerStateHandler &&
                log.param.powerStateHandler.state == PowerState_FullAwake &&
                log.param.powerStateHandler.handlerType == PowerStateHandlerType_OnMessage &&
                ( log.param.powerStateHandler.message == PowerStateMessage_EventAutoPowerDownTimerExpired ||
                  log.param.powerStateHandler.message == PowerStateMessage_EventPowerButtonPressedBriefly ||
                   log.param.powerStateHandler.message == PowerStateMessage_EventSleepRequiredByLowBattery ||
                   log.param.powerStateHandler.message == PowerStateMessage_EventSleepRequiredByHighTemperature ) )
            {
                if ( pData->timeStampSleepTrigger == 0LL )
                {
                    pData->timeStampSleepTrigger = log.tick.GetInt64Value();
                }
            }
            else if ( log.type == LogType_PowerStateHandler &&
                log.param.powerStateHandler.state == PowerState_FullAwake &&
                log.param.powerStateHandler.handlerType == PowerStateHandlerType_Exit )
            {
                if ( pData->timeStampFullAwakeExit == 0LL )
                {
                    pData->timeStampFullAwakeExit = log.tick.GetInt64Value();
                }
            }
            else if ( log.type == LogType_PscControl &&
                log.param.pscControl.pmState == nn::psc::PmState_MinimumAwake &&
                log.param.pscControl.order == nn::psc::PmTransitionOrder_ToLowerPowerState )
            {
                if ( pData->timeStampMinimumAwakeDownwardDone == 0LL )
                {
                    pData->timeStampMinimumAwakeDownwardDone = log.tick.GetInt64Value();
                }

            }
            else if ( log.type == LogType_PscControl &&
                log.param.pscControl.pmState == nn::psc::PmState_SleepReady )
            {
                if ( pData->timeStampSleepReadyDone == 0LL )
                {
                    pData->timeStampSleepReadyDone = log.tick.GetInt64Value();
                }
            }
            else if ( log.type == LogType_PscControl &&
                log.param.pscControl.pmState == nn::psc::PmState_EssentialServicesSleepReady )
            {
                if ( pData->timeStampEssentialServicesSleepReadyDone == 0LL )
                {
                    pData->timeStampEssentialServicesSleepReadyDone = log.tick.GetInt64Value();
                }
            }
            else if ( log.type == LogType_Sc7Entry )
            {
                if ( pData->timeStampSc7Entry == 0LL )
                {
                    pData->timeStampSc7Entry = log.tick.GetInt64Value();
                }

            }
            else if ( log.type == LogType_Sc7Exit )
            {
                if ( pData->timeStampSc7Exit == 0LL )
                {
                    pData->timeStampSc7Exit = log.tick.GetInt64Value();
                }

            }
            else if ( log.type == LogType_PscControl &&
                log.param.pscControl.pmState == nn::psc::PmState_EssentialServicesAwake )
            {
                if ( pData->timeStampEssentialServicesAwakeDone == 0LL )
                {
                    pData->timeStampEssentialServicesAwakeDone = log.tick.GetInt64Value();
                }

            }
            else if ( log.type == LogType_PscControl &&
                log.param.pscControl.pmState == nn::psc::PmState_MinimumAwake &&
                log.param.pscControl.order == nn::psc::PmTransitionOrder_ToHigherPowerState )
            {
                if ( pData->timeStampMinimumAwakeUpwardDone == 0LL )
                {
                    pData->timeStampMinimumAwakeUpwardDone = log.tick.GetInt64Value();
                }

            }
            else if ( log.type == LogType_PscControl &&
                log.param.pscControl.pmState == nn::psc::PmState_FullAwake &&
                log.param.pscControl.order == nn::psc::PmTransitionOrder_ToHigherPowerState )
            {
                if ( pData->timeStampFullAwakeDone == 0LL )
                {
                    pData->timeStampFullAwakeDone = log.tick.GetInt64Value();
                }

            }
        }
    } // NOLINT(impl/function_size)
}}}

