﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/erpt.h>
#include <nn/err.h>
#include <nn/err/detail/err_ErrorCodeConvert.h>
#include <nn/spsm/spsm_ResultPrivate.h>
#include <nn/spsm/detail/spsm_TransitionInfo.h>

#include "spsm_ErrorReporter.h"

namespace nn { namespace spsm { namespace server {

namespace {

    nn::Result ConvertModuleTransitionInformationToErrorReportFields(
        uint8_t* pOutPscInitializedList,
        uint32_t* pOutPscCurrentPmStateList,
        uint32_t* pOutPscNextPmStateList,
        int maxModuleCount,
        const nn::psc::util::ModuleTransitionInformation *pModuleTransitionInformation,
        int moduleCount) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(moduleCount <= maxModuleCount, nn::spsm::ResultInvalidPscTransitionInfo());

        for ( auto i = 0; i < moduleCount; ++i )
        {
            auto& p = pModuleTransitionInformation[i];
            auto id = p.id;
            NN_RESULT_THROW_UNLESS(0 <= id && id < maxModuleCount, nn::spsm::ResultInvalidPscTransitionInfo());
            pOutPscInitializedList[id] = p.initialized ? 1 : 0;
            pOutPscCurrentPmStateList[id] = static_cast<uint32_t>(p.currentState);
            pOutPscNextPmStateList[id] = static_cast<uint32_t>(p.nextState);
        }

        NN_RESULT_SUCCESS;
    }

} // anonymous namespace

    nn::Result UpdateErrorReportContext(PowerState current, PowerState previous, PowerState destination) NN_NOEXCEPT
    {
        nn::erpt::Context context(nn::erpt::SystemPowerStateInfo);

        NN_RESULT_DO(context.Add(nn::erpt::FieldId::CurrentSystemPowerState, static_cast<uint32_t>(current)));
        NN_RESULT_DO(context.Add(nn::erpt::FieldId::PreviousSystemPowerState, static_cast<uint32_t>(previous)));
        NN_RESULT_DO(context.Add(nn::erpt::FieldId::DestinationSystemPowerState, static_cast<uint32_t>(destination)));
        NN_RESULT_DO(context.SubmitContext());

        NN_RESULT_SUCCESS;
    }

    nn::Result ReportStateTransitionError(nn::psc::PmControl* pPmControl, nn::Result resultToReport) NN_NOEXCEPT
    {
        // スタック食いそうなので static に
        static nn::spsm::detail::PscTransition s_PscTransitionInfo = nn::spsm::detail::PscTransition();
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        NN_RESULT_DO(pPmControl->GetModuleInformation(
            &(s_PscTransitionInfo.transitionInfo),
            &(s_PscTransitionInfo.moduleCount),
            s_PscTransitionInfo.moduleInfoList,
            sizeof(s_PscTransitionInfo.moduleInfoList) / sizeof(s_PscTransitionInfo.moduleInfoList[0]),
            &(s_PscTransitionInfo.dependencyCount), // 取るけどレポートには載せない
            s_PscTransitionInfo.dependencyInfoList,
            sizeof(s_PscTransitionInfo.dependencyInfoList) / sizeof(s_PscTransitionInfo.dependencyInfoList[0])
        ));
#else
        NN_UNUSED(pPmControl);
#endif
        // これらもスタック食いそうなので static に
        static uint8_t  s_PscInitializedList[nn::spsm::detail::MaxModuleCount] = { 0 };
        static uint32_t s_PscCurrentPmStateList[nn::spsm::detail::MaxModuleCount] = { 0 };
        static uint32_t s_PscNextPmStateList[nn::spsm::detail::MaxModuleCount] = { 0 };
        NN_RESULT_DO(ConvertModuleTransitionInformationToErrorReportFields(
            s_PscInitializedList,
            s_PscCurrentPmStateList,
            s_PscNextPmStateList,
            nn::spsm::detail::MaxModuleCount,
            s_PscTransitionInfo.moduleInfoList,
            s_PscTransitionInfo.moduleCount
        ));

        static uint8_t contextLocalBuffer[512 * 3];
        nn::erpt::Context context(nn::erpt::CategoryId::ErrorInfo, contextLocalBuffer, sizeof(contextLocalBuffer));

        NN_RESULT_DO(context.Add(nn::erpt::FieldId::PscTransitionCurrentState, static_cast<uint32_t>(s_PscTransitionInfo.transitionInfo.currentState)));
        NN_RESULT_DO(context.Add(nn::erpt::FieldId::PscTransitionPreviousState, static_cast<uint32_t>(s_PscTransitionInfo.transitionInfo.previousState)));
        NN_RESULT_DO(context.Add(nn::erpt::FieldId::PscInitializedList, s_PscInitializedList, nn::spsm::detail::MaxModuleCount));
        NN_RESULT_DO(context.Add(nn::erpt::FieldId::PscCurrentPmStateList, s_PscCurrentPmStateList, nn::spsm::detail::MaxModuleCount));
        NN_RESULT_DO(context.Add(nn::erpt::FieldId::PscNextPmStateList, s_PscNextPmStateList, nn::spsm::detail::MaxModuleCount));

        // エラーレポートの作成にエラーコードの指定が必要で、エラーコードは文字列で指定する必要がある
        auto errorCode = nn::err::detail::ConvertResultToErrorCode(resultToReport);
        char errorCodeString[nn::err::ErrorCode::StringLengthMax];
        util::SNPrintf(errorCodeString, sizeof(errorCodeString), "%04d-%04d", errorCode.category, errorCode.number);
        NN_RESULT_DO(context.Add(nn::erpt::FieldId::ErrorCode, errorCodeString, static_cast<uint32_t>(sizeof(errorCodeString))));

        NN_RESULT_DO(context.CreateReport(nn::erpt::ReportType_Invisible));

        NN_RESULT_SUCCESS;
    }

}}}
