﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/settings/settings_Language.h>
#include <nn/settings/system/settings_Account.h>
#include <nn/settings/system/settings_Audio.h>
#include <nn/settings/system/settings_Backlight.h>
#include <nn/settings/system/settings_BluetoothDevices.h>
#include <nn/settings/system/settings_Clock.h>
#include <nn/settings/system/settings_DataDeletion.h>
#include <nn/settings/system/settings_Eula.h>
#include <nn/settings/system/settings_Fatal.h>
#include <nn/settings/system/settings_Hid.h>
#include <nn/settings/system/settings_Mii.h>
#include <nn/settings/system/settings_Network.h>
#include <nn/settings/system/settings_News.h>
#include <nn/settings/system/settings_OnlineStorage.h>
#include <nn/settings/system/settings_SixAxisSensor.h>
#include <nn/settings/system/settings_Sleep.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/settings/system/settings_Telemetry.h>
#include <nn/settings/system/settings_Tv.h>
#include <nn/settings/system/settings_Web.h>
#include <nn/time/time_LocationName.h>
#include <nn/util/util_Uuid.h>

namespace nn { namespace settings { namespace detail {

//!< システム設定の変更通知のハンドルを FATAL 表示向けに取得します。
::nn::Result AcquireSystemSettingsFatalDirtyFlagEvent(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

//!< システム設定のダーティーフラグを FATAL 表示向けに取得します。
::nn::Result GetSystemSettingsFatalDirtyFlags(
    ::nn::settings::system::FatalDirtyFlagSet* pOutValue) NN_NOEXCEPT;

//!< システム設定の変更通知のハンドルを障害情報収集向けに取得します。
::nn::Result AcquireSystemSettingsTelemetryDirtyFlagEvent(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

//!< システム設定のダーティーフラグを障害情報収集向けに取得します。
::nn::Result GetSystemSettingsTelemetryDirtyFlags(
    ::nn::settings::system::TelemetryDirtyFlagSet* pOutValue) NN_NOEXCEPT;

//!< システム設定から言語コードを取得します。
::nn::Result GetSystemSettingsLanguageCode(LanguageCode* pOutValue) NN_NOEXCEPT;

//!< システム設定の言語コードを変更します。
::nn::Result SetSystemSettingsLanguageCode(LanguageCode value) NN_NOEXCEPT;

//!< システム設定から利用可能な言語コードの数を取得します。
::nn::Result GetSystemSettingsAvailableLanguageCodeCount(
    int32_t* pOutCount) NN_NOEXCEPT;

//!< システム設定から指定された利用可能な言語コードを取得します。
::nn::Result GetSystemSettingsAvailableLanguageCode(
    LanguageCode* pOutValue, int32_t index) NN_NOEXCEPT;

//!< システム設定から利用可能な言語コードの一覧を取得します。
::nn::Result GetSystemSettingsAvailableLanguageCodes(
    int32_t* pOutCount, LanguageCode outCodes[], int32_t count) NN_NOEXCEPT;

//!< システム設定からネットワーク設定を取得します。
::nn::Result GetSystemSettingsNetworkSettings(
    int32_t* pOutCount,
    ::nn::settings::system::NetworkSettings outValues[],
    int32_t count) NN_NOEXCEPT;

//!< システム設定のネットワーク設定を変更します。
::nn::Result SetSystemSettingsNetworkSettings(
    const ::nn::settings::system::NetworkSettings values[],
    int32_t count) NN_NOEXCEPT;

//!< システム設定のリージョンコードを取得します。
::nn::Result GetSystemSettingsRegionCode(int32_t* pOutValue) NN_NOEXCEPT;

//!< システム設定のリージョンコードを変更します。
::nn::Result SetSystemSettingsRegionCode(int32_t value) NN_NOEXCEPT;

//!< システム設定から Bluetooth デバイス設定を取得します。
::nn::Result GetSystemSettingsBluetoothDevicesSettings(
    int32_t* pOutCount,
    ::nn::settings::system::BluetoothDevicesSettings outValues[],
    int32_t count) NN_NOEXCEPT;

//!< システム設定の Bluetooth デバイス設定を変更します。
::nn::Result SetSystemSettingsBluetoothDevicesSettings(
    const ::nn::settings::system::BluetoothDevicesSettings values[],
    int32_t count) NN_NOEXCEPT;

//!< システム設定から Bluetooth が有効か否かを表す値を取得します。
::nn::Result GetSystemSettingsBluetoothEnableFlag(
    bool* pOutValue) NN_NOEXCEPT;

//!< システム設定の Bluetooth が有効か否かを表す値を変更します。
::nn::Result SetSystemSettingsBluetoothEnableFlag(bool value) NN_NOEXCEPT;

//!< システム設定から Bluetooth の AFH が有効か否かを表す値を取得します。
::nn::Result GetSystemSettingsBluetoothAfhEnableFlag(
    bool* pOutValue) NN_NOEXCEPT;

//!< システム設定の Bluetooth の AFH  が有効か否かを表す値を変更します。
::nn::Result SetSystemSettingsBluetoothAfhEnableFlag(bool value) NN_NOEXCEPT;

//!< システム設定から Bluetooth の Boost が有効か否かを表す値を取得します。
::nn::Result GetSystemSettingsBluetoothBoostEnableFlag(
    bool* pOutValue) NN_NOEXCEPT;

//!< システム設定の Bluetooth の Boost が有効か否かを表す値を変更します。
::nn::Result SetSystemSettingsBluetoothBoostEnableFlag(bool value) NN_NOEXCEPT;

//!< システム設定から BLE ペアリング設定を取得します。
::nn::Result GetSystemSettingsBlePairingSettings(
    int32_t* pOutCount,
    ::nn::settings::system::BlePairingSettings outValues[],
    int32_t count) NN_NOEXCEPT;

//!< システム設定の BLE ペアリング設定を変更します。
::nn::Result SetSystemSettingsBlePairingSettings(
    const ::nn::settings::system::BlePairingSettings values[],
    int32_t count) NN_NOEXCEPT;

//!< システム設定から携帯モード時誤操作防止画面の有無を取得します。
::nn::Result GetSystemSettingsLockScreenFlag(bool* pOutValue) NN_NOEXCEPT;

//!< システム設定の携帯モード時誤操作防止画面の有無を変更します。
::nn::Result SetSystemSettingsLockScreenFlag(bool value) NN_NOEXCEPT;

//!< システム設定から本体 EULA の同意バージョンを取得します。
::nn::Result GetSystemSettingsEulaVersions(
    int32_t* pOutCount,
    ::nn::settings::system::EulaVersion outValues[],
    size_t count) NN_NOEXCEPT;

//!< システム設定の本体 EULA の同意バージョンを変更します。
::nn::Result SetSystemSettingsEulaVersions(
    const ::nn::settings::system::EulaVersion values[],
    size_t count) NN_NOEXCEPT;

//!< システム設定からエラーレポートを他社に提供してよいかの許可状態を取得します。
::nn::Result GetSystemSettingsErrorReportSharePermission(
    int32_t* pOutValue) NN_NOEXCEPT;

//!< システム設定のエラーレポートを他社に提供してよいかの許可状態を変更します。
::nn::Result SetSystemSettingsErrorReportSharePermission(
    int32_t value) NN_NOEXCEPT;

//!< システム設定からカラーセット ID を取得します。
::nn::Result GetSystemSettingsColorSetId(int32_t* pOutValue) NN_NOEXCEPT;

//!< システム設定のカラーセット ID を変更します。
::nn::Result SetSystemSettingsColorSetId(int32_t value) NN_NOEXCEPT;

//!< システム設定からバッテリー残量のパーセンテージ表示を行うか否か表す値を取得します。
::nn::Result GetSystemSettingsBatteryPercentageFlag(
    bool* pOutValue) NN_NOEXCEPT;

//!< システム設定のバッテリー残量のパーセンテージ表示を行うか否か表す値を変更します。
::nn::Result SetSystemSettingsBatteryPercentageFlag(bool value) NN_NOEXCEPT;

//!< システム設定から Quest メニューを起動するか否かを表す値を取得します。
::nn::Result GetSystemSettingsQuestFlag(bool* pOutValue) NN_NOEXCEPT;

//!< システム設定の Quest メニューを起動するか否かを表す値を変更します。
::nn::Result SetSystemSettingsQuestFlag(bool value) NN_NOEXCEPT;

//!< システム設定から画面の明るさの設定を取得します。
::nn::Result GetSystemSettingsBacklightSettings(
    ::nn::settings::system::BacklightSettings* pOutValue) NN_NOEXCEPT;

//!< システム設定の画面の明るさの設定を変更します。
::nn::Result SetSystemSettingsBacklightSettings(
    const ::nn::settings::system::BacklightSettings& value) NN_NOEXCEPT;

//!< システム設定から画面の明るさ (VR モード対応版) の設定を取得します。
::nn::Result GetSystemSettingsBacklightSettings(
    ::nn::settings::system::BacklightSettingsEx* pOutValue) NN_NOEXCEPT;

//!< システム設定の画面の明るさ (VR モード対応版) の設定を変更します。
::nn::Result SetSystemSettingsBacklightSettings(
    const ::nn::settings::system::BacklightSettingsEx& value) NN_NOEXCEPT;

//!< システム設定から ExternalSteadyClock のソース ID を取得します。
::nn::Result GetSystemSettingsExternalSteadyClockSourceId(
    ::nn::settings::system::ClockSourceId* pOutValue) NN_NOEXCEPT;

//!< システム設定の ExternalSteadyClock のソース ID を変更します。
::nn::Result SetSystemSettingsExternalSteadyClockSourceId(
    const ::nn::settings::system::ClockSourceId& value) NN_NOEXCEPT;

//!< システム設定からシャットダウン時の RTC 値を取得します。
::nn::Result GetSystemSettingsShutdownRtcValue(int64_t* pOutValue) NN_NOEXCEPT;

//!< システム設定のシャットダウン時の RTC 値を変更します。
::nn::Result SetSystemSettingsShutdownRtcValue(int64_t value) NN_NOEXCEPT;

//!< システム設定から ExternalSteadyClock の内部オフセットを取得します。
::nn::Result GetSystemSettingsExternalSteadyClockInternalOffset(
    int64_t* pOutValue) NN_NOEXCEPT;

//!< システム設定の ExternalSteadyClock の内部オフセットを変更します。
::nn::Result SetSystemSettingsExternalSteadyClockInternalOffset(
    int64_t value) NN_NOEXCEPT;

//!< システム設定から UserSystemClock のコンテキストを取得します。
::nn::Result GetSystemSettingsUserSystemClockContext(
    ::nn::settings::system::SystemClockContext* pOutValue) NN_NOEXCEPT;

//!< システム設定の UserSystemClock のコンテキストを変更します。
::nn::Result SetSystemSettingsUserSystemClockContext(
    const ::nn::settings::system::SystemClockContext& value) NN_NOEXCEPT;

//!< システム設定から NetworkSystemClock のコンテキストを取得します。
::nn::Result GetSystemSettingsNetworkSystemClockContext(
    ::nn::settings::system::SystemClockContext* pOutValue) NN_NOEXCEPT;

//!< システム設定の NetworkSystemClock のコンテキストを変更します。
::nn::Result SetSystemSettingsNetworkSystemClockContext(
    const ::nn::settings::system::SystemClockContext& value) NN_NOEXCEPT;

//!< システム設定の UserSystemClock の自動補正フラグを取得します。
::nn::Result GetSystemSettingsUserSystemClockAutomaticCorrectionFlag(
    bool* pOutValue) NN_NOEXCEPT;

//!< システム設定の UserSystemClock の自動補正フラグを変更します。
::nn::Result SetSystemSettingsUserSystemClockAutomaticCorrectionFlag(
    bool value) NN_NOEXCEPT;

//!< システム設定の UserSystemClock の自動補正フラグの更新時間を取得します。
::nn::Result GetSystemSettingsUserSystemClockAutomaticCorrectionUpdatedTime(
    ::nn::time::SteadyClockTimePoint* pOutValue) NN_NOEXCEPT;

//!< システム設定の UserSystemClock の自動補正フラグの更新時間を変更します。
::nn::Result SetSystemSettingsUserSystemClockAutomaticCorrectionUpdatedTime(
    const ::nn::time::SteadyClockTimePoint& value) NN_NOEXCEPT;

//!< システム設定からアカウント設定を取得します。
::nn::Result GetSystemSettingsAccountSettings(
    ::nn::settings::system::AccountSettings* pOutValue) NN_NOEXCEPT;

//!< システム設定のアカウント設定を変更します。
::nn::Result SetSystemSettingsAccountSettings(
    const ::nn::settings::system::AccountSettings& value) NN_NOEXCEPT;

//!< システム設定から本体音量設定を取得します。
::nn::Result GetSystemSettingsAudioVolume(
    ::nn::settings::system::AudioVolume* pOutValue,
    int32_t index) NN_NOEXCEPT;

//!< システム設定の本体音量設定を変更します。
::nn::Result SetSystemSettingsAudioVolume(
    const ::nn::settings::system::AudioVolume& value,
    int32_t index) NN_NOEXCEPT;

//!< システム設定から本体オーディオ出力設定を取得します。
::nn::Result GetSystemSettingsAudioOutputMode(
    int32_t* pOutValue, int32_t index) NN_NOEXCEPT;

//!< システム設定の本体オーディオ出力設定を変更します。
::nn::Result SetSystemSettingsAudioOutputMode(
    int32_t value, int32_t index) NN_NOEXCEPT;

//!< システム設定からヘッドフォン抜け時の強制ミュート設定を取得します。
::nn::Result GetSystemSettingsForceMuteOnHeadphoneRemovedFlag(
    bool* pOutValue) NN_NOEXCEPT;

//!< システム設定のヘッドフォン抜け時の強制ミュート設定を変更します。
::nn::Result SetSystemSettingsForceMuteOnHeadphoneRemovedFlag(
    bool value) NN_NOEXCEPT;

//!< システム設定からヘッドフォン音量の警告表示回数を取得します。
::nn::Result GetSystemSettingsHeadphoneVolumeWarningCount(
    int32_t* pOutValue) NN_NOEXCEPT;

//!< システム設定のヘッドフォン音量の警告表示回数を変更します。
::nn::Result SetSystemSettingsHeadphoneVolumeWarningCount(
    int32_t value) NN_NOEXCEPT;

//!< システム設定からヘッドフォン音量を更新済みか否かを表す値を取得します。
::nn::Result GetSystemSettingsHeadphoneVolumeUpdateFlag(
    bool* pOutValue) NN_NOEXCEPT;

//!< システム設定のヘッドフォン音量を更新済みか否かを表す値を変更します。
::nn::Result SetSystemSettingsHeadphoneVolumeUpdateFlag(bool value) NN_NOEXCEPT;

//!< システム設定からヘッドフォン音量を更新する必要があるか否かを表す値を取得します。
::nn::Result NeedsToUpdateSystemSettingsHeadphoneVolume(
    bool* pOutValue,
    bool* pOutIsHighPowerAvailable,
    int8_t* pOutVolume,
    bool isRestricted) NN_NOEXCEPT;

//!< システム設定から本体情報アップロードが有効か否かを表す値を取得します。
::nn::Result GetSystemSettingsConsoleInformationUploadFlag(
    bool* pOutValue) NN_NOEXCEPT;

//!< システム設定の本体情報アップロードが有効か否かを表す値を設定します。
::nn::Result SetSystemSettingsConsoleInformationUploadFlag(
    bool value) NN_NOEXCEPT;

//!< システム設定からアプリケーション自動配信が有効か否かを表す値を取得します。
::nn::Result GetSystemSettingsAutomaticApplicationDownloadFlag(
    bool* pOutValue) NN_NOEXCEPT;

//!< システム設定のアプリケーション自動配信が有効か否かを表す値を設定します。
::nn::Result SetSystemSettingsAutomaticApplicationDownloadFlag(
    bool value) NN_NOEXCEPT;

//!< システム設定から通知設定を取得します。
::nn::Result GetSystemSettingsNotificationSettings(
    ::nn::settings::system::NotificationSettings* pOutValue) NN_NOEXCEPT;

//!< システム設定の通知設定を変更します。
::nn::Result SetSystemSettingsNotificationSettings(
    const ::nn::settings::system::NotificationSettings& value) NN_NOEXCEPT;

//!< システム設定からアカウント毎の通知設定を取得します。
::nn::Result GetSystemSettingsAccountNotificationSettings(
    int32_t* pOutCount,
    ::nn::settings::system::AccountNotificationSettings outValues[],
    int32_t count) NN_NOEXCEPT;

//!< システム設定のアカウント毎の通知設定を変更します。
::nn::Result SetSystemSettingsAccountNotificationSettings(
    const ::nn::settings::system::AccountNotificationSettings values[],
    int32_t count) NN_NOEXCEPT;


//!< システム設定からアカウント毎のオンラインストレージ設定を取得します。
::nn::Result GetSystemSettingsAccountOnlineStorageSettings(
    int32_t* pOutCount,
    ::nn::settings::system::AccountOnlineStorageSettings outValues[],
    int32_t count) NN_NOEXCEPT;

//!< システム設定のアカウント毎のオンラインストレージ設定を変更します。
::nn::Result SetSystemSettingsAccountOnlineStorageSettings(
    const ::nn::settings::system::AccountOnlineStorageSettings values[],
    int32_t count) NN_NOEXCEPT;

//!< システム設定から Pro Controller 有線 USB 通信機能が有効か否かを表す値を取得します。
::nn::Result GetSystemSettingsUsbFullKeyEnableFlag(bool* pOutValue) NN_NOEXCEPT;

//!< システム設定の Pro Controller 有線 USB 通信機能が有効か否かを表す値を変更します。
::nn::Result SetSystemSettingsUsbFullKeyEnableFlag(bool value) NN_NOEXCEPT;

//!< システム設定から振動のマスターボリュームを取得します。
::nn::Result GetSystemSettingsVibrationMasterVolume(
    float* pOutValue) NN_NOEXCEPT;

//!< システム設定の振動のマスターボリュームを変更します。
::nn::Result SetSystemSettingsVibrationMasterVolume(float value) NN_NOEXCEPT;

//!< システム設定から本体 6 軸センサーの加速度バイアス値を取得します。
::nn::Result GetSystemSettingsConsoleSixAxisSensorAccelerationBias(
    ::nn::settings::system::ConsoleSixAxisSensorAccelerationBias* pOutValue
    ) NN_NOEXCEPT;

//!< システム設定の本体 6 軸センサーの加速度バイアス値を変更します。
::nn::Result SetSystemSettingsConsoleSixAxisSensorAccelerationBias(
    const ::nn::settings::system::ConsoleSixAxisSensorAccelerationBias& value
    ) NN_NOEXCEPT;

//!< システム設定から本体 6 軸センサーの角速度バイアス値を取得します。
::nn::Result GetSystemSettingsConsoleSixAxisSensorAngularVelocityBias(
    ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityBias* pOutValue
    ) NN_NOEXCEPT;

//!< システム設定の本体 6 軸センサーの角速度バイアス値を変更します。
::nn::Result SetSystemSettingsConsoleSixAxisSensorAngularVelocityBias(
    const ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityBias& value
    ) NN_NOEXCEPT;

//!< システム設定から本体 6 軸センサーの加速度ゲイン値を取得します。
::nn::Result GetSystemSettingsConsoleSixAxisSensorAccelerationGain(
    ::nn::settings::system::ConsoleSixAxisSensorAccelerationGain* pOutValue
    ) NN_NOEXCEPT;

//!< システム設定の本体 6 軸センサーの加速度ゲイン値を変更します。
::nn::Result SetSystemSettingsConsoleSixAxisSensorAccelerationGain(
    const ::nn::settings::system::ConsoleSixAxisSensorAccelerationGain& value
    ) NN_NOEXCEPT;

//!< システム設定から本体 6 軸センサーの角速度ゲイン値を取得します。
::nn::Result GetSystemSettingsConsoleSixAxisSensorAngularVelocityGain(
    ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityGain* pOutValue
    ) NN_NOEXCEPT;

//!< システム設定の本体 6 軸センサーの角速度ゲイン値を変更します。
::nn::Result SetSystemSettingsConsoleSixAxisSensorAngularVelocityGain(
    const ::nn::settings::system::ConsoleSixAxisSensorAngularVelocityGain& value
    ) NN_NOEXCEPT;

//!< システム設定から本体 6 軸センサーの角速度の時間バイアス値を取得します。
::nn::Result GetSystemSettingsConsoleSixAxisSensorAngularVelocityTimeBias(
    ::nn::settings::system::
        ConsoleSixAxisSensorAngularVelocityTimeBias* pOutValue) NN_NOEXCEPT;

//!< システム設定の本体 6 軸センサーの角速度の時間バイアス値を変更します。
::nn::Result SetSystemSettingsConsoleSixAxisSensorAngularVelocityTimeBias(
    const ::nn::settings::system::
        ConsoleSixAxisSensorAngularVelocityTimeBias& value) NN_NOEXCEPT;

//!< システム設定から本体 6 軸センサーの角加速度の係数を取得します。
::nn::Result GetSystemSettingsConsoleSixAxisSensorAngularAcceleration(
    ::nn::settings::system::ConsoleSixAxisSensorAngularAcceleration* pOutValue
    ) NN_NOEXCEPT;

//!< システム設定の本体 6 軸センサーの角加速度の係数を変更します。
::nn::Result SetSystemSettingsConsoleSixAxisSensorAngularAcceleration(
    const ::nn::settings::system::ConsoleSixAxisSensorAngularAcceleration& value
    ) NN_NOEXCEPT;

//!< システム設定からコントローラ情報を取得します。
::nn::Result GetSystemSettingsNxControllerSettings(
    int32_t* pOutCount,
    ::nn::settings::system::NxControllerSettings outValues[],
    int32_t count) NN_NOEXCEPT;

//!< システム設定のコントローラ情報を変更します。
::nn::Result SetSystemSettingsNxControllerSettings(
    const ::nn::settings::system::NxControllerSettings values[],
    int32_t count) NN_NOEXCEPT;

//!< システム設定から TV 設定を取得します。
::nn::Result GetSystemSettingsTvSettings(
    ::nn::settings::system::TvSettings* pOutValue) NN_NOEXCEPT;

//!< システム設定の TV 設定を変更します。
::nn::Result SetSystemSettingsTvSettings(
    const ::nn::settings::system::TvSettings& value) NN_NOEXCEPT;

//!< システム設定から EDID を取得します。
::nn::Result GetSystemSettingsEdid(
    ::nn::settings::system::Edid* pOutValue) NN_NOEXCEPT;

//!< システム設定の EDID を変更します。
::nn::Result SetSystemSettingsEdid(
    const ::nn::settings::system::Edid& value) NN_NOEXCEPT;

//!< システム設定からおすすめ削除設定を取得します。
::nn::Result GetSystemSettingsDataDeletionSettings(
    ::nn::settings::system::DataDeletionSettings* pOutValue) NN_NOEXCEPT;

//!< システム設定のおすすめ削除設定を変更します。
::nn::Result SetSystemSettingsDataDeletionSettings(
    const ::nn::settings::system::DataDeletionSettings& value) NN_NOEXCEPT;

//!< システム設定から初回ブートするシステムアプレットの ProgramId を取得します。
::nn::Result GetSystemSettingsInitialSystemAppletProgramId(
    ::nn::ncm::ProgramId* pOutValue) NN_NOEXCEPT;

//!< システム設定の初回ブートするシステムアプレットの ProgramId を設定します。
::nn::Result SetSystemSettingsInitialSystemAppletProgramId(
    const ::nn::ncm::ProgramId& value) NN_NOEXCEPT;

//!< システム設定から初回ブートするオーバーレイ表示の ProgramId を取得します。
::nn::Result GetSystemSettingsOverlayDispProgramId(
    ::nn::ncm::ProgramId* pOutValue) NN_NOEXCEPT;

//!< システム設定の初回ブートするオーバーレイ表示の ProgramId を設定します。
::nn::Result SetSystemSettingsOverlayDispProgramId(
    const ::nn::ncm::ProgramId& value) NN_NOEXCEPT;

//!< システム設定からデバイスの地域名（タイムゾーン名）を取得します。
::nn::Result GetSystemSettingsDeviceTimeZoneLocationName(
    ::nn::time::LocationName* pOutValue) NN_NOEXCEPT;

//!< システム設定のデバイスの地域名（タイムゾーン名）を変更します。
::nn::Result SetSystemSettingsDeviceTimeZoneLocationName(
    const ::nn::time::LocationName& value) NN_NOEXCEPT;

//!< システム設定のデバイスの地域更新時間を取得します。
::nn::Result GetSystemSettingsDeviceTimeZoneLocationUpdatedTime(
    ::nn::time::SteadyClockTimePoint* pOutValue) NN_NOEXCEPT;

//!< システム設定のデバイスの地域更新時間を変更します。
::nn::Result SetSystemSettingsDeviceTimeZoneLocationUpdatedTime(
    const ::nn::time::SteadyClockTimePoint& value) NN_NOEXCEPT;

//!< システム設定から撮影した画面写真を保存するストレージを取得します。
::nn::Result GetSystemSettingsPrimaryAlbumStorage(
    int32_t* pOutValue) NN_NOEXCEPT;

//!< システム設定の撮影した画面写真を保存するストレージを変更します。
::nn::Result SetSystemSettingsPrimaryAlbumStorage(int32_t value) NN_NOEXCEPT;

//!< システム設定から USB 3.0 が有効か否かを表す値を取得します。
::nn::Result GetSystemSettingsUsb30EnableFlag(bool* pOutValue) NN_NOEXCEPT;

//!< システム設定の USB 3.0 が有効か否かを表す値を変更します。
::nn::Result SetSystemSettingsUsb30EnableFlag(bool value) NN_NOEXCEPT;

//!< システム設定から NFC が有効か否かを表す値を取得します。
::nn::Result GetSystemSettingsNfcEnableFlag(bool* pOutValue) NN_NOEXCEPT;

//!< システム設定の NFC が有効か否かを表す値を変更します。
::nn::Result SetSystemSettingsNfcEnableFlag(bool value) NN_NOEXCEPT;

//!< システム設定からスリープ設定を取得します。
::nn::Result GetSystemSettingsSleepSettings(
    ::nn::settings::system::SleepSettings* pOutValue) NN_NOEXCEPT;

//!< システム設定のスリープ設定を変更します。
::nn::Result SetSystemSettingsSleepSettings(
    const ::nn::settings::system::SleepSettings& value) NN_NOEXCEPT;

//!< システム設定から無線 LAN が有効か否かを表す値を取得します。
::nn::Result GetSystemSettingsWirelessLanEnableFlag(
    bool* pOutValue) NN_NOEXCEPT;

//!< システム設定の無線 LAN が有効か否かを表す値を変更します。
::nn::Result SetSystemSettingsWirelessLanEnableFlag(bool value) NN_NOEXCEPT;

//!< システム設定から初回起動設定を取得します。
::nn::Result GetSystemSettingsInitialLaunchSettings(
    ::nn::settings::system::InitialLaunchSettings* pOutValue) NN_NOEXCEPT;

//!< システム設定の初回起動設定を変更します。
::nn::Result SetSystemSettingsInitialLaunchSettings(
    const ::nn::settings::system::InitialLaunchSettings& value) NN_NOEXCEPT;

//!< システム設定から本体の識別名を取得します。
::nn::Result GetSystemSettingsDeviceNickName(
    ::nn::settings::system::DeviceNickName* pOutValue) NN_NOEXCEPT;

//!< システム設定の本体の識別名を変更します。
::nn::Result SetSystemSettingsDeviceNickName(
    const ::nn::settings::system::DeviceNickName& value) NN_NOEXCEPT;

//!< システム設定からアプレット起動の制御フラグを取得します。
::nn::Result GetSystemSettingsAppletLaunchFlags(
    ::nn::settings::system::AppletLaunchFlagSet* pOutValue) NN_NOEXCEPT;

//!< システム設定のアプレット起動の制御フラグを変更します。
::nn::Result SetSystemSettingsAppletLaunchFlags(
    const ::nn::settings::system::AppletLaunchFlagSet& value) NN_NOEXCEPT;

//!< システム設定からローカル通信のチャンネルを取得します。
::nn::Result GetSystemSettingsLdnChannel(int32_t* pOutValue) NN_NOEXCEPT;

//!< システム設定のローカル通信のチャンネルを変更します。
::nn::Result SetSystemSettingsLdnChannel(int32_t value) NN_NOEXCEPT;

//!< システム設定の Mii 作者 ID を取得します。
::nn::Result GetSystemSettingsMiiAuthorId(
    ::nn::util::Uuid* pOutValue) NN_NOEXCEPT;

//!< システム設定からパッチの自動受信が有効か否かを表す値を取得します。
::nn::Result GetSystemSettingsAutoUpdateEnableFlag(bool* pOutValue) NN_NOEXCEPT;

//!< システム設定のパッチの自動受信が有効か否かを表す値を変更します。
::nn::Result SetSystemSettingsAutoUpdateEnableFlag(bool value) NN_NOEXCEPT;

//!< システム設定から外部 RTC のリセットが発生したか否かを表す値を取得します。
::nn::Result GetSystemSettingsExternalRtcResetFlag(bool* pOutValue) NN_NOEXCEPT;

//!< システム設定の外部 RTC のリセットが発生したか否かを表す値を変更します。
::nn::Result SetSystemSettingsExternalRtcResetFlag(bool value) NN_NOEXCEPT;

//!< システム設定からプッシュ通知のスリープ状態における動作モードを取得します。
::nn::Result GetSystemSettingsPushNotificationActivityModeOnSleep(
    int32_t* pOutValue) NN_NOEXCEPT;

//!< システム設定のプッシュ通知のスリープ状態における動作モードを変更します。
::nn::Result SetSystemSettingsPushNotificationActivityModeOnSleep(
    int32_t value) NN_NOEXCEPT;

//!< システム設定からキーボード配列の種別を取得します。
::nn::Result GetSystemSettingsKeyboardLayout(int32_t* pOutValue) NN_NOEXCEPT;

//!< システム設定のキーボード配列の種別を変更します。
::nn::Result SetSystemSettingsKeyboardLayout(int32_t value) NN_NOEXCEPT;

//!< システム設定から Web インスペクタを使用するか否かを表す値を取得します。
::nn::Result GetSystemSettingsWebInspectorFlag(bool* pOutValue) NN_NOEXCEPT;

//!< システム設定の Web インスペクタを使用するか否かを表す値を変更します。
::nn::Result SetSystemSettingsWebInspectorFlag(bool value) NN_NOEXCEPT;

//!< システム設定から SSL チェック無効化ホストを取得します。
::nn::Result GetSystemSettingsAllowedSslHosts(
    int32_t* pOutCount,
    ::nn::settings::system::AllowedSslHost outValues[],
    size_t count) NN_NOEXCEPT;

//!< システム設定の SSL チェック無効化ホストを変更します。
::nn::Result SetSystemSettingsAllowedSslHosts(
    const ::nn::settings::system::AllowedSslHost values[],
    size_t count) NN_NOEXCEPT;

//!< システム設定からホスト FS のマウントポイントを取得します。
::nn::Result GetSystemSettingsHostFsMountPoint(
    ::nn::settings::system::HostFsMountPoint* pOutValue) NN_NOEXCEPT;

//!< システム設定のホスト FS のマウントポイントを変更します。
::nn::Result SetSystemSettingsHostFsMountPoint(
    const ::nn::settings::system::HostFsMountPoint& value) NN_NOEXCEPT;

//!< システム設定から 時計補正ツールの実行が必要か否かを取得します。
::nn::Result GetSystemSettingsRequiresRunRepairTimeReviser(
    bool* pOutValue) NN_NOEXCEPT;

//!< システム設定の 時計補正ツールの実行が必要か否かを変更します。
::nn::Result SetSystemSettingsRequiresRunRepairTimeReviser(
    bool value) NN_NOEXCEPT;

//!< システム設定を読み出します。
::nn::Result ReadSystemSettings(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT;

//!< システム設定をリセットします。
::nn::Result ResetSystemSettings() NN_NOEXCEPT;

//!< システム設定のインメモリモードを有効にします。（デバッグ用）
::nn::Result EnableSystemSettingsInMemoryModeForDebug(
    void* buffer, size_t size) NN_NOEXCEPT;

}}} // namespace nn::settings::detail
