﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>

#include "settings_StaticObject.h"
#include "settings_SystemData-os.horizon.h"

namespace nn { namespace settings { namespace detail {

namespace {

//!< マウント名のセパレータ
const char MountNameSeparator[] = ":";

//!< ディレクトリ名のセパレータ
const char DirectoryNameSeparator[] = "/";

//!< システムデータのファイル名
const char SystemDataFileName[] = "file";

//!< 遅延ファイルアクセスを扱うクラスです。
class LazyFileAccessor final
{
    NN_DISALLOW_COPY(LazyFileAccessor);
    NN_DISALLOW_MOVE(LazyFileAccessor);

private:
    //!< バッファを占有中か否か
    bool m_IsOccupied;

    //!< ファイルハンドル
    ::nn::fs::FileHandle m_FileHandle;

    //!< 遅延アクセスの開始位置
    int64_t m_Offset;

    //!< 遅延アクセスのバイトサイズ
    size_t m_Size;

    //!< 遅延アクセス用のバッファ
    ::nn::Bit8 m_Buffer[256 << 10];

public:
    LazyFileAccessor() NN_NOEXCEPT;

    //!< ファイルをオープンします。
    ::nn::Result Open(const char path[], int mode) NN_NOEXCEPT;

    //!< ファイルをクローズします。
    void Close() NN_NOEXCEPT;

    //!< ファイルから読み込みます。
    ::nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT;

private:
    //!< 指定箇所のキャッシュを取得します。
    bool GetCache(int64_t offset, void* buffer, size_t size) const NN_NOEXCEPT;
};

//!< 遅延ファイルアクセッサを返します。
LazyFileAccessor& GetLazyFileAccessor() NN_NOEXCEPT;

} // namespace

SystemData::SystemData() NN_NOEXCEPT
    : m_SystemDataId()
{
    // 何もしない
}

void SystemData::SetSystemDataId(::nn::ncm::SystemDataId value) NN_NOEXCEPT
{
    m_SystemDataId = value;
}

void SystemData::SetMountName(const char* name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(name);

    ::nn::util::Strlcpy(
        m_MountName,
        name,
        static_cast<int>(::std::extent<decltype(m_MountName)>::value));

    int position = 0;

    position += ::nn::util::Strlcpy(
        &m_FilePath[position],
        name,
        static_cast<int>(::std::extent<decltype(m_MountName)>::value));

    position += ::nn::util::Strlcpy(
        &m_FilePath[position],
        MountNameSeparator,
        static_cast<int>(::std::extent<decltype(MountNameSeparator)>::value));

    position += ::nn::util::Strlcpy(
        &m_FilePath[position],
        DirectoryNameSeparator,
        static_cast<int>(
            ::std::extent<decltype(DirectoryNameSeparator)>::value));

    ::nn::util::Strlcpy(
        &m_FilePath[position],
        SystemDataFileName,
        static_cast<int>(::std::extent<decltype(SystemDataFileName)>::value));
}

::nn::Result SystemData::Mount() NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::fs::MountSystemData(m_MountName, m_SystemDataId));
    NN_RESULT_SUCCESS;
}

::nn::Result SystemData::OpenToRead() NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetLazyFileAccessor().Open(m_FilePath, ::nn::fs::OpenMode_Read));
    NN_RESULT_SUCCESS;
}

void SystemData::Close() NN_NOEXCEPT
{
    GetLazyFileAccessor().Close();
}

::nn::Result SystemData::Read(
    int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_RESULT_DO(GetLazyFileAccessor().Read(offset, buffer, size));
    NN_RESULT_SUCCESS;
}

namespace {

LazyFileAccessor::LazyFileAccessor() NN_NOEXCEPT
    : m_IsOccupied(false)
    , m_FileHandle()
    , m_Offset()
    , m_Size()
{
    ::std::memset(m_Buffer, 0, sizeof(m_Buffer));
}

::nn::Result LazyFileAccessor::Open(const char path[], int mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    NN_RESULT_DO(::nn::fs::OpenFile(&m_FileHandle, path, mode));

    NN_RESULT_SUCCESS;
}

void LazyFileAccessor::Close() NN_NOEXCEPT
{
    ::nn::fs::CloseFile(m_FileHandle);

    m_IsOccupied = false;

    m_FileHandle = ::nn::fs::FileHandle();

    m_Offset = 0LL;

    m_Size = 0u;
}

::nn::Result LazyFileAccessor::Read(
    int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    if (this->GetCache(offset, buffer, size))
    {
        NN_RESULT_SUCCESS;
    }

    if (size > sizeof(m_Buffer))
    {
        NN_RESULT_DO(::nn::fs::ReadFile(m_FileHandle, offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    auto fileSize = int64_t();

    NN_RESULT_DO(::nn::fs::GetFileSize(&fileSize, m_FileHandle));

    if (fileSize < offset + static_cast<int64_t>(size))
    {
        NN_RESULT_DO(::nn::fs::ReadFile(m_FileHandle, offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    const auto cacheSize = ::std::min(
        static_cast<size_t>(fileSize - offset), sizeof(m_Buffer));

    if (cacheSize > 0)
    {
        NN_RESULT_DO(
            ::nn::fs::ReadFile(m_FileHandle, offset, m_Buffer, cacheSize));
    }

    m_Offset = offset;

    m_Size = cacheSize;

    m_IsOccupied = true;

    const bool succeeds = this->GetCache(offset, buffer, size);

    NN_SDK_ASSERT(succeeds);

    NN_UNUSED(succeeds);

    NN_RESULT_SUCCESS;
}

bool LazyFileAccessor::GetCache(
    int64_t offset, void* buffer, size_t size) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    if (!m_IsOccupied)
    {
        return false;
    }

    if (offset < m_Offset)
    {
        return false;
    }

    const auto bias = static_cast<size_t>(offset - m_Offset);

    if (m_Size < bias + size)
    {
        return false;
    }

    ::std::memcpy(buffer, &m_Buffer[bias], size);

    return true;
}

LazyFileAccessor& GetLazyFileAccessor() NN_NOEXCEPT
{
    return StaticObject<LazyFileAccessor>::Get();
}

} // namespace

}}} // namespace nn::settings::detail
