﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <atomic>
#include <cstring>
#include <functional>
#include <map>
#include <mutex>
#include <utility>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_Result.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/settings/settings_ServiceTypes.h>
#include <nn/settings/detail/settings_Log.h>
#include <nn/settings/fwdbg/settings_SettingsCommon.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "settings_KeyValueStore.h"
#include "settings_Platform.h"
#include "settings_Spl.h"

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include "settings_SystemData-os.horizon.h"
#include "settings_SystemSaveData-os.horizon.h"
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
#include "settings_SystemData-os.win.h"
#include "settings_SystemSaveData-os.win.h"
#endif

//!< ファームウェアデバッグ設定の警告ログを出力します。
#define NN_SETTINGS_FWDBG_WARN(...) \
    NN_DETAIL_SETTINGS_WARN("[firmware debug settings] Warning: " __VA_ARGS__)

namespace nn { namespace settings { namespace detail {

namespace {

//!< ファームウェアデバッグ設定のシステムデータの識別子
const ::nn::ncm::SystemDataId FwdbgSystemDataId = { 0x0100000000000818 };

//!< ファームウェアデバッグ設定のシステムデータのマウント名
const char FwdbgSystemDataMountName[] = "FwdbgSettingsD";

//!< プラットフォーム構成情報のシステムデータの識別子
const ::nn::ncm::SystemDataId PfCfgIcosaSystemDataId = {
    0x010000000000081F };
const ::nn::ncm::SystemDataId PfCfgIcosaMarikoSystemDataId = {
    0x0100000000000824 };
const ::nn::ncm::SystemDataId PfCfgCopperSystemDataId = {
    0x0100000000000820 };
const ::nn::ncm::SystemDataId PfCfgHoagSystemDataId = {
    0x0100000000000821 };

//!< プラットフォーム構成情報のシステムデータのマウント名
const char PfCfgSystemDataMountName[] = "PfcfgSettingsD";

//!< システムセーブデータの識別子
const ::nn::fs::SystemSaveDataId SystemSaveDataId = 0x8000000000000051;

//!< システムセーブデータの合計サイズ
const int64_t SystemSaveDataTotalSize = (512 + 32) << 10;

//!< システムセーブデータのジャーナルサイズ
const int64_t SystemSaveDataJournalSize = (512 + 32) << 10;

//!< システムセーブデータのフラグ集合
const uint32_t SystemSaveDataFlags =
    ::nn::fs::SaveDataFlags_KeepAfterResettingSystemSaveData |
    ::nn::fs::SaveDataFlags_KeepAfterRefurbishment;

//!< システムセーブデータのマウント名
const char SystemSaveDataMountName[] = "FwdbgSettingsS";

//!< ヒープに割り当てるメモリブロックのバイトサイズ
const size_t HeapMemorySize = 512 << 10;

//!< ヒープからメモリ領域を確保します。
void* AllocateFromHeap(size_t size) NN_NOEXCEPT;

//!< ヒープへメモリ領域を開放します。
void FreeToHeap(void* pointer, size_t size) NN_NOEXCEPT;

//!< ヒープから確保可能なメモリ領域の最大サイズを返します。
size_t GetHeapAllocatableSize() NN_NOEXCEPT;

//!< メモリアロケーションを扱うクラスです。
template<class T>
class Allocator
{
public:
    typedef size_t size_type;

    typedef ptrdiff_t difference_type;

    typedef T* pointer;

    typedef const T* const_pointer;

    typedef T& reference;

    typedef const T& const_reference;

    typedef T value_type;

    template <class U> struct rebind { typedef Allocator<U> other; };

public:
    Allocator() NN_NOEXCEPT { /* 何もしない */ }

    Allocator(const Allocator& other) NN_NOEXCEPT
    {
        NN_UNUSED(other);
    }

    Allocator(Allocator&& other) NN_NOEXCEPT
    {
        NN_UNUSED(other);
    }

    template<class U>
    Allocator(const Allocator<U>& other) NN_NOEXCEPT
    {
        NN_UNUSED(other);
    }

    template<class U>
    Allocator(Allocator<U>&& other) NN_NOEXCEPT
    {
        NN_UNUSED(other);
    }

    virtual ~Allocator() NN_NOEXCEPT { /* 何もしない */ }

    T* allocate(size_t n) NN_NOEXCEPT
    {
        return reinterpret_cast<T*>(AllocateFromHeap(n * sizeof(T)));
    }

    void deallocate(T* pointer, size_t n) NN_NOEXCEPT
    {
        FreeToHeap(pointer, n * sizeof(T));
    }

    void construct(T* pointer, const T& value) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pointer);
        new(reinterpret_cast<void*>(pointer)) T(value);
    }

    void destroy(T* pointer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pointer);
        pointer->~T();
    }

private:
    Allocator& operator=(const Allocator& other) NN_NOEXCEPT;

    Allocator& operator=(Allocator&& other) NN_NOEXCEPT;
};

template <class T, class U>
bool operator==(const Allocator<T>& lhs, const Allocator<U>& rhs) NN_NOEXCEPT
{
    NN_UNUSED(lhs);
    NN_UNUSED(rhs);
    return true;
}

template <class T, class U>
bool operator!=(const Allocator<T>& lhs, const Allocator<U>& rhs) NN_NOEXCEPT
{
    NN_UNUSED(lhs);
    NN_UNUSED(rhs);
    return false;
}

//!< マップのキーを扱うクラスです。
class MapKey final
{
private:
    char* m_Chars = nullptr;

    size_t m_Size = 0;

public:
    NN_IMPLICIT MapKey(const MapKey& value) NN_NOEXCEPT
    {
        this->Assign(value.GetString(), value.GetCount());
    }

    NN_IMPLICIT MapKey(MapKey&& value) NN_NOEXCEPT
    {
        ::std::swap(m_Chars, value.m_Chars);
        ::std::swap(m_Size, value.m_Size);
    }

    explicit MapKey(const char chars[]) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(chars);
        const auto countMax =
            static_cast<int>(sizeof(SettingsName) + sizeof(SettingsItemKey));
        this->Assign(chars, ::nn::util::Strnlen(chars, countMax));
    }

    MapKey(const char chars[], int count) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(chars);
        NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);
        this->Assign(chars, count);
    }

    ~MapKey() NN_NOEXCEPT
    {
        this->Reset();
    }

    const char* GetString() const NN_NOEXCEPT
    {
        return m_Chars;
    }

    int GetCount() const NN_NOEXCEPT
    {
        return static_cast<int>(m_Size) - 1;
    }

    MapKey& operator=(const MapKey& value) NN_NOEXCEPT
    {
        if (this != &value)
        {
            this->Assign(value.GetString(), value.GetCount());
        }

        return *this;
    }

    MapKey& operator=(MapKey&& value) NN_NOEXCEPT
    {
        if (this != &value)
        {
            this->Reset();
            ::std::swap(m_Chars, value.m_Chars);
            ::std::swap(m_Size, value.m_Size);
        }

        return *this;
    }

    MapKey& Assign(const char chars[], int count) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(chars);
        NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);
        this->Reset();
        m_Size = static_cast<size_t>(count + 1);
        m_Chars = reinterpret_cast<char*>(AllocateFromHeap(m_Size));
        ::std::memcpy(m_Chars, chars, static_cast<size_t>(count));
        m_Chars[count] = '\0';
        return *this;
    }

    MapKey& Append(char value) NN_NOEXCEPT
    {
        const char chars[] = { value, '\0' };
        return this->Append(chars, 1);
    }

    MapKey& Append(const char chars[], int count) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(chars);
        NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);
        auto nextSize = m_Size + static_cast<size_t>(count);
        auto nextChars = reinterpret_cast<char*>(AllocateFromHeap(nextSize));
        ::std::memcpy(
            nextChars,
            this->GetString(), static_cast<size_t>(this->GetCount()));
        ::std::memcpy(
            nextChars + this->GetCount(), chars, static_cast<size_t>(count));
        nextChars[nextSize - 1] = '\0';
        this->Reset();
        m_Size = nextSize;
        m_Chars = nextChars;
        return *this;
    }

    size_t Find(const MapKey& value) const NN_NOEXCEPT
    {
        return static_cast<size_t>(
            ::std::search(
                this->GetString(), this->GetString() + this->GetCount(),
                value.GetString(), value.GetString() + value.GetCount()
                ) - this->GetString());
    }

private:
    void Reset() NN_NOEXCEPT
    {
        if (m_Chars != nullptr)
        {
            FreeToHeap(m_Chars, m_Size);
            m_Chars = nullptr;
            m_Size = 0;
        }
    }
};

bool operator<(const MapKey& lhs, const MapKey& rhs) NN_NOEXCEPT
{
    return ::std::strncmp(
        lhs.GetString(),
        rhs.GetString(),
        static_cast<size_t>(::std::max(lhs.GetCount(), rhs.GetCount()))) < 0;
}

//!< マップのキーのセパレータ
const char MapKeySeparator = '!';

//!< マップのキーのアロケーションに必要なバイトサイズ
const size_t MapKeyBufferSize =
    2 * (sizeof(SettingsName) + sizeof(SettingsItemKey));

//!< マップのキーを作成します。
MapKey MakeMapKey(
    const SettingsName& name, const SettingsItemKey& key) NN_NOEXCEPT;

//!< マップの値を表す構造体です。
struct MapValue final
{
    uint8_t type;       //!< マップの値の種別
    NN_PADDING7;
    size_t currentSize; //!< マップの現在値のサイズ
    size_t defaultSize; //!< マップの既定値のサイズ
    void* currentValue; //!< マップの現在値
    void* defaultValue; //!< マップの既定値
};

//!< キーバリューストアの項目からマップの値の取得します。（デバッグ用）
::nn::Result GetMapValueOfKeyValueStoreItemForDebug(
    MapValue* pOutValue, const KeyValueStoreItemForDebug& item) NN_NOEXCEPT;

//!< ヒープへマップの値を開放します。
void FreeMapValueToHeap(const MapValue& mapValue) NN_NOEXCEPT;

//!< マップのアロケータを表す型です。
typedef Allocator<::std::pair<const MapKey, MapValue>> MapAllocator;

//!< マップを表す型です。
typedef ::std::map<MapKey, MapValue, ::std::less<MapKey>, MapAllocator> Map;

//!< マップのエントリのアロケーションに必要なバイトサイズ
const size_t MapEntryBufferSize = 64 + sizeof(Map::value_type);

//!< キーバリューストアの実体となるマップを取得します。
::nn::Result GetKeyValueStoreMap(Map** ppOutValue) NN_NOEXCEPT;

//!< キーバリューストアの実体となるマップを強制的に取得します。（デバッグ用）
::nn::Result GetKeyValueStoreMapForciblyForDebug(Map** ppOutValue) NN_NOEXCEPT;

//!< キーバリューストアの実体となるマップをロードします。
::nn::Result LoadKeyValueStoreMap(Map* pOutValue) NN_NOEXCEPT;

//!< キーバリューストアの実体となるマップをロードします。（デバッグ用）
::nn::Result LoadKeyValueStoreMapForDebug(
    Map* pOutValue,
    SystemSaveData* pSystemSaveData,
    SystemSaveData* pFwdbgSystemData,
    SystemSaveData* pPfCfgSystemData) NN_NOEXCEPT;

//!< キーバリューストアの実体となるマップをセーブします。
::nn::Result SaveKeyValueStoreMap(const Map& value) NN_NOEXCEPT;

//!< キーバリューストアの実体となるマップの既定値をセーブします。（デバッグ用）
::nn::Result SaveKeyValueStoreMapDefaultForDebug(
    SystemSaveData& data, const Map& map) NN_NOEXCEPT;

//!< マップの値を比較します。
bool CompareValue(
    const void* lhsBuffer, size_t lhsSize,
    const void* rhsBuffer, size_t rhsSize) NN_NOEXCEPT;

//!< システムセーブデータを読み出します。
::nn::Result ReadSystemSaveData(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT;

//!< システムデータからファームウェアデバッグ設定を読み出します。
::nn::Result ReadSystemDataFirmwareDebug(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT;

//!< システムデータからプラットフォーム構成情報を読み出します。
::nn::Result ReadSystemDataPlatformConfigration(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT;

//!< キーバリューストア操作排他用ミューテックス
::nn::os::SdkMutexType g_KeyValueStoreMutex = NN_OS_SDK_MUTEX_INITIALIZER();

} // namespace

KeyValueStore::KeyValueStore(const SettingsName& name) NN_NOEXCEPT
    : m_Name(name)
{
}

::nn::Result KeyValueStore::GetValueSize(
    uint64_t* pOutCount, const SettingsItemKey& key) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);

    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    Map* pMap = nullptr;

    NN_RESULT_DO(GetKeyValueStoreMap(&pMap));

    NN_SDK_ASSERT_NOT_NULL(pMap);

    NN_RESULT_THROW_UNLESS(
        GetHeapAllocatableSize() >= MapKeyBufferSize,
        ResultSettingsItemKeyAllocationFailed());

    const Map::const_iterator iter = pMap->find(MakeMapKey(m_Name, key));

    NN_RESULT_THROW_UNLESS(iter != pMap->end(), ResultSettingsItemNotFound());

    *pOutCount = iter->second.currentSize;

    NN_RESULT_SUCCESS;
}

::nn::Result KeyValueStore::GetValue(
    uint64_t* pOutCount, char* outBuffer, size_t count,
    const SettingsItemKey& key) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);

    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    Map* pMap = nullptr;

    NN_RESULT_DO(GetKeyValueStoreMap(&pMap));

    NN_SDK_ASSERT_NOT_NULL(pMap);

    NN_RESULT_THROW_UNLESS(
        GetHeapAllocatableSize() >= MapKeyBufferSize,
        ResultSettingsItemKeyAllocationFailed());

    const Map::const_iterator iter = pMap->find(MakeMapKey(m_Name, key));

    NN_RESULT_THROW_UNLESS(iter != pMap->end(), ResultSettingsItemNotFound());

    const MapValue& mapValue = iter->second;

    const size_t size = ::std::min(mapValue.currentSize, count);

    if (size > 0)
    {
        NN_SDK_ASSERT_NOT_NULL(mapValue.currentValue);

        ::std::memcpy(outBuffer, mapValue.currentValue, size);
    }

    *pOutCount = size;

    NN_RESULT_SUCCESS;
}

::nn::Result KeyValueStore::SetValue(
    const SettingsItemKey& key, const void* buffer, size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    Map* pMap = nullptr;

    NN_RESULT_DO(GetKeyValueStoreMap(&pMap));

    NN_SDK_ASSERT_NOT_NULL(pMap);

    NN_RESULT_THROW_UNLESS(
        GetHeapAllocatableSize() >= MapKeyBufferSize,
        ResultSettingsItemKeyAllocationFailed());

    const Map::iterator iter = pMap->find(MakeMapKey(m_Name, key));

    NN_RESULT_THROW_UNLESS(iter != pMap->end(), ResultSettingsItemNotFound());

    MapValue& mapValue = iter->second;

    if (CompareValue(
            mapValue.currentValue, mapValue.currentSize, buffer, count))
    {
        NN_RESULT_SUCCESS;
    }

    size_t valueSize = count;

    void* valueBuffer = nullptr;

    if (CompareValue(
            mapValue.defaultValue, mapValue.defaultSize, buffer, count))
    {
        valueBuffer = mapValue.defaultValue;
    }
    else if (count > 0)
    {
        NN_RESULT_THROW_UNLESS(
            GetHeapAllocatableSize() >= valueSize,
            ResultSettingsItemValueAllocationFailed());

        valueBuffer = AllocateFromHeap(valueSize);

        NN_SDK_ASSERT_NOT_NULL(valueBuffer);

        ::std::memcpy(valueBuffer, buffer, valueSize);
    }

    ::std::swap(mapValue.currentSize, valueSize);

    ::std::swap(mapValue.currentValue, valueBuffer);

    const ::nn::Result result = SaveKeyValueStoreMap(*pMap);

    if (result.IsFailure())
    {
        ::std::swap(mapValue.currentSize, valueSize);

        ::std::swap(mapValue.currentValue, valueBuffer);
    }

    if (valueBuffer != nullptr && valueBuffer != mapValue.defaultValue)
    {
        FreeToHeap(valueBuffer, valueSize);
    }

    if (result.IsFailure())
    {
        SaveKeyValueStoreMap(*pMap);

        NN_RESULT_THROW(result);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result KeyValueStore::ResetValue(const SettingsItemKey& key) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    Map* pMap = nullptr;

    NN_RESULT_DO(GetKeyValueStoreMap(&pMap));

    NN_SDK_ASSERT_NOT_NULL(pMap);

    NN_RESULT_THROW_UNLESS(
        GetHeapAllocatableSize() >= MapKeyBufferSize,
        ResultSettingsItemKeyAllocationFailed());

    const Map::iterator iter = pMap->find(MakeMapKey(m_Name, key));

    NN_RESULT_THROW_UNLESS(iter != pMap->end(), ResultSettingsItemNotFound());

    MapValue& mapValue = iter->second;

    if (mapValue.currentValue == mapValue.defaultValue)
    {
        NN_RESULT_SUCCESS;
    }

    size_t valueSize = mapValue.currentSize;

    void* valueBuffer = mapValue.currentValue;

    mapValue.currentSize = mapValue.defaultSize;

    mapValue.currentValue = mapValue.defaultValue;

    const ::nn::Result result = SaveKeyValueStoreMap(*pMap);

    if (result.IsSuccess())
    {
        if (valueBuffer != nullptr && valueBuffer != mapValue.defaultValue)
        {
            FreeToHeap(valueBuffer, valueSize);
        }
    }
    else
    {
        mapValue.currentSize = valueSize;

        mapValue.currentValue = valueBuffer;

        SaveKeyValueStoreMap(*pMap);

        NN_RESULT_THROW(result);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result KeyValueStore::CreateKeyIterator(
    KeyValueStoreKeyIterator* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    Map* pMap = nullptr;

    NN_RESULT_DO(GetKeyValueStoreMap(&pMap));

    NN_SDK_ASSERT_NOT_NULL(pMap);

    NN_RESULT_THROW_UNLESS(
        GetHeapAllocatableSize() >= MapKeyBufferSize,
        ResultSettingsItemKeyAllocationFailed());

    MapKey mapKeyHeader(m_Name.value);

    mapKeyHeader.Append(MapKeySeparator);

    const MapKey* pMapKey = nullptr;

    for (const Map::value_type& pair : *pMap)
    {
        if (mapKeyHeader < pair.first && pair.first.Find(mapKeyHeader) == 0)
        {
            pMapKey = &pair.first;

            break;
        }
    }

    NN_RESULT_THROW_UNLESS(pMapKey != nullptr, ResultSettingsItemNotFound());

    const auto entireSize = static_cast<size_t>(pMapKey->GetCount() + 1);

    NN_RESULT_THROW_UNLESS(
        GetHeapAllocatableSize() >= entireSize,
        ResultSettingsItemKeyIteratorAllocationFailed());

    auto buffer = reinterpret_cast<char*>(AllocateFromHeap(entireSize));

    NN_SDK_ASSERT_NOT_NULL(buffer);

    ::std::memcpy(buffer, pMapKey->GetString(), entireSize);

    pOutValue->headerSize = static_cast<size_t>(mapKeyHeader.GetCount());
    pOutValue->entireSize = entireSize;
    pOutValue->mapKey = buffer;

    NN_RESULT_SUCCESS;
}

void DestroyKeyValueStoreKeyIterator(
    KeyValueStoreKeyIterator* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_GREATER(pOutValue->headerSize, size_t(0));
    NN_SDK_REQUIRES_LESS(pOutValue->headerSize, pOutValue->entireSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue->mapKey);

    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    FreeToHeap(pOutValue->mapKey, pOutValue->entireSize);

    pOutValue->headerSize = 0;
    pOutValue->entireSize = 0;
    pOutValue->mapKey = nullptr;
}

::nn::Result AdvanceKeyValueStoreKeyIterator(
    KeyValueStoreKeyIterator* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_GREATER(pOutValue->headerSize, size_t(0));
    NN_SDK_REQUIRES_LESS(pOutValue->headerSize, pOutValue->entireSize);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue->mapKey);

    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    Map* pMap = nullptr;

    NN_RESULT_DO(GetKeyValueStoreMap(&pMap));

    NN_SDK_ASSERT_NOT_NULL(pMap);

    NN_RESULT_THROW_UNLESS(
        GetHeapAllocatableSize() >= MapKeyBufferSize,
        ResultSettingsItemKeyAllocationFailed());

    Map::const_iterator iter = pMap->find(MapKey(
        pOutValue->mapKey, static_cast<int>(pOutValue->entireSize) - 1));

    NN_RESULT_THROW_UNLESS(
        iter != pMap->end(), ResultInvalidSettingsItemKeyIterator());

    ++iter;

    NN_RESULT_THROW_UNLESS(iter != pMap->end(), ResultStopIteration());

    const MapKey& mapKey = iter->first;

    const size_t headerSize = pOutValue->headerSize;

    NN_RESULT_THROW_UNLESS(
        ::std::strncmp(mapKey.GetString(), pOutValue->mapKey, headerSize) == 0,
        ResultStopIteration());

    const auto entireSize = static_cast<size_t>(mapKey.GetCount() + 1);

    NN_RESULT_THROW_UNLESS(
        GetHeapAllocatableSize() >= entireSize,
        ResultSettingsItemKeyIteratorAllocationFailed());

    FreeToHeap(pOutValue->mapKey, pOutValue->entireSize);

    auto buffer = reinterpret_cast<char*>(AllocateFromHeap(entireSize));

    NN_SDK_ASSERT_NOT_NULL(buffer);

    ::std::memcpy(buffer, mapKey.GetString(), entireSize);

    pOutValue->entireSize = entireSize;
    pOutValue->mapKey = buffer;

    NN_RESULT_SUCCESS;
}

::nn::Result GetKeyValueStoreKeyIteratorKeySize(
    uint64_t* pOutCount,
    const KeyValueStoreKeyIterator& iterator) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_GREATER(iterator.headerSize, size_t(0));
    NN_SDK_REQUIRES_LESS(iterator.headerSize, iterator.entireSize);
    NN_SDK_REQUIRES_NOT_NULL(iterator.mapKey);

    *pOutCount = ::std::min(
        iterator.entireSize - iterator.headerSize,
        static_cast<size_t>(
            ::nn::settings::fwdbg::SettingsItemKeyLengthMax + 1));

    NN_RESULT_SUCCESS;
}

::nn::Result GetKeyValueStoreKeyIteratorKey(
    uint64_t* pOutCount, char* outBuffer, size_t count,
    const KeyValueStoreKeyIterator& iterator) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES_GREATER(iterator.headerSize, size_t(0));
    NN_SDK_REQUIRES_LESS(iterator.headerSize, iterator.entireSize);
    NN_SDK_REQUIRES_NOT_NULL(iterator.mapKey);

    const size_t size = ::std::min(
        count,
        ::std::min(iterator.entireSize - iterator.headerSize,
                   static_cast<size_t>(
                       ::nn::settings::fwdbg::SettingsItemKeyLengthMax + 1)));

    ::strncpy(outBuffer, &iterator.mapKey[iterator.headerSize], size);

    if (size > 0)
    {
        outBuffer[size - 1] = '\0';
    }

    *pOutCount = size;

    NN_RESULT_SUCCESS;
}

::nn::Result GetKeyValueStoreItemCountForDebug(uint64_t* pOutCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);

    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    Map* pMap = nullptr;

    NN_RESULT_DO(GetKeyValueStoreMap(&pMap));

    NN_SDK_ASSERT_NOT_NULL(pMap);

    *pOutCount = pMap->size();

    NN_RESULT_SUCCESS;
}

::nn::Result GetKeyValueStoreItemForDebug(
    uint64_t* pOutCount,
    KeyValueStoreItemForDebug outItems[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outItems);

    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    Map* pMap = nullptr;

    NN_RESULT_DO(GetKeyValueStoreMap(&pMap));

    NN_SDK_ASSERT_NOT_NULL(pMap);

    size_t index = 0;

    for (const Map::value_type& pair : *pMap)
    {
        if (index >= count)
        {
            break;
        }
        else
        {
            KeyValueStoreItemForDebug& item = outItems[index++];
            item.mapKey = pair.first.GetString();
            item.type = pair.second.type;
            item.currentSize = pair.second.currentSize;
            item.defaultSize = pair.second.defaultSize;
            item.currentValue = pair.second.currentValue;
            item.defaultValue = pair.second.defaultValue;
        }
    }

    *pOutCount = index;

    NN_RESULT_SUCCESS;
}

::nn::Result AddKeyValueStoreItemForDebug(
    const KeyValueStoreItemForDebug items[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(items);

    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    Map* pMap = nullptr;

    NN_RESULT_DO(GetKeyValueStoreMapForciblyForDebug(&pMap));

    NN_SDK_ASSERT_NOT_NULL(pMap);

    for (size_t i = 0; i < count; ++i)
    {
        const KeyValueStoreItemForDebug& item = items[i];

        MapValue mapValue = {};

        NN_UTIL_SCOPE_EXIT
        {
            FreeMapValueToHeap(mapValue);
        };

        NN_RESULT_DO(GetMapValueOfKeyValueStoreItemForDebug(&mapValue, item));

        NN_RESULT_THROW_UNLESS(
            GetHeapAllocatableSize() >= MapKeyBufferSize,
            ResultSettingsItemKeyAllocationFailed());

        MapKey mapKey(item.mapKey);

        const Map::iterator iter = pMap->find(mapKey);

        if (iter != pMap->end())
        {
            FreeMapValueToHeap(iter->second);

            iter->second = mapValue;
        }
        else
        {
            NN_RESULT_THROW_UNLESS(
                GetHeapAllocatableSize() >= MapEntryBufferSize,
                ResultSettingsItemValueAllocationFailed());

            (*pMap)[::std::move(mapKey)] = mapValue;
        }

        mapValue.currentValue = nullptr;
        mapValue.defaultValue = nullptr;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SaveKeyValueStoreAllForDebug(SystemSaveData* pData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);

    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    Map* pMap = nullptr;

    NN_RESULT_DO(GetKeyValueStoreMap(&pMap));

    NN_SDK_ASSERT_NOT_NULL(pMap);

    NN_RESULT_DO(SaveKeyValueStoreMapDefaultForDebug(*pData, *pMap));

    NN_RESULT_DO(SaveKeyValueStoreMap(*pMap));

    NN_RESULT_SUCCESS;
}

::nn::Result ReloadKeyValueStoreForDebug() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    Map* pMap = nullptr;

    NN_RESULT_DO(GetKeyValueStoreMap(&pMap));

    NN_SDK_ASSERT_NOT_NULL(pMap);

    NN_RESULT_DO(LoadKeyValueStoreMap(pMap));

    NN_RESULT_SUCCESS;
}

::nn::Result ReloadKeyValueStoreForDebug(
    SystemSaveData* pSystemSaveData,
    SystemSaveData* pFwdbgSystemData,
    SystemSaveData* pPfCfgSystemData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_SDK_REQUIRES_NOT_NULL(pFwdbgSystemData);
    NN_SDK_REQUIRES_NOT_NULL(pPfCfgSystemData);

    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    Map* pMap = nullptr;

    NN_RESULT_DO(GetKeyValueStoreMapForciblyForDebug(&pMap));

    NN_SDK_ASSERT_NOT_NULL(pMap);

    NN_RESULT_DO(
        LoadKeyValueStoreMapForDebug(
            pMap, pSystemSaveData, pFwdbgSystemData, pPfCfgSystemData));

    NN_RESULT_SUCCESS;
}

::nn::Result ReadKeyValueStoreSaveData(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);

    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    NN_RESULT_DO(ReadSystemSaveData(pOutCount, outBuffer, count));

    NN_RESULT_SUCCESS;
}

::nn::Result ReadKeyValueStoreFirmwareDebug(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);

    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    NN_RESULT_DO(ReadSystemDataFirmwareDebug(pOutCount, outBuffer, count));

    NN_RESULT_SUCCESS;
}

::nn::Result ReadKeyValueStorePlatformConfigration(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);

    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    NN_RESULT_DO(
        ReadSystemDataPlatformConfigration(pOutCount, outBuffer, count));

    NN_RESULT_SUCCESS;
}

::nn::Result ResetKeyValueStoreSaveData() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(g_KeyValueStoreMutex)
                      > locker(g_KeyValueStoreMutex);

    Map* pMap = nullptr;

    NN_RESULT_DO(GetKeyValueStoreMap(&pMap));

    NN_SDK_ASSERT_NOT_NULL(pMap);

    for (Map::value_type& pair : *pMap)
    {
        MapValue& mapValue = pair.second;

        if (mapValue.currentValue != mapValue.defaultValue)
        {
            size_t valueSize = mapValue.currentSize;

            void* valueBuffer = mapValue.currentValue;

            mapValue.currentSize = mapValue.defaultSize;

            mapValue.currentValue = mapValue.defaultValue;

            if (valueBuffer != nullptr)
            {
                FreeToHeap(valueBuffer, valueSize);
            }
        }
    }

    NN_RESULT_DO(SaveKeyValueStoreMap(*pMap));

    NN_RESULT_SUCCESS;
}

namespace {

//!< システムデータのタグ定義
struct SystemDataTag final
{
    //!< ファームウェアデバッグ設定
    struct Fwdbg final {};

    //!< プラットフォーム構成情報
    struct PfCfg final {};
};

//!< ヒープメモリのハンドルを返します。
::nn::lmem::HeapHandle& GetHeapHandle() NN_NOEXCEPT
{
    static ::nn::lmem::HeapHandle s_HeapHandle = nullptr;

    static bool s_IsInitialized = false;

    if (!s_IsInitialized)
    {
        static char s_HeapMemory[HeapMemorySize] = {};

        s_HeapHandle = ::nn::lmem::CreateExpHeap(
            s_HeapMemory,
            sizeof(s_HeapMemory),
            ::nn::lmem::CreationOption_NoOption);

        s_IsInitialized = true;
    }

    return s_HeapHandle;
}

void* AllocateFromHeap(size_t size) NN_NOEXCEPT
{
    const auto alignment = sizeof(void*);

    return ::nn::lmem::AllocateFromExpHeap(GetHeapHandle(), size, alignment);
}

void FreeToHeap(void* pointer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);

    ::nn::lmem::FreeToExpHeap(GetHeapHandle(), pointer);
}

size_t GetHeapAllocatableSize() NN_NOEXCEPT
{
    const auto alignment = static_cast<int>(sizeof(void*));

    return ::nn::lmem::GetExpHeapAllocatableSize(GetHeapHandle(), alignment);
}

MapKey MakeMapKey(
    const SettingsName& name, const SettingsItemKey& key) NN_NOEXCEPT
{
    MapKey mapKey(
        name.value, ::nn::util::Strnlen(name.value, NN_ARRAY_SIZE(name.value)));
    mapKey.Append(MapKeySeparator);
    mapKey.Append(
        key.value, ::nn::util::Strnlen(key.value, NN_ARRAY_SIZE(key.value)));
    return mapKey;
}

::nn::Result GetMapValueOfKeyValueStoreItemForDebug(
    MapValue* pOutValue, const KeyValueStoreItemForDebug& item) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    MapValue mapValue = {};
    mapValue.type = item.type;
    mapValue.currentSize = item.currentSize;
    mapValue.defaultSize = item.defaultSize;

    NN_UTIL_SCOPE_EXIT
    {
        FreeMapValueToHeap(mapValue);
    };

    if (mapValue.defaultSize > 0)
    {
        NN_RESULT_THROW_UNLESS(
            GetHeapAllocatableSize() >= mapValue.defaultSize,
            ResultSettingsItemValueAllocationFailed());

        mapValue.defaultValue = AllocateFromHeap(mapValue.defaultSize);

        NN_SDK_ASSERT_NOT_NULL(mapValue.defaultValue);

        ::std::memcpy(
            mapValue.defaultValue, item.defaultValue, mapValue.defaultSize);
    }

    if (CompareValue(item.currentValue, item.currentSize,
                     item.defaultValue, item.defaultSize))
    {
        mapValue.currentSize = mapValue.defaultSize;

        mapValue.currentValue = mapValue.defaultValue;
    }
    else if (mapValue.currentSize > 0)
    {
        NN_RESULT_THROW_UNLESS(
            GetHeapAllocatableSize() >= mapValue.currentSize,
            ResultSettingsItemValueAllocationFailed());

        mapValue.currentValue = AllocateFromHeap(mapValue.currentSize);

        NN_SDK_ASSERT_NOT_NULL(mapValue.currentValue);

        ::std::memcpy(
            mapValue.currentValue, item.currentValue, mapValue.currentSize);
    }

    *pOutValue = mapValue;

    mapValue.currentValue = nullptr;
    mapValue.defaultValue = nullptr;

    NN_RESULT_SUCCESS;
}

void FreeMapValueToHeap(const MapValue& mapValue) NN_NOEXCEPT
{
    if (mapValue.currentValue != nullptr &&
        mapValue.currentValue != mapValue.defaultValue)
    {
        FreeToHeap(mapValue.currentValue, mapValue.currentSize);
    }

    if (mapValue.defaultValue != nullptr)
    {
        FreeToHeap(mapValue.defaultValue, mapValue.defaultSize);
    }
}

//!< キーバリューストアの実体となるマップを取得します。
::nn::Result GetKeyValueStoreMap(Map** ppOutValue, bool isForced) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutValue);

    static ::nn::util::TypedStorage<
        Map, sizeof(Map), NN_ALIGNOF(Map)> s_Storage;

    Map& map = ::nn::util::Get(s_Storage);

    static bool s_IsInitialized = false;

    if (!s_IsInitialized)
    {
        new(&map) Map();

        s_IsInitialized = true;
    }

    static bool s_IsLoaded = false;

    if (!s_IsLoaded)
    {
        const ::nn::Result result = LoadKeyValueStoreMap(&map);

        if (!isForced)
        {
            NN_RESULT_DO(result);
        }

        s_IsLoaded = true;
    }

    *ppOutValue = &map;

    NN_RESULT_SUCCESS;
}

::nn::Result GetKeyValueStoreMap(Map** ppOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutValue);

    NN_RESULT_DO(GetKeyValueStoreMap(ppOutValue, false));

    NN_RESULT_SUCCESS;
}

::nn::Result GetKeyValueStoreMapForciblyForDebug(Map** ppOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutValue);

    NN_RESULT_DO(GetKeyValueStoreMap(ppOutValue, true));

    NN_RESULT_SUCCESS;
}

//!< キーバリューストアの実体となるマップをクリアします。
void ClearKeyValueStoreMap(Map& map) NN_NOEXCEPT
{
    for (const Map::value_type& pair : map)
    {
        FreeMapValueToHeap(pair.second);
    }

    map.clear();
}

//!< システムデータのマウント名を返します。
template<typename Tag>
const char* GetSystemDataMountName() NN_NOEXCEPT;

//!< ファームウェアデバッグ設定のシステムデータのマウント名を返します。
template<>
const char* GetSystemDataMountName<SystemDataTag::Fwdbg>() NN_NOEXCEPT
{
    return FwdbgSystemDataMountName;
}

//!< //!< プラットフォーム構成情報のシステムデータのマウント名を返します。
template<>
const char* GetSystemDataMountName<SystemDataTag::PfCfg>() NN_NOEXCEPT
{
    return PfCfgSystemDataMountName;
}

//!< システムデータを返します。
template<typename Tag>
::nn::Result GetSystemData(
    SystemData** ppOutValue, ::nn::ncm::SystemDataId systemDataId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutValue);

    static ::nn::util::TypedStorage<SystemData,
                             sizeof(SystemData),
                         NN_ALIGNOF(SystemData)> s_Storage;

    SystemData& systemData = ::nn::util::Get(s_Storage);

    static bool s_IsInitialized = false;

    if (!s_IsInitialized)
    {
        new(&systemData) SystemData();
        systemData.SetSystemDataId(systemDataId);
        systemData.SetMountName(GetSystemDataMountName<Tag>());

        s_IsInitialized = true;
    }

    static bool s_IsMounted = false;

    if (!s_IsMounted)
    {
        NN_RESULT_DO(systemData.Mount());

        s_IsMounted = true;
    }

    *ppOutValue = &systemData;

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータを返します。
::nn::Result GetSystemSaveData(
    SystemSaveData** ppOutValue, bool creates) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutValue);

    static ::nn::util::TypedStorage<SystemSaveData,
                             sizeof(SystemSaveData),
                         NN_ALIGNOF(SystemSaveData)> s_Storage;

    SystemSaveData& systemSaveData = ::nn::util::Get(s_Storage);

    static bool s_IsInitialized = false;

    if (!s_IsInitialized)
    {
        new(&systemSaveData) SystemSaveData();
        systemSaveData.SetSystemSaveDataId(SystemSaveDataId);
        systemSaveData.SetTotalSize(SystemSaveDataTotalSize);
        systemSaveData.SetJournalSize(SystemSaveDataJournalSize);
        systemSaveData.SetFlags(SystemSaveDataFlags);
        systemSaveData.SetMountName(SystemSaveDataMountName);

        s_IsInitialized = true;
    }

    static bool s_IsMounted = false;

    if (!s_IsMounted)
    {
        NN_RESULT_DO(systemSaveData.Mount(creates));

        s_IsMounted = true;
    }

    *ppOutValue = &systemSaveData;

    NN_RESULT_SUCCESS;
}

//!< ファイルの内容を読み出します。
template<typename T>
::nn::Result ReadData(
    T& data, int64_t& offset, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    NN_RESULT_DO(data.Read(offset, buffer, size));

    offset += static_cast<int64_t>(size);

    NN_RESULT_SUCCESS;
}

//!< ファイルの内容をアロケートしたメモリ領域に読み出します。
template<typename T>
::nn::Result ReadDataToHeap(
    T& data, int64_t& offset, void** pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER(size, size_t(0));

    *pBuffer = AllocateFromHeap(size);

    NN_SDK_ASSERT_NOT_NULL(*pBuffer);

    const ::nn::Result result = ReadData(data, offset, *pBuffer, size);

    if (result.IsFailure())
    {
        FreeToHeap(*pBuffer, size);

        *pBuffer = nullptr;

        NN_RESULT_THROW(result);
    }

    NN_RESULT_SUCCESS;
}

//!< キーバリューストアの実体となるマップのエントリをロードします。
template<typename T>
::nn::Result LoadKeyValueStoreMapEntry(
    T& data,
    int64_t& offset,
    ::std::function<
        ::nn::Result(const MapKey&, uint8_t, const void*, uint32_t) NN_NOEXCEPT
        > load) NN_NOEXCEPT
{
    auto keySize = uint32_t();

    NN_RESULT_DO(ReadData(data, offset, &keySize, sizeof(keySize)));

    NN_SDK_ASSERT_GREATER(keySize, uint32_t(1));

    NN_RESULT_THROW_UNLESS(
        GetHeapAllocatableSize() >= keySize,
        ResultSettingsItemKeyAllocationFailed());

    void* keyBuffer = nullptr;

    NN_RESULT_DO(ReadDataToHeap(data, offset, &keyBuffer, keySize));

    NN_SDK_ASSERT_NOT_NULL(keyBuffer);

    NN_UTIL_SCOPE_EXIT
    {
        FreeToHeap(keyBuffer, keySize);
    };

    NN_RESULT_THROW_UNLESS(
        GetHeapAllocatableSize() >= MapKeyBufferSize,
        ResultSettingsItemKeyAllocationFailed());

    MapKey mapKey(
        reinterpret_cast<char*>(keyBuffer), static_cast<int>(keySize) - 1);

    auto type = uint8_t();

    NN_RESULT_DO(ReadData(data, offset, &type, sizeof(type)));

    auto valueSize = uint32_t();

    NN_RESULT_DO(ReadData(data, offset, &valueSize, sizeof(valueSize)));

    void* valueBuffer = nullptr;

    NN_UTIL_SCOPE_EXIT
    {
        if (valueBuffer != nullptr)
        {
            FreeToHeap(valueBuffer, valueSize);
        }
    };

    if (valueSize > 0)
    {
        NN_RESULT_THROW_UNLESS(
            GetHeapAllocatableSize() >= valueSize,
            ResultSettingsItemValueAllocationFailed());

        NN_RESULT_DO(ReadDataToHeap(data, offset, &valueBuffer, valueSize));
    }

    NN_RESULT_DO(load(mapKey, type, valueBuffer, valueSize));

    NN_RESULT_SUCCESS;
}

//!< キーバリューストアの実体となるマップのエントリをロードします。
template<typename T>
::nn::Result LoadKeyValueStoreMapEntries(
    T& data,
    ::std::function<
        ::nn::Result(const MapKey&, uint8_t, const void*, uint32_t) NN_NOEXCEPT
        > load) NN_NOEXCEPT
{
    int64_t offset = 0;

    auto totalSize = uint32_t();

    NN_RESULT_DO(ReadData(data, offset, &totalSize, sizeof(totalSize)));

    while (offset < totalSize)
    {
        NN_RESULT_DO(LoadKeyValueStoreMapEntry(data, offset, load));
    }

    NN_RESULT_SUCCESS;
}

//!< キーバリューストアの実体となるマップの既定値をロードします。
template<typename T>
::nn::Result LoadKeyValueStoreMapDefault(Map* pOutValue, T& data) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NN_RESULT_DO(data.OpenToRead());

    const ::nn::Result result = LoadKeyValueStoreMapEntries(data, [&] (
        const MapKey& mapKey,
        uint8_t type,
        const void* valueBuffer, uint32_t valueSize) NN_NOEXCEPT -> ::nn::Result
    {
        NN_RESULT_THROW_UNLESS(
            GetHeapAllocatableSize() >= MapKeyBufferSize,
            ResultSettingsItemKeyAllocationFailed());

        MapKey movableMapKey(mapKey);

        void* defaultBuffer = nullptr;

        NN_UTIL_SCOPE_EXIT
        {
            if (defaultBuffer != nullptr)
            {
                FreeToHeap(defaultBuffer, valueSize);
            }
        };

        if (valueSize > 0)
        {
            NN_RESULT_THROW_UNLESS(
                GetHeapAllocatableSize() >= valueSize,
                ResultSettingsItemValueAllocationFailed());

            defaultBuffer = AllocateFromHeap(valueSize);

            NN_SDK_ASSERT_NOT_NULL(defaultBuffer);

            ::std::memcpy(defaultBuffer, valueBuffer, valueSize);
        }

        MapValue mapValue = {};
        mapValue.type = type;
        mapValue.currentSize = static_cast<size_t>(valueSize);
        mapValue.defaultSize = static_cast<size_t>(valueSize);
        mapValue.currentValue = defaultBuffer;
        mapValue.defaultValue = defaultBuffer;

        NN_RESULT_THROW_UNLESS(
            GetHeapAllocatableSize() >= MapEntryBufferSize,
            ResultSettingsItemValueAllocationFailed());

        (*pOutValue)[::std::move(movableMapKey)] = mapValue;

        defaultBuffer = nullptr;

        NN_RESULT_SUCCESS;
    });

    data.Close();

    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);

    NN_RESULT_SUCCESS;
}

//!< キーバリューストアの実体となるマップの現在値をロードします。
template<typename T>
::nn::Result LoadKeyValueStoreMapCurrent(Map* pOutValue, T& data) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NN_RESULT_DO(data.OpenToRead());

    const ::nn::Result result = LoadKeyValueStoreMapEntries(data, [&] (
        const MapKey& mapKey,
        uint8_t type,
        const void* valueBuffer, uint32_t valueSize) NN_NOEXCEPT -> ::nn::Result
    {
        NN_UNUSED(type);

        const Map::iterator iter = pOutValue->find(mapKey);

        if (iter == pOutValue->end())
        {
            NN_RESULT_SUCCESS;
        }

        MapValue& mapValue = iter->second;

        size_t currentSize = valueSize;

        void* currentBuffer = nullptr;

        if (currentSize > 0)
        {
            NN_RESULT_THROW_UNLESS(
                GetHeapAllocatableSize() >= currentSize,
                ResultSettingsItemValueAllocationFailed());

            currentBuffer = AllocateFromHeap(currentSize);

            NN_SDK_ASSERT_NOT_NULL(currentBuffer);

            ::std::memcpy(currentBuffer, valueBuffer, currentSize);
        }

        ::std::swap(mapValue.currentSize, currentSize);

        ::std::swap(mapValue.currentValue, currentBuffer);

        if (currentBuffer != nullptr && currentBuffer != mapValue.defaultValue)
        {
            FreeToHeap(currentBuffer, currentSize);
        }

        NN_RESULT_SUCCESS;
    });

    data.Close();

    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);

    NN_RESULT_SUCCESS;
}

::nn::Result LoadKeyValueStoreMap(Map* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ClearKeyValueStoreMap(*pOutValue);

    SystemData* pFwdbgSystemData = nullptr;
    NN_RESULT_DO(GetSystemData<SystemDataTag::Fwdbg>(
                     &pFwdbgSystemData, FwdbgSystemDataId));
    NN_SDK_ASSERT_NOT_NULL(pFwdbgSystemData);
    NN_RESULT_DO(LoadKeyValueStoreMapDefault(pOutValue, *pFwdbgSystemData));

    SystemData* pPfCfgSystemData = nullptr;
    switch (GetSplHardwareType())
    {
    case SplHardwareType::None:
        break;

    case SplHardwareType::Icosa:
        NN_RESULT_DO(GetSystemData<SystemDataTag::PfCfg>(
                         &pPfCfgSystemData, PfCfgIcosaSystemDataId));
        NN_SDK_ASSERT_NOT_NULL(pPfCfgSystemData);
        NN_RESULT_DO(LoadKeyValueStoreMapDefault(pOutValue, *pPfCfgSystemData));
        break;

    case SplHardwareType::IcosaMariko:
        NN_RESULT_DO(GetSystemData<SystemDataTag::PfCfg>(
                         &pPfCfgSystemData, PfCfgIcosaMarikoSystemDataId));
        NN_SDK_ASSERT_NOT_NULL(pPfCfgSystemData);
        NN_RESULT_DO(LoadKeyValueStoreMapDefault(pOutValue, *pPfCfgSystemData));
        break;

    case SplHardwareType::Copper:
        NN_RESULT_DO(GetSystemData<SystemDataTag::PfCfg>(
                         &pPfCfgSystemData, PfCfgCopperSystemDataId));
        NN_SDK_ASSERT_NOT_NULL(pPfCfgSystemData);
        NN_RESULT_DO(LoadKeyValueStoreMapDefault(pOutValue, *pPfCfgSystemData));
        break;

    case SplHardwareType::Hoag:
        NN_RESULT_DO(GetSystemData<SystemDataTag::PfCfg>(
                         &pPfCfgSystemData, PfCfgHoagSystemDataId));
        NN_SDK_ASSERT_NOT_NULL(pPfCfgSystemData);
        NN_RESULT_DO(LoadKeyValueStoreMapDefault(pOutValue, *pPfCfgSystemData));
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    if (IsSplDevelopment())
    {
        // 開発環境においてのみ書き換え値を適用

        SystemSaveData* pSystemSaveData = nullptr;

        if (GetSystemSaveData(&pSystemSaveData, false).IsFailure())
        {
            NN_RESULT_SUCCESS;
        }

        NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);

        const ::nn::Result result =
            LoadKeyValueStoreMapCurrent(pOutValue, *pSystemSaveData);

        if (result.IsFailure())
        {
            for (Map::value_type& pair : *pOutValue)
            {
                if (pair.second.currentValue != nullptr &&
                    pair.second.currentValue != pair.second.defaultValue)
                {
                    FreeToHeap(
                        pair.second.currentValue, pair.second.currentSize);
                }

                pair.second.currentSize = pair.second.defaultSize;

                pair.second.currentValue = pair.second.defaultValue;
            }

            NN_SETTINGS_FWDBG_WARN(
                "Failed to load the system save data. (%08x, %d%03d-%04d)\n",
                result.GetInnerValueForDebug(),
                ErrorCodePlatformId,
                result.GetModule(), result.GetDescription());
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result LoadKeyValueStoreMapForDebug(
    Map* pOutValue,
    SystemSaveData* pSystemSaveData,
    SystemSaveData* pFwdbgSystemData,
    SystemSaveData* pPfCfgSystemData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_SDK_REQUIRES_NOT_NULL(pFwdbgSystemData);
    NN_SDK_REQUIRES_NOT_NULL(pPfCfgSystemData);

    ClearKeyValueStoreMap(*pOutValue);

    NN_RESULT_DO(LoadKeyValueStoreMapDefault(pOutValue, *pFwdbgSystemData));
    NN_RESULT_DO(LoadKeyValueStoreMapDefault(pOutValue, *pPfCfgSystemData));

    NN_RESULT_DO(LoadKeyValueStoreMapCurrent(pOutValue, *pSystemSaveData));

    NN_RESULT_SUCCESS;
}

//!< キーバリューストアの実体となるマップのエントリをセーブします。
template<typename T>
::nn::Result SaveKeyValueStoreMapEntry(
    T& data, int64_t& offset,
    const MapKey& mapKey,
    uint8_t type, const void* buffer, uint32_t size) NN_NOEXCEPT
{
    const auto keySize = static_cast<uint32_t>(mapKey.GetCount() + 1);

    NN_RESULT_DO(data.Write(offset, &keySize, sizeof(keySize)));

    offset += static_cast<int64_t>(sizeof(keySize));

    NN_RESULT_DO(data.Write(offset, mapKey.GetString(), keySize));

    offset += static_cast<int64_t>(keySize);

    NN_RESULT_DO(data.Write(offset, &type, sizeof(type)));

    offset += static_cast<int64_t>(sizeof(type));

    NN_RESULT_DO(data.Write(offset, &size, sizeof(size)));

    offset += static_cast<int64_t>(sizeof(size));

    if (size > 0)
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);

        NN_RESULT_DO(data.Write(offset, buffer, size));

        offset += static_cast<int64_t>(size);
    }

    NN_RESULT_SUCCESS;
}

//!< キーバリューストアの実体となるマップをセーブします。
template<typename T>
::nn::Result SaveKeyValueStoreMap(
    T& data, const Map& map,
    ::std::function<
        bool(uint8_t*, const void**, uint32_t*, const MapValue&) NN_NOEXCEPT
        > test) NN_NOEXCEPT
{
    auto result = ::nn::Result();

    result = data.Create(static_cast<int64_t>(HeapMemorySize));

    NN_RESULT_THROW_UNLESS(
        result.IsSuccess() ||
        ::nn::fs::ResultPathAlreadyExists::Includes(result), result);

    NN_RESULT_DO(data.OpenToWrite());

    int64_t offset = sizeof(uint32_t);

    {
        const auto totalSize = static_cast<uint32_t>(offset);

        result = data.Write(0, &totalSize, sizeof(totalSize));
    }

    for (const Map::value_type& pair : map)
    {
        if (result.IsFailure())
        {
            break;
        }

        auto type = uint8_t();
        auto valueSize = uint32_t();
        const void* valueBuffer = nullptr;
        if (test(&type, &valueBuffer, &valueSize, pair.second))
        {
            result = SaveKeyValueStoreMapEntry(
                data, offset, pair.first, type, valueBuffer, valueSize);
        }
    }

    if (result.IsSuccess())
    {
        const auto totalSize = static_cast<uint32_t>(offset);

        result = data.Write(0, &totalSize, sizeof(totalSize));
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(data.Flush());

    data.Close();

    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);

    NN_RESULT_DO(data.Commit(false));

    NN_RESULT_SUCCESS;
}

//!< キーバリューストアの実体となるマップの既定値をセーブします。
template<typename T>
::nn::Result SaveKeyValueStoreMapDefault(T& data, const Map& map) NN_NOEXCEPT
{
    NN_RESULT_DO(SaveKeyValueStoreMap(data, map, [] (
        uint8_t* pOutType,
        const void** pOutValueBuffer, uint32_t* pOutValueSize,
        const MapValue& mapValue) NN_NOEXCEPT -> bool
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutType);
        NN_SDK_REQUIRES_NOT_NULL(pOutValueBuffer);
        NN_SDK_REQUIRES_NOT_NULL(pOutValueSize);

        *pOutType = static_cast<uint8_t>(mapValue.type);
        *pOutValueBuffer = mapValue.defaultValue;
        *pOutValueSize = static_cast<uint32_t>(mapValue.defaultSize);

        return true;
    }));

    NN_RESULT_SUCCESS;
}

//!< キーバリューストアの実体となるマップの現在値をセーブします。
template<typename T>
::nn::Result SaveKeyValueStoreMapCurrent(T& data, const Map& map) NN_NOEXCEPT
{
    NN_RESULT_DO(SaveKeyValueStoreMap(data, map, [] (
        uint8_t* pOutType,
        const void** pOutValueBuffer, uint32_t* pOutValueSize,
        const MapValue& mapValue) NN_NOEXCEPT -> bool
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutType);
        NN_SDK_REQUIRES_NOT_NULL(pOutValueBuffer);
        NN_SDK_REQUIRES_NOT_NULL(pOutValueSize);

        if (CompareValue(
                mapValue.currentValue, mapValue.currentSize,
                mapValue.defaultValue, mapValue.defaultSize))
        {
            return false;
        }
        else
        {
            *pOutType = static_cast<uint8_t>(mapValue.type);
            *pOutValueBuffer = mapValue.currentValue;
            *pOutValueSize = static_cast<uint32_t>(mapValue.currentSize);

            return true;
        }
    }));

    NN_RESULT_SUCCESS;
}

::nn::Result SaveKeyValueStoreMap(const Map& value) NN_NOEXCEPT
{
    SystemSaveData* pSystemSaveData = nullptr;

    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData, true));

    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);

    NN_RESULT_DO(SaveKeyValueStoreMapCurrent(*pSystemSaveData, value));

    NN_RESULT_SUCCESS;
}

::nn::Result SaveKeyValueStoreMapDefaultForDebug(
    SystemSaveData& data, const Map& map) NN_NOEXCEPT
{
    NN_RESULT_DO(SaveKeyValueStoreMapDefault(data, map));

    NN_RESULT_SUCCESS;
}

bool CompareValue(
    const void* lhsBuffer, size_t lhsSize,
    const void* rhsBuffer, size_t rhsSize) NN_NOEXCEPT
{
    if (lhsBuffer == rhsBuffer)
    {
        return true;
    }

    if (lhsSize != rhsSize)
    {
        return false;
    }

    if (lhsSize == 0)
    {
        return true;
    }

    if (lhsBuffer != nullptr &&
        rhsBuffer != nullptr &&
        ::std::memcmp(lhsBuffer, rhsBuffer, lhsSize) == 0)
    {
        return true;
    }

    return false;
}

template<class T>
::nn::Result ReadAllBytes(
    T* pData, uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);

    NN_RESULT_DO(pData->OpenToRead());

    NN_UTIL_SCOPE_EXIT
    {
        pData->Close();
    };

    auto totalSize = uint32_t();

    NN_RESULT_DO(pData->Read(0, &totalSize, sizeof(totalSize)));

    totalSize = ::std::min(totalSize, static_cast<uint32_t>(count));

    NN_RESULT_DO(pData->Read(0, outBuffer, totalSize));

    *pOutCount = totalSize;

    NN_RESULT_SUCCESS;
}

::nn::Result ReadSystemSaveData(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);

    SystemSaveData* pSystemSaveData = nullptr;

    if (GetSystemSaveData(&pSystemSaveData, false).IsFailure())
    {
        *pOutCount = 0;
    }
    else
    {
        NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);

        NN_RESULT_DO(
            ReadAllBytes(pSystemSaveData, pOutCount, outBuffer, count));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ReadSystemDataFirmwareDebug(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);

    ::nn::ncm::SystemDataId id = FwdbgSystemDataId;

    SystemData* pSystemData = nullptr;

    if (GetSystemData<SystemDataTag::Fwdbg>(&pSystemData, id).IsFailure())
    {
        *pOutCount = 0;
    }
    else
    {
        NN_SDK_ASSERT_NOT_NULL(pSystemData);

        NN_RESULT_DO(ReadAllBytes(pSystemData, pOutCount, outBuffer, count));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ReadSystemDataPlatformConfigration(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);

    auto found = false;

    ::nn::ncm::SystemDataId id = {};

    switch (GetSplHardwareType())
    {
    case SplHardwareType::None:
        break;
    case SplHardwareType::Icosa:
        found = true;
        id = PfCfgIcosaSystemDataId;
        break;
    case SplHardwareType::IcosaMariko:
        found = true;
        id = PfCfgIcosaMarikoSystemDataId;
        break;
    case SplHardwareType::Copper:
        found = true;
        id = PfCfgCopperSystemDataId;
        break;
    case SplHardwareType::Hoag:
        found = true;
        id = PfCfgHoagSystemDataId;
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    SystemData* pSystemData = nullptr;

    if (!found ||
        GetSystemData<SystemDataTag::PfCfg>(&pSystemData, id).IsFailure())
    {
        *pOutCount = 0;
    }
    else
    {
        NN_SDK_ASSERT_NOT_NULL(pSystemData);

        NN_RESULT_DO(ReadAllBytes(pSystemData, pOutCount, outBuffer, count));
    }

    NN_RESULT_SUCCESS;
}

} // namespace

}}} // namespace nn::settings::detail
