﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <utility>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_IFirmwareDebugSettingsServer.sfdl.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/settings/fwdbg/settings_SettingsManagement.h>
#include <nn/settings/system/settings_Web.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_Out.h>
#include <nn/TargetConfigs/build_Base.h>

#include "settings_FirmwareDebugSettingsDatabase.h"
#include "settings_FirmwareDebugSettingsServer.h"
#include "settings_LockableMutexType.h"
#include "settings_PrivateSettingsDatabase.h"
#include "settings_StaticObject.h"
#include "settings_SystemSettingsDatabase.h"

namespace nn { namespace settings { namespace detail {

namespace {

//!< 共有オブジェクト用ヒープメモリを扱うクラスです。
typedef ::nn::sf::ExpHeapStaticAllocator<16 * 1024, ISettingsItemKeyIterator>
         Allocator;

} // namespace

FirmwareDebugSettingsServer::FirmwareDebugSettingsServer() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result FirmwareDebugSettingsServer::SetSettingsItemValue(
    const SettingsName& name,
    const SettingsItemKey& key,
    const ::nn::sf::InBuffer& value) NN_NOEXCEPT
{
    NN_RESULT_DO(
        SetFwdbgSettingsItemValue(
            name, key, value.GetPointerUnsafe(), value.GetSize()));
    NN_RESULT_SUCCESS;
}

::nn::Result FirmwareDebugSettingsServer::ResetSettingsItemValue(
    const SettingsName& name,
    const SettingsItemKey& key) NN_NOEXCEPT
{
    NN_RESULT_DO(ResetFwdbgSettingsItemValue(name, key));
    NN_RESULT_SUCCESS;
}

::nn::Result FirmwareDebugSettingsServer::CreateSettingsItemKeyIterator(
    ::nn::sf::Out<::nn::sf::SharedPointer<ISettingsItemKeyIterator>> outValue,
    const SettingsName& name) NN_NOEXCEPT
{
    FwdbgSettingsItemKeyIterator iter;
    NN_RESULT_DO(iter.Initialize(name));

    NN_FUNCTION_LOCAL_STATIC(::std::atomic<bool>, s_IsInitialized, (false));

    if (!s_IsInitialized)
    {
        static LockableMutexType s_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

        ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

        if (!s_IsInitialized)
        {
            Allocator::Initialize(::nn::lmem::CreationOption_NoOption);

            s_IsInitialized = true;
        }
    }

    outValue.Set(
        ::nn::sf::ObjectFactory<Allocator::Policy>::CreateSharedEmplaced<
            ISettingsItemKeyIterator,
            SettingsItemKeyIterator<FwdbgSettingsItemKeyIterator>
            >(::std::move(iter)));

    NN_RESULT_SUCCESS;
}

::nn::Result FirmwareDebugSettingsServer::ReadSettings(
    ::nn::sf::Out<uint64_t> outCount, const ::nn::sf::OutBuffer& buffer,
    int32_t target) NN_NOEXCEPT
{
    const auto settingsTarget =
        static_cast<::nn::settings::fwdbg::SettingsTarget>(target);

    switch (settingsTarget)
    {
    case ::nn::settings::fwdbg::SettingsTarget_SystemSaveSystem:
        NN_RESULT_DO(
            ReadSystemSettings(
                outCount.GetPointer(),
                buffer.GetPointerUnsafe(), buffer.GetSize()));
        break;

    case ::nn::settings::fwdbg::SettingsTarget_SystemSavePrivate:
        NN_RESULT_DO(
            ReadPrivateSettings(
                outCount.GetPointer(),
                buffer.GetPointerUnsafe(), buffer.GetSize()));
        break;

    default:
        NN_RESULT_DO(
            ReadFwdbgSettings(
                outCount.GetPointer(),
                buffer.GetPointerUnsafe(), buffer.GetSize(), settingsTarget));
        break;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result FirmwareDebugSettingsServer::ResetSettings(
    int32_t target) NN_NOEXCEPT
{
    const auto settingsTarget =
        static_cast<::nn::settings::fwdbg::SettingsTarget>(target);

    switch (settingsTarget)
    {
    case ::nn::settings::fwdbg::SettingsTarget_SystemSaveSystem:
        NN_RESULT_DO(ResetSystemSettings());
        break;

    default:
        NN_RESULT_DO(ResetFwdbgSettings(settingsTarget));
        break;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result FirmwareDebugSettingsServer::SetWebInspectorFlag(
    bool value) NN_NOEXCEPT
{
    NN_RESULT_DO(SetSystemSettingsWebInspectorFlag(value));
    NN_RESULT_SUCCESS;
}

::nn::Result FirmwareDebugSettingsServer::SetAllowedSslHosts(
    const ::nn::sf::InArray<::nn::settings::system::AllowedSslHost>&
        values) NN_NOEXCEPT
{
    NN_RESULT_DO(
        SetSystemSettingsAllowedSslHosts(values.GetData(), values.GetLength()));
    NN_RESULT_SUCCESS;
}

::nn::Result FirmwareDebugSettingsServer::SetHostFsMountPoint(
    const ::nn::settings::system::HostFsMountPoint& value) NN_NOEXCEPT
{
    NN_RESULT_DO(SetSystemSettingsHostFsMountPoint(value));
    NN_RESULT_SUCCESS;
}

::nn::Result CreateFirmwareDebugSettingsServerProxy(
    ::nn::sf::SharedPointer<IFirmwareDebugSettingsServer>* pOutValue
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    *pOutValue = StaticObject<
        ::nn::sf::UnmanagedServiceObject<
            IFirmwareDebugSettingsServer, FirmwareDebugSettingsServer>
        >::Get().GetShared();
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::settings::detail
