﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <atomic>
#include <cstring>
#include <memory>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/settings/detail/settings_Log.h>
#include <nn/settings/factory/settings_Amiibo.h>
#include <nn/settings/factory/settings_BatteryLot.h>
#include <nn/settings/factory/settings_Bluetooth.h>
#include <nn/settings/factory/settings_ConfigurationId.h>
#include <nn/settings/factory/settings_DeviceCertificate.h>
#include <nn/settings/factory/settings_DeviceKey.h>
#include <nn/settings/factory/settings_GameCard.h>
#include <nn/settings/factory/settings_MotionSensor.h>
#include <nn/settings/factory/settings_Result.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include <nn/settings/factory/settings_Speaker.h>
#include <nn/settings/factory/settings_Ssl.h>
#include <nn/settings/factory/settings_WirelessLan.h>
#include <nn/settings/system/settings_BatteryLot.h>
#include <nn/settings/system/settings_ProductModel.h>
#include <nn/settings/system/settings_SerialNumber.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/util/util_MathTypes.h>
#include <nn/util/util_StringUtil.h>

#include "settings_CalibrationBinaryDatabase.h"
#include "settings_CalibrationInfo.h"
#include "settings_Crc16.h"
#include "settings_Platform.h"

//!< 生産時設定の警告ログを出力します。
#define NN_SETTINGS_FACTORY_WARN(...) \
    NN_DETAIL_SETTINGS_WARN("[calibration binary] Warning: " __VA_ARGS__)

namespace nn { namespace settings { namespace detail {

namespace {

//!< 固有鍵による秘密鍵の暗号化を開始したフォーマットバージョン
const uint32_t EncryptedKeyFormatVersion = 9;

//!< 秘密鍵の暗号化に使用する固有鍵世代のサイズ
const size_t EncryptionKeyGenerationSize = 4;

//!< 構成識別子 1 のデフォルト値
const char DefaultConfigurationId1[] = "";

//!< 製品シリアル番号のデフォルト値
const ::nn::settings::factory::SerialNumber DefaultSerialNumber = {};

//!< 電池 LOT のデフォルト値
const ::nn::settings::factory::BatteryLot DefaultBatteryLot = {};

//!< 製品モデルのデフォルト値
const ::nn::settings::system::ProductModel DefaultProductModel =
    ::nn::settings::system::ProductModel_Invalid;

//!< ファイルシステムに関する Result 値を settings で定義するものに変換します。
::nn::Result ConvertFileSystemResult(::nn::Result result) NN_NOEXCEPT;

//!< 生産時較正情報の amiibo 用本体秘密鍵を取得します。
::nn::Result GetCalibInfoAmiiboKey(
    ::nn::settings::factory::AmiiboKey* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の amiibo 用 ECQV 証明書を取得します。
::nn::Result GetCalibInfoAmiiboEcqvCertificate(
    ::nn::settings::factory::AmiiboEcqvCertificate* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の amiibo 用 ECDSA 証明書を取得します。
::nn::Result GetCalibInfoAmiiboEcdsaCertificate(
    ::nn::settings::factory::AmiiboEcdsaCertificate* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の amiibo 用本体秘密鍵 (ECQV-BLS 版) を取得します。
::nn::Result GetCalibInfoAmiiboEcqvBlsKey(
    ::nn::settings::factory::AmiiboEcqvBlsKey* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の amiibo 用 ECQV-BLS 証明書を取得します。
::nn::Result GetCalibInfoAmiiboEcqvBlsCertificate(
    ::nn::settings::factory::AmiiboEcqvBlsCertificate* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の amiibo 用ルート証明書 (ECQV-BLS 版) を取得します。
::nn::Result GetCalibInfoAmiiboEcqvBlsRootCertificate(
    ::nn::settings::factory::AmiiboEcqvBlsRootCertificate* pOutValue
    ) NN_NOEXCEPT;

//!< 生産時較正情報の Bluetooth に対して設定された BD アドレスを取得します。
::nn::Result GetCalibInfoBluetoothBdAddress(
    ::nn::settings::factory::BdAddress* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の構成識別子 1 を取得します。
::nn::Result GetCalibInfoConfigurationId1(
    ::nn::settings::factory::ConfigurationId1* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の加速度センサのオフセット値を取得します。
::nn::Result GetCalibInfoAccelerometerOffset(
    ::nn::settings::factory::AccelerometerOffset* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の加速度センサのスケール値を取得します。
::nn::Result GetCalibInfoAccelerometerScale(
    ::nn::settings::factory::AccelerometerScale* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報のジャイロスコープのオフセット値を取得します。
::nn::Result GetCalibInfoGyroscopeOffset(
    ::nn::settings::factory::GyroscopeOffset* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報のジャイロスコープのスケール値を取得します。
::nn::Result GetCalibInfoGyroscopeScale(
    ::nn::settings::factory::GyroscopeScale* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の製品シリアル番号を取得します。
::nn::Result GetCalibInfoSerialNumber(
    ::nn::settings::factory::SerialNumber* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の無線 LAN に対して設定された MAC アドレスを取得します。
::nn::Result GetCalibInfoWirelessLanMacAddress(
    ::nn::settings::factory::MacAddress* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の無線 LAN に対して設定された国名コードの数を取得します。
::nn::Result GetCalibInfoWirelessLanCountryCodeCount(
    int32_t* pOutCount) NN_NOEXCEPT;

//!< 生産時較正情報の無線 LAN に対して設定された国名コードを取得します。
::nn::Result GetCalibInfoWirelessLanCountryCodes(
    int32_t* pOutCount,
    ::nn::settings::factory::CountryCode outCodes[],
    size_t count) NN_NOEXCEPT;

//!< 生産時較正情報のゲームカード相互認証秘密鍵を取得します。
::nn::Result GetCalibInfoGameCardKey(
    ::nn::settings::factory::GameCardKey* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報のゲームカード相互認証証明書を取得します。
::nn::Result GetCalibInfoGameCardCertificate(
    ::nn::settings::factory::GameCardCertificate* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の電池 LOT を取得します
::nn::Result GetCalibInfoBatteryLot(
    ::nn::settings::factory::BatteryLot* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報のスピーカーのパラメータを取得します。
::nn::Result GetCalibInfoSpeakerParameter(
    ::nn::settings::factory::SpeakerParameter* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の ECC-B233 版のデバイス登録用デバイス証明書を取得します。
::nn::Result GetCalibInfoEccB233DeviceCertificate(
    ::nn::settings::factory::EccB233DeviceCertificate* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の RSA-2048 版のデバイス登録用デバイス証明書を取得します。
::nn::Result GetCalibInfoRsa2048DeviceCertificate(
    ::nn::settings::factory::Rsa2048DeviceCertificate* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の RSA-2048 版の eTicket 発行用デバイス証明書を取得します。
::nn::Result GetCalibInfoRsa2048ETicketCertificate(
    ::nn::settings::factory::Rsa2048DeviceCertificate* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の製品モデルを取得します
::nn::Result GetCalibInfoProductModel(int32_t* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の SSL クライアント秘密鍵を取得します。
::nn::Result GetCalibInfoSslKey(
    ::nn::settings::factory::SslKey* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の SSL クライアント証明書を取得します。
::nn::Result GetCalibInfoSslCertificate(
    ::nn::settings::factory::SslCertificate* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の輝度レベルからバックライト出力への変換係数を取得します。
::nn::Result GetCalibInfoBacklightBrightnessCoefficients(
    ::nn::util::Float3* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の ECC-B233 版のデバイス登録用デバイス秘密鍵を取得します。
::nn::Result GetCalibInfoEccB233DeviceKey(
    ::nn::settings::factory::EccB233DeviceKey* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の RSA-2048 版のデバイス登録用デバイス秘密鍵を取得します。
::nn::Result GetCalibInfoRsa2048DeviceKey(
    ::nn::settings::factory::Rsa2048DeviceKey* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の RSA-2048 版の eTicket 発行用デバイス秘密鍵を取得します。
::nn::Result GetCalibInfoRsa2048ETicketKey(
    ::nn::settings::factory::Rsa2048DeviceKey* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の LCD ベンダ ID を取得します。
::nn::Result GetCalibInfoLcdVendorId(uint32_t* pOutValue) NN_NOEXCEPT;

//!< 生産時較正情報の USB Type-C Power Source 回路バージョンを取得します。
::nn::Result GetCalibInfoUsbTypeCPowerSourceCircuitVersion(
    ::nn::Bit8* pOutValue) NN_NOEXCEPT;

} // namespace

::nn::Result GetCalibBinAmiiboKey(
    ::nn::settings::factory::AmiiboKey* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullAmiiboKeyBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoAmiiboKey(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinAmiiboEcqvCertificate(
    ::nn::settings::factory::AmiiboEcqvCertificate* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullAmiiboEcqvCertificateBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoAmiiboEcqvCertificate(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinAmiiboEcdsaCertificate(
    ::nn::settings::factory::AmiiboEcdsaCertificate* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullAmiiboEcdsaCertificateBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoAmiiboEcdsaCertificate(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinAmiiboEcqvBlsKey(
    ::nn::settings::factory::AmiiboEcqvBlsKey* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullAmiiboEcqvBlsKeyBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoAmiiboEcqvBlsKey(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinAmiiboEcqvBlsCertificate(
    ::nn::settings::factory::AmiiboEcqvBlsCertificate* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullAmiiboEcqvBlsCertificateBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(
            GetCalibInfoAmiiboEcqvBlsCertificate(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinAmiiboEcqvBlsRootCertificate(
    ::nn::settings::factory::AmiiboEcqvBlsRootCertificate* pOutValue
    ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullAmiiboEcqvBlsRootCertificateBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(
            GetCalibInfoAmiiboEcqvBlsRootCertificate(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinBluetoothBdAddress(
    ::nn::settings::factory::BdAddress* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoBluetoothBdAddress(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinConfigurationId1(
    ::nn::settings::factory::ConfigurationId1* pOutValue) NN_NOEXCEPT
{
    const ::nn::Result result = GetCalibInfoConfigurationId1(pOutValue);

#ifdef NN_BUILD_CONFIG_OS_HORIZON
    if (result.IsFailure())
#else
    if (NN_STATIC_CONDITION(false))
#endif
    {
        NN_SETTINGS_FACTORY_WARN(
            "GetConfigurationId1() failed. (%08x, %d%03d-%04d)\n",
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());
    }

    if (result.IsFailure())
    {
        ::std::strcpy(pOutValue->string, DefaultConfigurationId1);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinAccelerometerOffset(
    ::nn::settings::factory::AccelerometerOffset* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoAccelerometerOffset(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinAccelerometerScale(
    ::nn::settings::factory::AccelerometerScale* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoAccelerometerScale(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinGyroscopeOffset(
    ::nn::settings::factory::GyroscopeOffset* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoGyroscopeOffset(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinGyroscopeScale(
    ::nn::settings::factory::GyroscopeScale* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoGyroscopeScale(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinSerialNumber(
    ::nn::settings::factory::SerialNumber* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoSerialNumber(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinSerialNumberForSystemSettings(
    ::nn::settings::system::SerialNumber* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultNullSerialNumberBuffer());

    ::nn::settings::factory::SerialNumber serialNumber = {};

    const ::nn::Result result = GetCalibInfoSerialNumber(&serialNumber);

    if (result.IsFailure())
    {
        NN_SETTINGS_FACTORY_WARN(
            "GetSerialNumber() failed. (%08x, %d%03d-%04d)\n",
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());

        serialNumber = DefaultSerialNumber;
    }

    ::nn::util::Strlcpy(pOutValue->string,
                        serialNumber.string,
                        sizeof(pOutValue->string));

    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinWirelessLanMacAddress(
    ::nn::settings::factory::MacAddress* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoWirelessLanMacAddress(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinWirelessLanCountryCodeCount(
    int32_t* pOutCount) NN_NOEXCEPT
{
    NN_RESULT_DO(
        ConvertFileSystemResult(
            GetCalibInfoWirelessLanCountryCodeCount(pOutCount)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinWirelessLanCountryCodes(
    int32_t* pOutCount,
    ::nn::settings::factory::CountryCode outCodes[],
    size_t count) NN_NOEXCEPT
{
    NN_RESULT_DO(
        ConvertFileSystemResult(
            GetCalibInfoWirelessLanCountryCodes(pOutCount, outCodes, count)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinBatteryLot(
    ::nn::settings::factory::BatteryLot* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_DO(ConvertFileSystemResult(GetCalibInfoBatteryLot(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinBatteryLotForSystemSettings(
    ::nn::settings::system::BatteryLot* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullBatteryLotBuffer());

    ::nn::settings::factory::BatteryLot batteryLot = {};

    const ::nn::Result result = GetCalibInfoBatteryLot(&batteryLot);

    if (result.IsFailure())
    {
        NN_SETTINGS_FACTORY_WARN(
            "GetBatteryLot() failed. (%08x, %d%03d-%04d)\n",
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());

        batteryLot = DefaultBatteryLot;
    }

    ::nn::util::Strlcpy(pOutValue->string,
                        batteryLot.string,
                        sizeof(pOutValue->string));

    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinSpeakerParameter(
    ::nn::settings::factory::SpeakerParameter* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultNullSpeakerParameterBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoSpeakerParameter(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinEccB233DeviceCertificate(
    ::nn::settings::factory::EccB233DeviceCertificate* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullEciDeviceCertificateBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(
            GetCalibInfoEccB233DeviceCertificate(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinRsa2048DeviceCertificate(
    ::nn::settings::factory::Rsa2048DeviceCertificate* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullEciDeviceCertificateBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(
            GetCalibInfoRsa2048DeviceCertificate(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinRsa2048ETicketCertificate(
    ::nn::settings::factory::Rsa2048DeviceCertificate* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultNullEticketDeviceCertificateBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(
            GetCalibInfoRsa2048ETicketCertificate(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinProductModelForSystemSettings(
    int32_t* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultNullProductModelBuffer());

    const ::nn::Result result = GetCalibInfoProductModel(pOutValue);

    if (result.IsFailure())
    {
        NN_SETTINGS_FACTORY_WARN(
            "GetProductModel() failed. (%08x, %d%03d-%04d)\n",
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());

        *pOutValue = DefaultProductModel;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinSslKey(
    ::nn::settings::factory::SslKey* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultNullSslKeyBuffer());
    NN_RESULT_DO(ConvertFileSystemResult(GetCalibInfoSslKey(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinSslCertificate(
    ::nn::settings::factory::SslCertificate* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultNullSslCertificateBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoSslCertificate(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinGameCardKey(
    ::nn::settings::factory::GameCardKey* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultNullGameCardKeyBuffer());
    NN_RESULT_DO(ConvertFileSystemResult(GetCalibInfoGameCardKey(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinGameCardCertificate(
    ::nn::settings::factory::GameCardCertificate* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultNullGameCardCertificateBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoGameCardCertificate(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinBacklightBrightnessCoefficientsForInternal(
    ::nn::util::Float3* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    const ::nn::Result result =
        GetCalibInfoBacklightBrightnessCoefficients(pOutValue);

#ifdef NN_BUILD_CONFIG_OS_HORIZON
    if (result.IsFailure())
#else
    if (NN_STATIC_CONDITION(false))
#endif
    {
        NN_SETTINGS_FACTORY_WARN(
            "Failed to read brightnessCoefficients. (%08x, %d%03d-%04d)\n",
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());
    }

    NN_RESULT_DO(ConvertFileSystemResult(result));

    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinEccB233DeviceKey(
    ::nn::settings::factory::EccB233DeviceKey* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullEciDeviceKeyBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoEccB233DeviceKey(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinRsa2048DeviceKey(
    ::nn::settings::factory::Rsa2048DeviceKey* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullEciDeviceKeyBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoRsa2048DeviceKey(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinRsa2048ETicketKey(
    ::nn::settings::factory::Rsa2048DeviceKey* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultNullEticketDeviceKeyBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(GetCalibInfoRsa2048ETicketKey(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinLcdVendorId(uint32_t* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullLcdVendorIdBuffer());
    NN_RESULT_DO(ConvertFileSystemResult(GetCalibInfoLcdVendorId(pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibBinUsbTypeCPowerSourceCircuitVersion(
    ::nn::Bit8* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr,
        ResultNullUsbTypeCPowerSourceCircuitVersionBuffer());
    NN_RESULT_DO(
        ConvertFileSystemResult(
            GetCalibInfoUsbTypeCPowerSourceCircuitVersion(pOutValue)));
    NN_RESULT_SUCCESS;
}

namespace {

#ifdef NN_BUILD_CONFIG_OS_HORIZON
//!< 生産時較正情報を読み出します。
::nn::Result GetCalibrationInfo(CalibrationInfo** ppOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutValue);

    static CalibrationInfo s_CalibrationInfo = {};

    NN_FUNCTION_LOCAL_STATIC(::std::atomic<bool>, s_IsInitialized, (false));

    if (!s_IsInitialized)
    {
        static ::nn::os::SdkMutexType s_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

        ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

        if (!s_IsInitialized)
        {
            ::std::unique_ptr<::nn::fs::IStorage> storage;
            NN_RESULT_DO(
                ::nn::fs::OpenBisPartition(
                    &storage, ::nn::fs::BisPartitionId::CalibrationBinary));
            NN_RESULT_DO(
                storage.get()->Read(
                    0, &s_CalibrationInfo, sizeof(s_CalibrationInfo)));
            s_IsInitialized = true;
        }
    }

    *ppOutValue = &s_CalibrationInfo;

    NN_RESULT_SUCCESS;
}
#else
//!< 生産時較正情報を読み出します。
::nn::Result GetCalibrationInfo(CalibrationInfo** ppOutValue) NN_NOEXCEPT
{
    NN_UNUSED(ppOutValue);
    return ::nn::fs::ResultPartitionNotFound();
}
#endif

::nn::Result ConvertFileSystemResult(::nn::Result result) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        !::nn::fs::ResultDataCorrupted::Includes(result),
        ::nn::settings::factory::ResultCalibrationDataFileSystemCorrupted());
    NN_RESULT_THROW_UNLESS(
        !::nn::fs::ResultPartitionNotFound::Includes(result),
        ::nn::settings::factory::ResultCalibrationDataFileSystemCorrupted());
    return result;
}

::nn::Result GetCalibInfoAmiiboKey(
    ::nn::settings::factory::AmiiboKey* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.ecqvEcdsaAmiiboKeyBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    pOutValue->size = static_cast<uint32_t>(
        sizeof(block.data) - (
            pInfo->header.version < EncryptedKeyFormatVersion ?
                EncryptionKeyGenerationSize : 0));
    ::std::copy(block.data, block.data + pOutValue->size, pOutValue->data);
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoAmiiboEcqvCertificate(
    ::nn::settings::factory::AmiiboEcqvCertificate* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.ecqvEcdsaAmiiboCertificateBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    pOutValue->size = static_cast<uint32_t>(sizeof(block.data));
    ::std::copy(block.data, block.data + pOutValue->size, pOutValue->data);
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoAmiiboEcdsaCertificate(
    ::nn::settings::factory::AmiiboEcdsaCertificate* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.ecqvEcdsaAmiiboRootCertificateBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    pOutValue->size = static_cast<uint32_t>(sizeof(block.data));
    ::std::copy(block.data, block.data + pOutValue->size, pOutValue->data);
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoAmiiboEcqvBlsKey(
    ::nn::settings::factory::AmiiboEcqvBlsKey* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.ecqvBlsAmiiboKeyBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    pOutValue->size = static_cast<uint32_t>(
        sizeof(block.data) - (
            pInfo->header.version < EncryptedKeyFormatVersion ?
                EncryptionKeyGenerationSize : 0));
    ::std::copy(block.data, block.data + pOutValue->size, pOutValue->data);
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoAmiiboEcqvBlsCertificate(
    ::nn::settings::factory::AmiiboEcqvBlsCertificate* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.ecqvBlsAmiiboCertificateBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    pOutValue->size = static_cast<uint32_t>(sizeof(block.data));
    ::std::copy(block.data, block.data + pOutValue->size, pOutValue->data);
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoAmiiboEcqvBlsRootCertificate(
    ::nn::settings::factory::AmiiboEcqvBlsRootCertificate* pOutValue
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.ecqvBlsAmiiboRootCertificateBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    pOutValue->size = static_cast<uint32_t>(sizeof(block.data));
    ::std::copy(block.data, block.data + pOutValue->size, pOutValue->data);
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoBluetoothBdAddress(
    ::nn::settings::factory::BdAddress* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.bluetoothBdAddressBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block.bdAddress, sizeof(block.bdAddress)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.bdAddress;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoConfigurationId1(
    ::nn::settings::factory::ConfigurationId1* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.configurationId1Block;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block.configurationId1,
                 sizeof(block.configurationId1)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.configurationId1;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoAccelerometerOffset(
    ::nn::settings::factory::AccelerometerOffset* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.accelerometerOffsetBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block.accelerometerOffset,
                 sizeof(block.accelerometerOffset)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.accelerometerOffset;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoAccelerometerScale(
    ::nn::settings::factory::AccelerometerScale* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.accelerometerScaleBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block.accelerometerScale,
                 sizeof(block.accelerometerScale)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.accelerometerScale;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoGyroscopeOffset(
    ::nn::settings::factory::GyroscopeOffset* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.gyroscopeOffsetBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block.gyroscopeOffset,
                 sizeof(block.gyroscopeOffset)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.gyroscopeOffset;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoGyroscopeScale(
    ::nn::settings::factory::GyroscopeScale* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.gyroscopeScaleBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block.gyroscopeScale,
                 sizeof(block.gyroscopeScale)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.gyroscopeScale;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoSerialNumber(
    ::nn::settings::factory::SerialNumber* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.serialNumberBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.serialNumber;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoWirelessLanMacAddress(
    ::nn::settings::factory::MacAddress* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.wirelessLanMacAddressBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block.macAddress, sizeof(block.macAddress)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.macAddress;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoWirelessLanCountryCodeCount(int32_t* pOutCount
                                                     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.wirelessLanCountryCodesBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutCount = block.count;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoWirelessLanCountryCodes(
    int32_t* pOutCount,
    ::nn::settings::factory::CountryCode outCodes[],
    size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outCodes);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.wirelessLanCountryCodesBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutCount = static_cast<int32_t>(
        ::std::min<size_t>(count, block.count));
    ::std::copy(block.countryCodes, block.countryCodes + *pOutCount, outCodes);
    NN_RESULT_SUCCESS;
}

//!< 生産時較正情報のゲームカード相互認証秘密鍵を取得します。
template<typename T>
::nn::Result GetCalibInfoGameCardKey(
    ::nn::settings::factory::GameCardKey* pOutValue, const T& block) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    const auto size = static_cast<uint32_t>(sizeof(block.gameCardKey));
    pOutValue->size = size;
    ::std::copy(block.gameCardKey, block.gameCardKey + size, pOutValue->data);
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoGameCardKey(
    ::nn::settings::factory::GameCardKey* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    const CalibrationInfoBody& body = pInfo->body;
    if (GetCalibInfoGameCardKey(pOutValue,
                                body.extendedGameCardKeyBlock).IsSuccess())
    {
        const CalibrationInfoHeader& header = pInfo->header;
        if (header.version < EncryptedKeyFormatVersion)
        {
            const auto size = EncryptionKeyGenerationSize;
            const auto offset = sizeof(pOutValue->data) - size;
            ::std::memset(&pOutValue->data[offset], 0, size);
        }
        NN_RESULT_SUCCESS;
    }
    else
    {
        NN_RESULT_DO(GetCalibInfoGameCardKey(pOutValue, body.gameCardKeyBlock));
        NN_RESULT_SUCCESS;
    }
}

::nn::Result GetCalibInfoGameCardCertificate(
    ::nn::settings::factory::GameCardCertificate* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    const auto& block = pInfo->body.gameCardCertificateBlock;
    const auto size = sizeof(block.gameCardCertificate);
    Sha256Hash hash = {};
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(&block.gameCardCertificate, size);
    generator.GetHash(hash.data, sizeof(hash.data));
    NN_RESULT_THROW_UNLESS(
        ::std::memcmp(block.gameCardCertificateHash.data,
                      hash.data, sizeof(hash.data)) == 0,
        ::nn::settings::factory::ResultCalibrationDataShaError());
    ::std::memcpy(pOutValue, &block.gameCardCertificate, size);
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoBatteryLot(
    ::nn::settings::factory::BatteryLot* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.batteryLotBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.batteryLot;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoSpeakerParameter(
    ::nn::settings::factory::SpeakerParameter* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.codecIcParameterBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.speakerParameter;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoEccB233DeviceCertificate(
    ::nn::settings::factory::EccB233DeviceCertificate* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.eccB233DeviceCertificateBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.deviceCertificate;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoRsa2048DeviceCertificate(
    ::nn::settings::factory::Rsa2048DeviceCertificate* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.rsa2048DeviceCertificateBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.deviceCertificate;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoRsa2048ETicketCertificate(
    ::nn::settings::factory::Rsa2048DeviceCertificate* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.rsa2048ETicketCertificateBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.deviceCertificate;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoProductModel(int32_t* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.productModelBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.productModel;
    NN_RESULT_SUCCESS;
}

//!< 生産時較正情報の SSL クライアント秘密鍵を取得します。
template<typename T>
::nn::Result GetCalibInfoSslKey(::nn::settings::factory::SslKey* pOutValue,
                                const T& block) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    const auto size = static_cast<uint32_t>(sizeof(block.sslKey));
    pOutValue->size = size;
    ::std::copy(block.sslKey, block.sslKey + size, pOutValue->data);
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoSslKey(
    ::nn::settings::factory::SslKey* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    const CalibrationInfoBody& body = pInfo->body;
    if (GetCalibInfoSslKey(pOutValue, body.extendedSslKeyBlock).IsSuccess())
    {
        const CalibrationInfoHeader& header = pInfo->header;
        if (header.version < EncryptedKeyFormatVersion)
        {
            const auto size = EncryptionKeyGenerationSize;
            const auto offset = sizeof(pOutValue->data) - size;
            ::std::memset(&pOutValue->data[offset], 0, size);
        }
        NN_RESULT_SUCCESS;
    }
    else
    {
        NN_RESULT_DO(GetCalibInfoSslKey(pOutValue, body.sslKeyBlock));
        NN_RESULT_SUCCESS;
    }
}

::nn::Result GetCalibInfoSslCertificate(
    ::nn::settings::factory::SslCertificate* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.sslCertificateSizeBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    auto& dataBlock = pInfo->body.sslCertificateBlock;
    const auto sizeMax = sizeof(dataBlock.sslCertificate);
    const auto size = ::std::min(static_cast<size_t>(block.sslCertificateSize),
                                 sizeMax);
    Sha256Hash hash = {};
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(&dataBlock.sslCertificate, size);
    generator.GetHash(hash.data, sizeof(hash.data));
    NN_RESULT_THROW_UNLESS(
        ::std::memcmp(dataBlock.sslCertificateHash.data,
                      hash.data, sizeof(hash.data)) == 0,
        ::nn::settings::factory::ResultCalibrationDataShaError());
    pOutValue->size = static_cast<uint32_t>(size);
    ::std::copy(dataBlock.sslCertificate, dataBlock.sslCertificate + size,
                pOutValue->data);
    ::std::memset(&pOutValue->data[size], 0, sizeMax - size);
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoBacklightBrightnessCoefficients(
    ::nn::util::Float3* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.backlightBrightnessCoefficientsBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.coefficients;
    NN_RESULT_SUCCESS;
}

//!< 生産時較正情報の ECC-B233 版のデバイス登録用デバイス秘密鍵を取得します。
template<typename T>
::nn::Result GetCalibInfoEccB233DeviceKey(
    ::nn::settings::factory::EccB233DeviceKey* pOutValue,
    const T& block) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    const auto size = static_cast<uint32_t>(sizeof(block.deviceKey));
    pOutValue->size = size;
    ::std::copy(block.deviceKey, block.deviceKey + size, pOutValue->data);
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoEccB233DeviceKey(
    ::nn::settings::factory::EccB233DeviceKey* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    const CalibrationInfoBody& body = pInfo->body;
    if (GetCalibInfoEccB233DeviceKey(
            pOutValue, body.extendedEccB233DeviceKeyBlock).IsSuccess())
    {
        const CalibrationInfoHeader& header = pInfo->header;
        if (header.version < EncryptedKeyFormatVersion)
        {
            const auto size = EncryptionKeyGenerationSize;
            const auto offset = sizeof(pOutValue->data) - size;
            ::std::memset(&pOutValue->data[offset], 0, size);
        }
        NN_RESULT_SUCCESS;
    }
    else
    {
        NN_RESULT_DO(GetCalibInfoEccB233DeviceKey(
                         pOutValue, body.eccB233DeviceKeyBlock));
        NN_RESULT_SUCCESS;
    }
}

::nn::Result GetCalibInfoRsa2048DeviceKey(
    ::nn::settings::factory::Rsa2048DeviceKey* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.rsa2048DeviceKeyBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    const auto size = static_cast<uint32_t>(sizeof(block.deviceKey));
    pOutValue->size = size;
    ::std::copy(block.deviceKey, block.deviceKey + size, pOutValue->data);
    NN_RESULT_SUCCESS;
}

//!< RSA-2048 版の eTicket 発行用デバイス秘密鍵を取得します。
template<typename T>
::nn::Result GetCalibInfoRsa2048ETicketKey(
    ::nn::settings::factory::Rsa2048DeviceKey* pOutValue,
    const T& block) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    const auto size = static_cast<uint32_t>(sizeof(block.deviceKey));
    pOutValue->size = size;
    ::std::copy(block.deviceKey, block.deviceKey + size, pOutValue->data);
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoRsa2048ETicketKey(
    ::nn::settings::factory::Rsa2048DeviceKey* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    const CalibrationInfoBody& body = pInfo->body;
    if (GetCalibInfoRsa2048ETicketKey(
            pOutValue, body.extendedRsa2048ETicketKeyBlock).IsSuccess())
    {
        const CalibrationInfoHeader& header = pInfo->header;
        if (header.version < EncryptedKeyFormatVersion)
        {
            const auto size = EncryptionKeyGenerationSize;
            const auto offset = sizeof(pOutValue->data) - size;
            ::std::memset(&pOutValue->data[offset], 0, size);
        }
        NN_RESULT_SUCCESS;
    }
    else
    {
        NN_RESULT_DO(GetCalibInfoRsa2048ETicketKey(
                         pOutValue, body.rsa2048ETicketKeyBlock));
        NN_RESULT_SUCCESS;
    }
}

::nn::Result GetCalibInfoLcdVendorId(uint32_t* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.lcdVendorIdBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.lcdVendorId;
    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibInfoUsbTypeCPowerSourceCircuitVersion(
    ::nn::Bit8* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    CalibrationInfo* pInfo = nullptr;
    NN_RESULT_DO(GetCalibrationInfo(&pInfo));
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    auto& block = pInfo->body.usbTypeCPowerSourceCircuitBlock;
    NN_RESULT_THROW_UNLESS(
        GetCrc16(&block, sizeof(block) - sizeof(block.crc16)) == block.crc16,
        ::nn::settings::factory::ResultCalibrationDataCrcError());
    *pOutValue = block.version;
    NN_RESULT_SUCCESS;
}

} // namespace

}}} // nn::settings::detail
