﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   SASBUS ドライバの API インタフェース部分。
 */

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/nn_SdkLog.h>

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include <nne/spi/spi_results.h>
#include <nne/spi/spi_api.h>
#include <nn/sasbus/sasbus_PeriodicReceiveModeType.h>
#include <nn/sasbus/detail/sasbus_RingLifo.h>

#include <nn/sasbus/detail/sasbus_Log.h>
#include <nn/sasbus/driver/sasbus.h>
#include "detail/sasbus_SessionImpl-soc.tegra-x1.h"
#include "detail/sasbus_TargetSpec-soc.tegra-x1.h"

namespace
{
bool g_IsSuspend = false;

// TORIAEZU : メンバに持つとセッションサイズがでかくなるため、ここで定義
// PeriodicReceiveMode 用スレッド
// TORIAEZU : 今は 1つだけ
nn::os::ThreadType                            g_PeriodicReceiveModeThread;
NN_ALIGNAS(nn::os::ThreadStackAlignment) char g_PeriodicReceiveModeThreadStack[nn::os::ThreadStackAlignment];
bool                                          g_IsStartPeriodicReceiveModeThread = false;
bool                                          g_IsStartPeriodicReceiveModeThreadBeforeSleeping = false;

// Session のポインタを保存しておく配列
nn::sasbus::driver::Session* pSessionList[nn::sasbus::driver::detail::DeviceNum];

// PeriodicReceiveMode を行う実態
template <typename PeriodicReceiveModeDataT, int LifoEntryCount>
static void PeriodicReceiveModeThread(void* arg)  NN_NOEXCEPT
{
    PeriodicReceiveModeDataT tempBuffer;
    nn::os::TimerEvent   timerEvent(nn::os::EventClearMode_AutoClear);

    auto pSession    = reinterpret_cast<nn::sasbus::driver::Session*>(arg);
    auto sessionImpl = nn::sasbus::driver:: detail::ToSessionImpl(*pSession);

    timerEvent.StartPeriodic(0, sessionImpl.interval);

    while (1)
    {
        timerEvent.Wait();

        if (g_IsStartPeriodicReceiveModeThread)
        {
            tempBuffer.result = nn::sasbus::driver::Receive(tempBuffer.data,
                                                              pSession,
                                                              sessionImpl.dataSize,
                                                              sessionImpl.reg);

            tempBuffer.tick           = nn::os::GetSystemTick();
            tempBuffer.samplingNumber = sessionImpl.sampleNumber;
            sessionImpl.sampleNumber++;

            static_cast<nn::sasbus::detail::RingLifo<PeriodicReceiveModeDataT, LifoEntryCount>*>(sessionImpl.pRingLifo)->Append(tempBuffer);
        }
        else
        {
            break;
        }
    }
}

inline nn::Result ConvertNneResultToNnResult(nne::spi::Result nneResult)
{
    switch (nneResult)
    {
    case nne::spi::Result_Success:
        return nn::ResultSuccess();
    case nne::spi::Result_Invalid_Handle:
    case nne::spi::Result_Invalid_Name:
    case nne::spi::Result_Invalid_Request:
    case nne::spi::Result_Invalid_Arguments:
    case nne::spi::Result_Invalid_Configuration:
    case nne::spi::Result_Not_Suspended:
    case nne::spi::Result_Already_Suspended:
    case nne::spi::Result_Rx_Underflow:
    case nne::spi::Result_Rx_Overflow:
    case nne::spi::Result_Tx_Underflow:
    case nne::spi::Result_Tx_Overflow:
        // ここに該当するエラーはライブラリ側の実装ミスなので ABORT する。
        NN_ABORT("return unhandled nne::spi error (%d)\n", nneResult);
    case nne::spi::Result_Spurious_Interrupt:
        NN_DETAIL_SASBUS_ERROR("ResultSpriousInterruptOccured\n");
        return nn::sasbus::ResultSpuriousInterruptOccurred();
    case nne::spi::Result_No_Interrupt:
    case nne::spi::Result_RxTx_Timeout:
    case nne::spi::Result_FIFO_Timeout:
        NN_DETAIL_SASBUS_ERROR("ResultTimeout\n");
        return nn::sasbus::ResultTimeout();
    default:NN_UNEXPECTED_DEFAULT;
    }
}

}

namespace nn {
namespace sasbus {
namespace driver {

void Initialize() NN_NOEXCEPT
{
    nne::spi::Initialize();
}

void Finalize() NN_NOEXCEPT
{
    nne::spi::Finalize();
}

void OpenSession(Session* pOutSession, SasbusDevice device) NN_NOEXCEPT
{
    detail::SessionImpl& session = detail::ToSessionImpl(*pOutSession);

    int deviceIndex = detail::GetDeviceSettingIndex(device);

    auto result = nne::spi::OpenSession(&session.session,
                                        detail::DeviceSettings[deviceIndex].Name,
                                        detail::DeviceSettings[deviceIndex].Mode,
                                        detail::DeviceSettings[deviceIndex].Bits,
                                        detail::DeviceSettings[deviceIndex].Clock);

    // ここで nne から返される result はどれも実装ミスなので ABORT
    NN_ABORT_UNLESS_RESULT_SUCCESS(ConvertNneResultToNnResult(result));

    nn::os::InitializeMutex(&session.transactionMutex.mutex, false, 0);

    session.device = device;

    // TORIAEZU : 今は1つなので、決めうち。
    // 増えたら、 SearchSessionList 的なものが必要
    pSessionList[0] = pOutSession;
}

void CloseSession(Session* pSession) NN_NOEXCEPT
{
    detail::SessionImpl session = detail::ToSessionImpl(*pSession);

    StopPeriodicReceiveMode(pSession);

    auto result = nne::spi::CloseSession(&session.session);
    // ここで nne から返される result はどれも実装ミスなので ABORT
    NN_ABORT_UNLESS_RESULT_SUCCESS(ConvertNneResultToNnResult(result));

    nn::os::FinalizeMutex(&session.transactionMutex.mutex);
}

nn::Result Send(Session* pSession, const void* pInData, size_t dataBytes, const char reg) NN_NOEXCEPT
{
    detail::SessionImpl session = detail::ToSessionImpl(*pSession);

    std::lock_guard<detail::StaticMutex> lock(session.transactionMutex);

    auto result = nne::spi::Send(&session.session, reg, pInData, dataBytes);
    return ConvertNneResultToNnResult(result);
}

nn::Result Receive(void* pOutData, Session* pSession, size_t dataBytes, const char reg) NN_NOEXCEPT
{
    detail::SessionImpl session = detail::ToSessionImpl(*pSession);

    std::lock_guard<detail::StaticMutex> lock(session.transactionMutex);

    auto result = nne::spi::Receive(pOutData, reg, dataBytes, &session.session);
    return ConvertNneResultToNnResult(result);
}

Result StartPeriodicReceiveMode(Session* pSession, size_t dataSize, char reg, nn::TimeSpan interval, char* receiveBuffer, size_t receiveBufferLength) NN_NOEXCEPT
{
    // すでに PeriodicReceiveMode を行っているときは result を返す
    if (g_IsStartPeriodicReceiveModeThread)
    {
        return nn::sasbus::ResultAlreadyStartPeriodicReceiveMode();
    }

    detail::SessionImpl& session = detail::ToSessionImpl(*pSession);

    session.reg      = reg;
    session.interval = interval;
    session.dataSize = dataSize;
    pSession->_receiveBuffer = receiveBuffer;
    pSession->_receiveBufferSize = receiveBufferLength;

    // 対応デバイスごとに分岐
    // TORIAEZU : 今は lsm6ds3 のみ
    if (session.device == nn::sasbus::SasbusDevice_Lsm6ds3)
    {
        // receiveBuffer 上に placement new で RingLifo を作成する
        NN_SDK_ASSERT(sizeof(nn::sasbus::detail::RingLifo<nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataCountMax>) <= receiveBufferLength,
                      "receiveBufferLength is short. RingLifo = %d, receiveBufferLength = %d\n\n",
                      sizeof(nn::sasbus::detail::RingLifo<nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataCountMax>), receiveBufferLength);
        session.pRingLifo = new (receiveBuffer) nn::sasbus::detail::RingLifo<nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataCountMax>;

        // Interval Receiveスレッドを作成
        // 立ち上げるスレッドはモジュールごとに指定
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread(&g_PeriodicReceiveModeThread, PeriodicReceiveModeThread<Lsm6ds3PeriodicReceiveModeDataType, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataCountMax>, pSession,
                g_PeriodicReceiveModeThreadStack, nn::os::ThreadStackAlignment, NN_SYSTEM_THREAD_PRIORITY(sasbus, PeriodicReceiveModeThread))
        );
    }
    else
    {
        NN_ABORT("This device does not support this function\n");
    }

    nn::os::SetThreadNamePointer(&g_PeriodicReceiveModeThread, NN_SYSTEM_THREAD_NAME(sasbus, PeriodicReceiveModeThread));
    g_IsStartPeriodicReceiveModeThread = true;
    nn::os::StartThread(&g_PeriodicReceiveModeThread);

    return nn::ResultSuccess();
}

void StopPeriodicReceiveMode(Session* pSession) NN_NOEXCEPT
{
    // PeriodicReceiveMode が開始されてなかった時は何もせず返る
    if (!g_IsStartPeriodicReceiveModeThread)
    {
        return;
    }

    g_IsStartPeriodicReceiveModeThread = false;
    nn::os::WaitThread(&g_PeriodicReceiveModeThread);
    nn::os::DestroyThread(&g_PeriodicReceiveModeThread);

    return;
}

void Suspend()
{
    if (!g_IsSuspend)
    {
        if (g_IsStartPeriodicReceiveModeThread)
        {
            // TORIAEZU : 今は 1つだけなので、決めうち
            StopPeriodicReceiveMode(pSessionList[0]);
            g_IsStartPeriodicReceiveModeThreadBeforeSleeping = true;
        }
        else
        {
            g_IsStartPeriodicReceiveModeThreadBeforeSleeping = false;
        }

        g_IsSuspend = true;
        nne::spi::Suspend();
    }
}

void Resume()
{
    if(g_IsSuspend)
    {
        nne::spi::Resume();
        if (g_IsStartPeriodicReceiveModeThreadBeforeSleeping)
        {
            // TORIAEZU : 今は 1つだけなので、決めうち
            detail::SessionImpl& session = detail::ToSessionImpl(*pSessionList[0]);

            StartPeriodicReceiveMode(pSessionList[0], session.dataSize, session.reg, session.interval, pSessionList[0]->_receiveBuffer, pSessionList[0]->_receiveBufferSize);
        }

        g_IsSuspend = false;
    }
}

} // driver
} // sasbus
} // nn
