﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Macro.h>

#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>

#include "psm_ISupplyRouteDriver.h"
#include "psm_SupplyRouteManager.h"

namespace nn { namespace psm { namespace driver { namespace detail {

SupplyRouteManager::SupplyRouteManager() NN_NOEXCEPT
    : m_pSupplyRouteDriver(nullptr)
    , m_DataMutex(false)
    , m_IsVdd50BAvailable(false)
    , m_IsVdd50Enabled(false)
    , m_IsForceVdd50AEnabled(false)
    , m_SupplyRouteControlEnabled(true)
{
    // 何もしません。
}

SupplyRouteManager::~SupplyRouteManager() NN_NOEXCEPT
{
    // 何もしません。
}

::nn::Result SupplyRouteManager::Initialize(ISupplyRouteDriver* pSupplyRouteDriver, const SettingsHolder* pSettingsHolder) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pSupplyRouteDriver);

    m_pSupplyRouteDriver = pSupplyRouteDriver;
    m_pSettingsHolder = pSettingsHolder;

    NN_RESULT_DO(EnableVdd50());

    NN_RESULT_SUCCESS;
}

void SupplyRouteManager::Finalize() NN_NOEXCEPT
{
    // 何もしません。
}

// SIGLO-70002
// TODO: UsbPowerManager::ProcessPowerDeliveryNoticeEvent() において
// ::nn::usb::pd::ReplyPowerRequest(&m_UpdSession, true) をすぐに返すために Update 関数は呼ばれていない。
// usb:pd -> usb:pm の順番で割り込みが来た場合、最終的に UpdateUpdState が呼ばれるため問題にならないが
// usb:pd -> usb:pm の順番で割り込みが来ることは仕様上保障されておらず、今後、割り込みの順が変わるなどして
// VDD50 の切り替えタイミングが想定と異なることになる可能性がある。
// ハードウェアの破壊を引き起こす可能性はないが、ファンと Joy-Con の給電の有無に影響するため、近いうちに再度修正をする必要がある。
::nn::Result SupplyRouteManager::SetVdd50BAvailable(bool isAvailable) NN_NOEXCEPT
{
    SetVdd50BAvailableWithoutUpdate(isAvailable);
    NN_RESULT_DO(Update());

    NN_RESULT_SUCCESS;
}

void SupplyRouteManager::SetVdd50BAvailableWithoutUpdate(bool isAvailable) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_DataMutex)> locker(m_DataMutex);
    m_IsVdd50BAvailable = isAvailable;
}

::nn::Result SupplyRouteManager::EnableVdd50() NN_NOEXCEPT
{
    {
        ::std::lock_guard<decltype(m_DataMutex)> locker(m_DataMutex);
        if (m_IsVdd50Enabled)
        {
            NN_RESULT_SUCCESS;
        }

        m_IsVdd50Enabled = true;
    }
    NN_RESULT_DO(Update());

    NN_RESULT_SUCCESS;
}

::nn::Result SupplyRouteManager::DisableVdd50() NN_NOEXCEPT
{
    {
        ::std::lock_guard<decltype(m_DataMutex)> locker(m_DataMutex);
        if (!m_IsVdd50Enabled)
        {
            NN_RESULT_SUCCESS;
        }

        m_IsVdd50Enabled = false;
    }
    NN_RESULT_DO(Update());

    NN_RESULT_SUCCESS;
}

::nn::Result SupplyRouteManager::EnableForceVdd50A() NN_NOEXCEPT
{
    {
        ::std::lock_guard<decltype(m_DataMutex)> locker(m_DataMutex);
        if (m_IsForceVdd50AEnabled)
        {
            NN_RESULT_SUCCESS;
        }

        m_IsForceVdd50AEnabled = true;
    }
    NN_RESULT_DO(Update());

    NN_RESULT_SUCCESS;
}

::nn::Result SupplyRouteManager::DisableForceVdd50A() NN_NOEXCEPT
{
    {
        ::std::lock_guard<decltype(m_DataMutex)> locker(m_DataMutex);
        if (!m_IsForceVdd50AEnabled)
        {
            NN_RESULT_SUCCESS;
        }

        m_IsForceVdd50AEnabled = false;
    }
    NN_RESULT_DO(Update());

    NN_RESULT_SUCCESS;
}

::nn::Result SupplyRouteManager::Update() NN_NOEXCEPT
{
    auto supplyRoute = SupplyRoute_Invalid;

    {
        ::std::lock_guard<decltype(m_DataMutex)> dataLocker(m_DataMutex);
        if ( m_IsVdd50Enabled )
        {
            supplyRoute = GetAppropriateSupplyRoute(SupplyRoute_Vdd50B, SupplyRoute_Vdd50A);
        }
        else
        {
            supplyRoute = GetAppropriateSupplyRoute(SupplyRoute_None, SupplyRoute_None);
        }
    }

    NN_SDK_ASSERT(supplyRoute != SupplyRoute_Invalid);

    if ( m_SupplyRouteControlEnabled )
    {
        NN_RESULT_DO(m_pSupplyRouteDriver->SetSupplyRoute(supplyRoute));
    }

    NN_RESULT_SUCCESS;
}

SupplyRoute SupplyRouteManager::GetAppropriateSupplyRoute(SupplyRoute primaryRoute, SupplyRoute secondaryRoute) NN_NOEXCEPT
{
    if ( CanUpdateSupplyRoute(primaryRoute) )
    {
        return primaryRoute;
    }
    else if ( CanUpdateSupplyRoute(secondaryRoute) )
    {
        return secondaryRoute;
    }

    return SupplyRoute_None;
}

bool SupplyRouteManager::CanUpdateSupplyRoute(SupplyRoute state) NN_NOEXCEPT
{
    if ( state == SupplyRoute_Vdd50B )
    {
        return (!m_IsForceVdd50AEnabled
            && m_pSettingsHolder->IsVdd50BEnabled()
            && m_IsVdd50BAvailable);
    }
    else
    {
        return true;
    }
}

void SupplyRouteManager::EnableSupplyRouteControl() NN_NOEXCEPT
{
    m_SupplyRouteControlEnabled = true;
}

void SupplyRouteManager::DisableSupplyRouteControl() NN_NOEXCEPT
{
    m_SupplyRouteControlEnabled = false;
}

}}}} // namespace nn::psm::driver::detail
