﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/psm/psm_ErrorReportTypes.h>
#include <nn/usb/usb_Pm.h>

namespace nn { namespace psm { namespace driver { namespace detail {

class ErrorReporter final
{
    NN_DISALLOW_COPY(ErrorReporter);
    NN_DISALLOW_MOVE(ErrorReporter);

public:
    ErrorReporter() NN_NOEXCEPT;
    ~ErrorReporter() NN_NOEXCEPT;

    //! エラーレポートのコンテキストの入力電流上限を更新します。
    void SetInputCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの出力電流上限を更新します。
    void SetBoostModeCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの充電電流上限を更新します。
    void SetFastChargeCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの充電電圧上限を更新します。
    void SetChargeVoltageLimit(int limitMilliVolt) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの充電設定を更新します。
    void SetChargeConfiguration(int chargeConfiguration) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの Hi-Z モードを更新します。
    void SetHizMode(bool hizMode) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの nCE 端子を更新します。
    void SetChargeEnabled(bool chargeEnabled) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの給電経路を更新します。
    void SetPowerSupplyPath(int state) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの電池温度を更新します。
    void SetBatteryTemperature(double temperature) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの電池残量を更新します。
    void SetBatteryChargePercent(double percent) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの電池電圧を更新します。
    void SetBatteryChargeVoltage(int milliVolt) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの電池劣化を更新します。
    void SetBatteryAge(double percent) NN_NOEXCEPT;

    //! エラーレポートのコンテキストのパワーロールを更新します。
    void SetPowerRole(::nn::usb::UsbPowerRole usbPowerRole) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの給電方式を更新します。
    void SetPowerSupplyType(::nn::usb::UsbChargerType usbChargerType) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの供給電圧を更新します。
    void SetPowerSupplyVoltage(int powerSupplyVoltage) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの供給電流を更新します。
    void SetPowerSupplyCurrent(int powerSupplyCurrent) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの高速充電モードを更新します。
    void SetFastBatteryChargingEnabled(bool fastBatteryChargingEnabled) NN_NOEXCEPT;

    //! エラーレポートのコンテキストのコントローラ充電要求を更新します。
    void SetControllerPowerSupplyAcquired(bool controllerPowerSupplyAcquired) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの OTG 要求を更新します。
    void SetOtgRequested(bool otgRequested) NN_NOEXCEPT;

    //! エラーレポートのコンテキストの更新通知システムイベントへのポインタを取得します。
    void GetBatteryChargeInfoEvent(::nn::os::SystemEventType** pOutSystemEventPointer) NN_NOEXCEPT;

    //! エラーレポートのコンテキストを取得します。
    void GetBatteryChargeInfoFields(BatteryChargeInfoFields* pOutFields) NN_NOEXCEPT;

    //! いずれかのフィールドに更新がある場合エラーレポートのコンテキストの更新通知システムイベントをシグナルします。
    void Submit() NN_NOEXCEPT;

private:
    //! いずれかのフィールドに更新があることを表すフラグです。
    bool m_Dirty;

    //! エラーレポートのコンテキストです。
    BatteryChargeInfoFields m_BatteryChargeInfoFields;

    //! エラーレポートのコンテキストの更新通知システムイベントです。
    ::nn::os::SystemEventType m_BatteryChargeInfoEvent;
};

ErrorReporter& GetErrorReporter() NN_NOEXCEPT;

}}}} // namespace nn::psm::driver::detail
