﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>

#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>

#include <nn/gpio/gpio_PadAccessor.h>
#include <nn/os/os_Mutex.h>

#include "psm_IChargerDriver.h"

namespace nn { namespace psm { namespace driver { namespace detail {

class ChargerDriver final : public IChargerDriver
{
    NN_DISALLOW_COPY(ChargerDriver);
    NN_DISALLOW_MOVE(ChargerDriver);

private:
    enum class GpioSettings
    {
        Uninitialized,  //!< 未初期化状態
        Vdd15Hb,        //!< ホストブリッジ給電
        UsbVbus,        //!< USB給電
    };

public:
    //! 設定できるウォッチドッグタイマの満了期間の最大値
    static const ::nn::TimeSpan WatchdogTimerPeriodMax;

public:
    ChargerDriver() NN_NOEXCEPT;

    virtual ~ChargerDriver() NN_NOEXCEPT NN_OVERRIDE
    {
        // 何もしない
    }

    //! チャージャを初期化します。
    virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE;

    //! チャージャの終了処理をします。
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE;

    //! 充電を有効にします。
    virtual ::nn::Result EnableBatteryCharging() NN_NOEXCEPT NN_OVERRIDE;

    //! OTG モードを有効にします。
    virtual ::nn::Result EnableOnTheGo() NN_NOEXCEPT NN_OVERRIDE;

    //! HiZ モードを開始します。
    virtual ::nn::Result EnterHiZMode() NN_NOEXCEPT NN_OVERRIDE;

    //! HiZ モードを終了します。
    virtual ::nn::Result ExitHiZMode() NN_NOEXCEPT NN_OVERRIDE;

    //! 入力電圧の下限を設定します。
    virtual ::nn::Result SetInputVoltageLimitMilliVolt(int inputVoltageLimitMilliVolt) NN_NOEXCEPT NN_OVERRIDE;

    //! 入力電流を制限します。
    virtual ::nn::Result SetInputCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT NN_OVERRIDE;

    //! 高速充電の入力電流を制限します。
    virtual ::nn::Result SetFastChargeCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT NN_OVERRIDE;

    //! 高速充電の入力電流を制限します。
    virtual ::nn::Result SetFastChargeCurrentLimit(int limitMilliAmpere, FastChargeCurrentLimitOption fastChargeCurrentLimitOption) NN_NOEXCEPT NN_OVERRIDE;

    //! ブーストモード時の出力電流を制限します。
    virtual ::nn::Result SetBoostModeCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT NN_OVERRIDE;

    //! 充電状況を取得します。
    virtual ::nn::Result GetStatus(Status* pStatus) NN_NOEXCEPT NN_OVERRIDE;

    //! 充電電圧を制限します。
    virtual ::nn::Result SetChargeVoltageLimit(int limitMilliVolt) NN_NOEXCEPT NN_OVERRIDE;

    //! フォルトステータスを取得します。
    virtual ::nn::Result GetFaultStatus(FaultStatus* pFaultStatus) NN_NOEXCEPT NN_OVERRIDE;

    //! ウォッチドッグの満了期間を設定します。
    virtual ::nn::Result SetWatchdogTimerPeriod(::nn::TimeSpan period) NN_NOEXCEPT NN_OVERRIDE;

    //! ウォッチドッグを有効にします。
    virtual ::nn::Result EnableWatchdogTimer() NN_NOEXCEPT NN_OVERRIDE;

    //! ウォッチドッグを無効にします。
    virtual ::nn::Result DisableWatchdogTimer() NN_NOEXCEPT NN_OVERRIDE;

    //! ウォッチドッグをリセットします。
    virtual ::nn::Result ResetWatchdogTimer() NN_NOEXCEPT NN_OVERRIDE;

    //! ウォッチドッグが有効かどうか取得します。
    virtual bool IsWatchdogTimerEnabled() NN_NOEXCEPT NN_OVERRIDE;

    //! 割込みイベントを取得します。
    virtual ::nn::Result BindInterrupt(::nn::os::SystemEventType* pOutInterruptEvent) NN_NOEXCEPT NN_OVERRIDE;

    //! 割込みイベントをクリアします。
    virtual ::nn::Result ClearInterrupt(::nn::os::SystemEventType* pInterruptEvent) NN_NOEXCEPT NN_OVERRIDE;

    //! ハードウェア上での充電可否を設定します。
    virtual ::nn::Result SetChargeEnable(bool enabled) NN_NOEXCEPT NN_OVERRIDE;

    //! 給電能力が十分な状態であるように振舞うかどうかを設定します。
    virtual ::nn::Result SetEnoughPowerChargeEmulation(bool isEnabled) NN_NOEXCEPT NN_OVERRIDE;

private:
    //! チャージャ用 GPIO を操作し, USBから給電する設定にします。
    void SetUpUsbVbusChargingGpio() NN_NOEXCEPT;

    //! チャージャ用 GPIO を操作し, 充電設定にします。
    //  呼び出し前には必ずVConnを遮断してください
    void SetUpVdd15HbChargingGpio() NN_NOEXCEPT;

    //! OnTheGo 設定と EnoughPowerChargeEmulation 設定を変更し, 給電元を適切な手順で切り替えます.
    ::nn::Result SetupPowerSupply(bool isOnTheGoRequested, bool isEnoughPowerChargeEmulationRequested) NN_NOEXCEPT;
private:
    //! VCONN 端子への 5V を供給を制御するためのセッション
    ::nn::gpio::GpioPadSession m_PdVConnEnSession;

    //! Charger の OTG 端子と TypeC コネクタ接続（SDEV 対応）を制御するためのセッション
    ::nn::gpio::GpioPadSession m_OtgFet1Session;

    //! WiiU アダプタからの Charger への電圧印可防止（SDEV 対応）を制御するためのセッション
    ::nn::gpio::GpioPadSession m_OtgFet2Session;

    //! ウォッチドッグタイマの満了期間
    ::nn::TimeSpan m_WatchdogTimerPeriod;

    //! ウォッチドッグタイマが有効かどうか
    ::std::atomic<bool> m_IsWatchdogTimerEnabled;

    //! USBから十分な給電がされているときの振る舞いを強制させるかどうか
    bool m_IsEnoughPowerChargeEmulationEnabled;

    //! OnTheGo 設定が有効かどうか
    bool m_IsOnTheGoEnabled;

    //! GPIO 設定
    GpioSettings m_GpioSettings;

    //! チャージャからの割り込みを受け取るためのセッション
    ::nn::gpio::GpioPadSession m_ChargerInterruptSession;

    //! 充電の有効・無効を制御するためのセッション
    ::nn::gpio::GpioPadSession m_ChargeEnableSession;

    //! 給電元の変更操作を排他処理するためのMutex
    //  シングルスレッドなので必要ないが念のため入れている
    nn::os::Mutex m_PowerSupplySwitchingMutex;
};

}}}} // namespace nn::psm::driver::detail
