﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/util/util_FormatString.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_ExpHeapAllocator.h>

#include <nn/psc/psc_Result.h>
#include <nn/psc/server/psc_Server.h>
#include <nn/psc/sfdl/psc.sfdl.h>

#include "psc_PmControlSfObject.h"
#include "psc_PmModuleSfObject.h"

namespace nn     {
namespace psc    {
namespace server {
namespace        {

struct ServerOptions
{
    static const int SubDomainCountMax         = 64;
    static const int ObjectInSubDomainCountMax = 200;
};

const int SessionThreadStackSize = 8192;
const int ControlSessionCount    = 1;
const int ModuleSessionCount     = 50;
const int SessionMaxCount        = ControlSessionCount + ModuleSessionCount;
const int SessionThreadCount     = 1;

nn::sf::ExpHeapAllocator g_ServiceFrameworkAllocator;

enum SessionPort
{
    SessionPort_Control,
    SessionPort_Module,
    SessionPort_Highest
};

class PmServiceSfObject :
    public nn::sf::ISharedObject
{

public:
    nn::Result Initialize(nn::sf::Out<nn::sf::SharedPointer<nn::psc::sf::IPmModule>> pModuleOut)
    NN_NOEXCEPT
    {
        auto pInterface = nn::sf::ObjectFactory<
                           nn::sf::ExpHeapAllocator::Policy
                          >::CreateSharedEmplaced<
                           nn::psc::sf::IPmModule,
                           PmModuleSfObject
                          >(&g_ServiceFrameworkAllocator);

        if (pInterface != nullptr)
        {
            pModuleOut.Set(pInterface);
            return ResultSuccess();
        }

        return nn::psc::ResultOutOfMemory();
    }
};

class PscService
    : public nn::sf::HipcSimpleAllInOneServerManager<SessionMaxCount, SessionPort_Highest, ServerOptions>
{
private:
    nn::os::ThreadType m_SessionThread[SessionThreadCount];
    uint8_t            m_SessionThreadStack[SessionThreadCount][SessionThreadStackSize] NN_ALIGNAS(os::StackRegionAlignment);

    static void SessionThreadEntry(void* arg)
    NN_NOEXCEPT
    {
        PscService* p = (PscService*)arg;
        p->LoopAuto();
    }

    nn::Result OnNeedsToAccept(int portIndex, PortObjectImpl* pPort)
    NN_NOEXCEPT
    {
        if (portIndex == SessionPort_Control)
        {
            auto interface = nn::sf::ObjectFactory<
                              nn::sf::ExpHeapAllocator::Policy
                             >::CreateSharedEmplaced<
                              nn::psc::sf::IPmControl,
                              PmControlSfObject
                             >(&g_ServiceFrameworkAllocator);

            return this->AcceptImpl(pPort, interface);
        }
        else if (portIndex == SessionPort_Module)
        {
            auto interface = nn::sf::ObjectFactory<
                              nn::sf::ExpHeapAllocator::Policy
                             >::CreateSharedEmplaced<
                              nn::psc::sf::IPmService,
                              PmServiceSfObject
                             >(&g_ServiceFrameworkAllocator);

            return this->AcceptImpl(pPort, interface);
        }

        return nn::psc::ResultNotSupported();
    }

public:
    void Wait()
    NN_NOEXCEPT
    {
        for (auto i = 0; i < SessionThreadCount; i++)
        {
            nn::os::WaitThread(&m_SessionThread[i]);
        }
    }

    nn::Result Initialize()
    NN_NOEXCEPT
    {
        nn::Result result;

        result = this->InitializePort(
                             SessionPort_Control,
                             ControlSessionCount,
                             PmControlPortName);

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        result = this->InitializePort(
                             SessionPort_Module,
                             ModuleSessionCount,
                             PmModulePortName);

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        this->Start();

        for (auto i = 0; i < SessionThreadCount; i++)
        {
            nn::os::CreateThread(
                             &m_SessionThread[i],
                             SessionThreadEntry,
                             this,
                             m_SessionThreadStack[i],
                             SessionThreadStackSize,
                             NN_SYSTEM_THREAD_PRIORITY(psc, IpcServer));
            nn::os::SetThreadNamePointer(&m_SessionThread[i], NN_SYSTEM_THREAD_NAME(psc, IpcServer));

            nn::os::StartThread(&m_SessionThread[i]);
        }

        return ResultSuccess();
    }

} g_PscServer;

} // namespacee

nn::lmem::HeapHandle g_HeapHandle;

nn::Result Initialize(void* pPscMemoryIn, size_t pscMemorySize)
NN_NOEXCEPT
{
    g_HeapHandle = nn::lmem::CreateExpHeap(
                             reinterpret_cast<uint8_t*>(pPscMemoryIn),
                             pscMemorySize,
                             nn::lmem::CreationOption_ThreadSafe);

    NN_ABORT_UNLESS(g_HeapHandle != nullptr);

    g_ServiceFrameworkAllocator.Attach(g_HeapHandle);

    return g_PscServer.Initialize();
}

void Wait()
NN_NOEXCEPT
{
    g_PscServer.Wait();
}

}}}
