﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/psc/psc_Types.h>
#include <nn/psc/psc_PmControlTypes.h>
#include <nn/psc/util/psc_Util.h>

namespace nn     {
namespace psc    {
namespace server {

class PmModule;

class PmControl
{

private:
    static const size_t ThreadStackSize = (16 * 1024);

#if defined(NN_DETAIL_PSC_ENABLE_PERFORMANCE_CHECK)
    nn::os::Tick                m_StartTick;
    nn::os::Tick                m_EndTick;
#endif

    nn::os::ThreadType          m_Thread;
    nn::os::MultiWaitType       m_MultiWait;
    nn::os::SystemEvent         m_StartEvent;
    nn::os::SystemEvent         m_DoneEvent;
    nn::os::SystemEvent         m_CancelEvent;
    nn::os::MultiWaitHolderType m_CancelEventHolder;

    bool                        m_Initialized;
    bool                        m_Running;

    PmState                     m_PreviousState;
    PmState                     m_State;
    PmFlagSet                   m_Flags;
    PmTransitionOrder           m_Order;
    PmControlState              m_ControlState;
    uint32_t                    m_PendingCount;
    nn::Result                  m_Result;

    NN_OS_ALIGNAS_THREAD_STACK uint8_t
                                m_Stack[ThreadStackSize];

    void LinkEvent(PmModule* pModule, nn::os::SystemEvent* pEvent)
    NN_NOEXCEPT;

    void UnlinkEvent(PmModule* pModule)
    NN_NOEXCEPT;

    void UnlinkAll()
    NN_NOEXCEPT;

    nn::Result SendRequestToOne(PmModule* pModule, PmState state, PmFlagSet flags, PmTransitionOrder order)
    NN_NOEXCEPT;

    nn::Result SendRequestToAll(PmState state, PmFlagSet flags, PmTransitionOrder order)
    NN_NOEXCEPT;

    nn::Result WaitCompletionAll(PmState state, PmFlagSet flags, PmTransitionOrder order)
    NN_NOEXCEPT;

    void Thread()
    NN_NOEXCEPT;

    static void ThreadEntry(void* arg)
    NN_NOEXCEPT
    {
        PmControl* p = reinterpret_cast<PmControl*>(arg);
        p->Thread();
    }

    void GetTransitionInformation(nn::psc::util::TransitionInformation* pOutInfo)
    NN_NOEXCEPT;

    int GetModuleCount()
    NN_NOEXCEPT;

    nn::Result GetModuleTransitionInformation(int *pOutCount, nn::psc::util::ModuleTransitionInformation* pOutInfo, int maxCount)
    NN_NOEXCEPT;

    int GetDependencyCount()
    NN_NOEXCEPT;

    nn::Result GetDependencyInformation(int *pOutCount, nn::psc::util::DependencyInformation* pOutInfo, int maxCount)
    NN_NOEXCEPT;

public:
    nn::Result GetModuleInformation(
        nn::psc::util::TransitionInformation* pOutTransitionInfo,
        int *pOutModuleCount,
        nn::psc::util::ModuleTransitionInformation* pOutModuleInfo,
        int maxModuleCount,
        int *pOutDependencyCount,
        nn::psc::util::DependencyInformation* pOutDependencyInfo,
        int maxDependencyCount
    ) NN_NOEXCEPT;

    nn::Result PrintModuleInformation()
    NN_NOEXCEPT;

    nn::Result DispatchRequest(PmState state, PmFlagSet flags, PmTransitionOrder order)
    NN_NOEXCEPT;

    nn::Result GetState(PmControlState* pStateOut)
    NN_NOEXCEPT;

    nn::Result GetResult()
    NN_NOEXCEPT;

    nn::Result Cancel()
    NN_NOEXCEPT;

    nn::Result Initialize(nn::os::SystemEvent** pEventPointerOut)
    NN_NOEXCEPT;

    void Finalize()
    NN_NOEXCEPT;

    PmControl()
    NN_NOEXCEPT;

    ~PmControl()
    NN_NOEXCEPT;
};

}}}
