﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>

#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_HipcClientProxyByName.h>     // for nn::sf::CreateHipcProxyByName
#include <nn/sf/sf_ExpHeapAllocator.h>          // for nn::sf::ExpHeapStaticAllocator
#include <nn/nn_Abort.h>

#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_SdkSystemEventApi.h>
#include <nn/dbg/dbg_Api.h>
#include <nn/svc/svc_Result.h>
#include <nn/profiler/profiler_Result.h>

#include "profiler_Hipc.h"
#include "profiler_IProfiler.sfdl.h"
#include "profiler_Logging.h"
#include "profiler_StringTable.h"
#include "profiler_Svc.autogen.h"
#include "profiler_TargetApplication.h"

namespace nn { namespace profiler {

    const char ProfilerServiceName[] = "banana";
    nn::os::SystemEventType     g_ProfilerProcessEvent;

    namespace /*anonymous*/ {

        nn::sf::SharedPointer<nn::profiler::IProfiler> g_Profiler = nullptr;

        // 拡張ヒープを使用したアロケータを、HIPC プロキシ用に準備します。
        struct CreateProfilerByHipcTag;
        typedef nn::sf::ExpHeapStaticAllocator<1024 * 16, CreateProfilerByHipcTag> MyAllocator;

        struct TransferMemoryDetails
        {
            uintptr_t startAddress;
            size_t size;
        } transferMemoryDetails;

        // MyAllocator を静的コンストラクタで初期化するためのヘルパー
        class MyAllocatorInitializer
        {
        public:
            MyAllocatorInitializer() NN_NOEXCEPT
            {
                MyAllocator::Initialize(nn::lmem::CreationOption_NoOption);
            }
        } g_MyAllocatorInitializer;

        uint64_t GetProcessId()
        {
#if defined(NN_BUILD_CONFIG_CPU_CORTEX_A57_AARCH64)
            unsigned long pid = 0;
#else
            unsigned long long pid = 0;
#endif
            nn::Result result = nn::svc::profiler::GetProcessId(&pid, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
            if (result.IsFailure())
            {
                DumpResultInformation(LOG_AS_FATAL, result);
                NN_SDK_REQUIRES(result.IsSuccess());
            }
            return static_cast<uint64_t>(pid);
        }

    } // anonymous

    nn::sf::SharedPointer<IProfiler> CreateProfilerByHipc() NN_NOEXCEPT
    {
        // ProfilerServiceName のサービス名で示される HIPC サービスオブジェクトを、
        // IProfiler としてプロキシクライアントを作成して返します。
        // アロケーションポリシーとして MyAllocator::Policy を渡します。
        nn::sf::SharedPointer<IProfiler> ret;
        auto result = nn::sf::CreateHipcProxyByName<IProfiler, MyAllocator::Policy>(&ret, ProfilerServiceName);
        if (result.IsFailure())
        {
            DumpResultInformation(LOG_AS_ERROR, result);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
        return ret;
    }

    void InitializeProfilerProcess() NN_NOEXCEPT
    {
        g_Profiler = nullptr;

        // Get service object
        g_Profiler = nn::profiler::CreateProfilerByHipc();
    }

    void FinalizeProfilerProcess() NN_NOEXCEPT
    {
        g_Profiler = nullptr;

        // Sleep here to give the shared pointer time to release.
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));
    }

    bool IsProfilerProcessInitialized() NN_NOEXCEPT
    {
        return g_Profiler != nullptr;
    }

    //void StartSignalingProfilerEvent(int64_t interval, int coreCount) NN_NOEXCEPT
    //{
    //    g_Profiler->StartSignalingEvent(interval);
    //}

    //void StopSignalingProfilerEvent() NN_NOEXCEPT
    //{
    //    g_Profiler->StopSignalingEvent();
    //}

    nn::Result LibraryInitialize(uint32_t sdkVersion) NN_NOEXCEPT
    {
        DUMP_CURRENT_LINE();

        nn::Result result;
        uint64_t pid = GetProcessId();
        result = g_Profiler->LibraryInitialize(sdkVersion, pid);
        if (result.IsFailure())
        {
            FATAL_LOG("Failed to initialize profiler library!\n");
            DumpResultInformation(LOG_AS_FATAL, result);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        nn::sf::NativeHandle nh;
        result = g_Profiler->GetSystemEvent(&nh);
        if (result.IsFailure())
        {
            FATAL_LOG("Failed to get system event from profiler process\n");
            DumpResultInformation(LOG_AS_FATAL, result);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        DUMP_CURRENT_LINE();

        nn::os::AttachReadableHandleToSystemEvent(
            &g_ProfilerProcessEvent,
            nh.GetOsHandle(),
            nh.IsManaged(),
            nn::os::EventClearMode_ManualClear);
        nh.Detach();

        SendBasicIpcMessage(ProfilerIpcMessage_CoreMask, TargetApplication::GetCoreMask());

        return result;
    }

    nn::Result LibraryFinalize() NN_NOEXCEPT
    {
        DUMP_CURRENT_LINE();
        nn::Bit64 pid = GetProcessId();
        return g_Profiler->LibraryFinalize(pid);
    }

    ProfilerStatus GetProfilerStatusFromServer() NN_NOEXCEPT
    {
        DUMP_CURRENT_LINE();
        ProfilerStatus status = ProfilerStatus_Offline;

        nn::Result result;
        int32_t tempStatus;
        result = g_Profiler->GetProfilerStatus(&tempStatus);
        if (result.IsSuccess())
        {
            status = static_cast<ProfilerStatus>(tempStatus);
        }
        else
        {
            DumpResultInformation(LOG_AS_ERROR, result);
        }

        return status;
    }

    nn::Result GetIpcEvent(IpcEventInfo* pOutEvent) NN_NOEXCEPT
    {
        nn::sf::OutBuffer buffer(reinterpret_cast<char*>(pOutEvent), sizeof(*pOutEvent));
        return g_Profiler->GetIpcEvent(buffer);
    }

    nn::Result CloseAndGetNextBuffer(nn::os::NativeHandle* pOutBuffer, size_t filledSize, uint32_t index) NN_NOEXCEPT
    {
        DUMP_CURRENT_LINE();

        nn::sf::NativeHandle nh;
        nn::Result result;

        result = g_Profiler->CloseAndGetNextBuffer(&nh, static_cast<uint64_t>(filledSize), index);
        if (result.IsFailure())
        {
            *pOutBuffer = nn::os::InvalidNativeHandle;
            return result;
        }

        *pOutBuffer = nh.GetOsHandle();
        nh.Detach();

        return nn::ResultSuccess();
    }

    nn::Result CloseAndFinalizeBuffer(size_t filledSize, uint32_t index) NN_NOEXCEPT
    {
        DUMP_CURRENT_LINE();
        return g_Profiler->CloseAndFinalizeBuffer(static_cast<uint64_t>(filledSize), index);
    }

    nn::Result SendMessageToPc(uint32_t id, const void* buffer, size_t size, bool callback) NN_NOEXCEPT
    {
        DUMP_CURRENT_LINE();
        nn::Result result;
        nn::sf::InBuffer data(reinterpret_cast<const char*>(buffer), size);
        result = g_Profiler->SendMessageToPcInBuffer(id, data, callback);
        if (nn::svc::ResultInvalidCurrentMemory::Includes(result))
        {
            nn::sf::InBuffer nullData(nullptr, size);
            result = g_Profiler->SendMessageToPcTransferMem(id, reinterpret_cast<uint64_t>(buffer), size, callback);
        }
        return result;
    }

    nn::Result StartMultipartMessageToPc(uint32_t id, uint64_t totalSize, uint32_t pieces, bool callback) NN_NOEXCEPT
    {
        DUMP_CURRENT_LINE();
        DEBUG_LOG("Attempting to start multipart message: %d, %lld, %d, %d\n", id, totalSize, pieces, callback);
        return g_Profiler->StartMultipartMessageToPc(id, totalSize, pieces, callback);
    }

    nn::Result SendMultipartMessageToPc(const void* buffer, size_t size, int remaining) NN_NOEXCEPT
    {
        DUMP_CURRENT_LINE();
        nn::Result result;
        nn::sf::InBuffer data(reinterpret_cast<const char*>(buffer), size);
        result = g_Profiler->SendMultipartMessageToPcInBuffer(data, remaining);
        if (nn::svc::ResultInvalidCurrentMemory::Includes(result))
        {
            nn::sf::InBuffer nullData(nullptr, size);
            result = g_Profiler->SendMultipartMessageToPcTransferMem(reinterpret_cast<uint64_t>(buffer), size, remaining);
        }
        return result;
    }

    nn::Result GetGlobalProfileSettings(void* buffer, size_t size) NN_NOEXCEPT
    {
        DUMP_CURRENT_LINE();
        nn::sf::OutBuffer data(reinterpret_cast<char*>(buffer), size);
        return g_Profiler->GetProfileSettings(data);
    }

    nn::Result SetGlobalProfileSettings(
        uint32_t affinityMask,
        uint32_t flags,
        uint32_t performanceCounterGroup,
        uint32_t sampleRate) NN_NOEXCEPT
    {
        DUMP_CURRENT_LINE();
        return g_Profiler->SetProfileSettings(affinityMask, flags, performanceCounterGroup, sampleRate);
    }

    nn::Result TransferSampleBuffers(nn::os::NativeHandle transferMemHandle, uintptr_t originalAddress, size_t size) NN_NOEXCEPT
    {
        DUMP_CURRENT_LINE();
        transferMemoryDetails.startAddress = originalAddress;
        transferMemoryDetails.size = size;
        return g_Profiler->TransferSampleBuffers(nn::sf::NativeHandle(transferMemHandle, true), originalAddress, size);
    }

    nn::Result SendSampleBufferAsMultipart(uint32_t blockId, size_t size, int remaining) NN_NOEXCEPT
    {
        DUMP_CURRENT_LINE();
        NN_ABORT();
        return g_Profiler->SendSampleBufferAsMultipart(blockId, size, remaining);
    }

    bool IsPcSynced() NN_NOEXCEPT
    {
        DUMP_CURRENT_LINE();
        bool isSynced = false;
        nn::Result result = g_Profiler->IsPcSynced(&isSynced);
        NN_UNUSED(result);
        NN_SDK_REQUIRES(result.IsSuccess());
        return isSynced;
    }


    nn::Result SendBufferToProcess(const void* buffer, size_t size, SendBufferType type) NN_NOEXCEPT
    {
        nn::sf::InBuffer sfbuf(reinterpret_cast<const char*>(buffer), size);
        return g_Profiler->SendBuffer(sfbuf, type);
    }


    nn::Result SendBasicIpcMessage(ProfilerIpcMessage message, uint64_t data) NN_NOEXCEPT
    {
        return g_Profiler->SendIpcMessage(message, data);
    }

}} // nn::profiler
