﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/service/core/prepo_UploadThread.h>
#include <nn/prepo/detail/service/core/prepo_CategoryObject.h>
#include <nn/prepo/detail/service/core/prepo_DebugSettings.h>
#include <nn/prepo/detail/service/core/prepo_StatisticsManager.h>
#include <nn/bgtc/bgtc_Task.h>
#include <nn/fs/fs_PriorityPrivate.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

namespace
{
    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK Bit8 g_ThreadStack[16 * 1024];

    nn::os::TimerEvent g_RequestTransmissionEvent(nn::os::EventClearMode_ManualClear);
    nn::os::Event g_StopEvent(nn::os::EventClearMode_ManualClear);

    nn::bgtc::Task g_BgTask;

    std::atomic<bool> g_IsImmediate(false);
    std::atomic<bool> g_IsInSleep(false);
    std::atomic<nn::os::Tick> g_PreviousExecutionTick; // tick = 0 で初期化される。
}

namespace
{
    const auto TransmissionIntervalMin = DebugSettings::GetTransmissionIntervalMin();
    const auto TransmissionRetryIntervalMin = DebugSettings::GetTransmissionRetryIntervalMin();
    const auto TransmissionRetryIntervalMax = DebugSettings::GetTransmissionRetryIntervalMax();
    const auto TransmissionIntervalSecondsInSleep = static_cast<nn::bgtc::Interval>(DebugSettings::GetTransmissionIntervalInSleep().GetSeconds());
}

namespace
{
    nn::TimeSpan GenerateRetryInterval() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_GREATER_EQUAL(TransmissionRetryIntervalMax, TransmissionRetryIntervalMin);

        const auto diff = TransmissionRetryIntervalMax - TransmissionRetryIntervalMin;

        int64_t randomValue;
        nn::os::GenerateRandomBytes(&randomValue, sizeof (randomValue));
        randomValue &= 0x7FFFFFFFFFFFFFFull;

        const auto delta = nn::TimeSpan::FromNanoSeconds(randomValue % (diff.GetNanoSeconds() + 1));

        const auto interval = TransmissionRetryIntervalMin + delta;

        NN_SDK_ASSERT_MINMAX(interval, TransmissionRetryIntervalMin, TransmissionRetryIntervalMax);

        return interval;
    }

    nn::TimeSpan GenerateNormalInterval() NN_NOEXCEPT
    {
        const auto elapsed = (nn::os::GetSystemTick() - g_PreviousExecutionTick).ToTimeSpan();

        const auto interval = std::max(TransmissionIntervalMin - elapsed, nn::TimeSpan::FromSeconds(0));

        NN_SDK_ASSERT_MINMAX(interval, nn::TimeSpan(0), TransmissionIntervalMin);

        return interval;
    }
}

namespace
{
    bool IsErrorReportRequired(nn::Result result) NN_NOEXCEPT
    {
        return true
            && !ResultNetworkUnavailable::Includes(result)
            && !ResultTransmissionNotAgreed::Includes(result)
            && !ResultHttpErrorCouldntResolveHost::Includes(result)
            && !ResultHttpErrorCouldntConnect::Includes(result)
            && !ResultHttpErrorOperationTimedout::Includes(result)
            && !ResultHttpErrorSslConnectError::Includes(result)
            && !ResultHttpErrorPeerFailedVerification::Includes(result);
    }

    void UploadThreadFunction(void*) NN_NOEXCEPT
    {
        if (!DebugSettings::IsBackgroundProcessingEnabled())
        {
            NN_DETAIL_PREPO_INFO("[prepo] Exit upload thread because which is disabled by debug settings.\n");
            return;
        }

        nn::fs::SetPriorityRawOnCurrentThread(nn::fs::PriorityRaw_Background);
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::SetPriorityRawOnCurrentThread(nn::fs::PriorityRaw_Normal);
        };

        auto& bgTaskScheduleEvent = g_BgTask.GetScheduleEvent();

        while (NN_STATIC_CONDITION(true))
        {
            nn::os::WaitAny(g_StopEvent.GetBase(), g_RequestTransmissionEvent.GetBase(), bgTaskScheduleEvent.GetBase());

            if (g_StopEvent.TryWait())
            {
                return;
            }

            if (g_RequestTransmissionEvent.TryWait())
            {
                g_RequestTransmissionEvent.Clear();
            }

            if (bgTaskScheduleEvent.TryWait())
            {
                bgTaskScheduleEvent.Clear();
            }

            g_PreviousExecutionTick = nn::os::GetSystemTick();

            const auto notifyResult = g_BgTask.NotifyStarting();

            NN_UTIL_SCOPE_EXIT
            {
                if (notifyResult.IsSuccess())
                {
                    g_BgTask.NotifyFinished();
                }
            };

            const auto uploadResult = CategoryObject::ExecuteUpload(g_IsImmediate.exchange(false));

            if (uploadResult.IsFailure())
            {
                g_RequestTransmissionEvent.StartOneShot(GenerateRetryInterval());

                if (IsErrorReportRequired(uploadResult))
                {
                    detail::service::util::ReportError(uploadResult);
                    NN_DETAIL_PREPO_ERROR(NN_DETAIL_PREPO_STRING_ERROR(
                        "Failed to upload with error %03d-%04d\n"),
                        uploadResult.GetModule(), uploadResult.GetDescription());
                }
            }
        }
    }
}

void UploadThread::Start() NN_NOEXCEPT
{
    g_BgTask.Initialize();
    g_BgTask.SetClientName("prepo");

    nn::os::CreateThread(&g_Thread, UploadThreadFunction, nullptr,
        g_ThreadStack, sizeof (g_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(prepo, Uploader));
    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(prepo, Uploader));

    nn::os::StartThread(&g_Thread);
}

void UploadThread::Stop() NN_NOEXCEPT
{
    g_StopEvent.Signal();

    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);

    g_BgTask.Finalize();
}

void UploadThread::RequestTransmission() NN_NOEXCEPT
{
    if (!g_IsInSleep)
    {
        g_IsImmediate = true;
        g_RequestTransmissionEvent.Signal();

        NN_DETAIL_PREPO_INFO("[prepo] Schedule immediate transmission.\n");
    }
}

void UploadThread::RequestTransmissionForBackground() NN_NOEXCEPT
{
    if (!g_IsInSleep)
    {
        const auto interval = GenerateNormalInterval();
        g_RequestTransmissionEvent.StartOneShot(interval);

        NN_DETAIL_PREPO_INFO("[prepo] Schedule transmission after %lld seconds.\n", interval.GetSeconds());
    }
}

TransmissionStatus UploadThread::GetTransmissionStatus(ReportCategory category) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(category, 0, ReportCategory_Count);
    return CategoryObject::GetUploadExecuter(category).GetTransmissionStatus();
}

nn::Result UploadThread::GetLastError(ReportCategory category) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(category, 0, ReportCategory_Count);
    return CategoryObject::GetUploadExecuter(category).GetLastError();
}

void UploadThread::NotifySystemFullAwaked() NN_NOEXCEPT
{
    g_IsInSleep = false;
    g_BgTask.Unschedule();
    g_RequestTransmissionEvent.Signal();

    NN_DETAIL_PREPO_INFO("[prepo] Schedule transmission on waking.\n");
}

void UploadThread::NotifySystemSleepReady(bool isFromFullAwake) NN_NOEXCEPT
{
    if (isFromFullAwake)
    {
        g_IsInSleep = true;
        g_RequestTransmissionEvent.Clear();
        g_BgTask.SchedulePeriodic(TransmissionIntervalSecondsInSleep, TransmissionIntervalSecondsInSleep);

        NN_DETAIL_PREPO_INFO("[prepo] Schedule transmission in %d seconds period.\n", TransmissionIntervalSecondsInSleep);
    }
}

}}}}}
