﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/service/core/prepo_ReportFileManager.h>
#include <nn/prepo/detail/service/core/prepo_IncrementalId.h>
#include <nn/prepo/detail/service/core/prepo_UserAgreementChecker.h>
#include <nn/util/util_Endian.h>

#define KEEP_UPLOADED_FILE 0 // NOLINT(preprocessor/const)

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

ReportFileManager::ReportFileManager(const char* mountName, bool isUserAgreementCheckRequired) NN_NOEXCEPT :
    m_LastId(0),
    m_LastFileSize(ReportFileHeaderSize),
    m_LastReportCount(0),
    m_IsUserAgreementCheckRequired(isUserAgreementCheckRequired)
{
    NN_ABORT_UNLESS(nn::util::Strlcpy(m_MountName, mountName, sizeof(m_MountName)) < sizeof(m_MountName));
    NN_ABORT_UNLESS(nn::util::SNPrintf(m_DataDirectoryPath, sizeof(m_DataDirectoryPath), "%s:/data", mountName) < sizeof(m_DataDirectoryPath));
    NN_ABORT_UNLESS(nn::util::SNPrintf(m_UploadFilePath, sizeof (m_UploadFilePath), "%s:/%s", m_MountName, UploadFileName) < sizeof (m_UploadFilePath));
}

void ReportFileManager::MakeReportFilePath(char* outPath, size_t pathSize, const char* mountName, uint64_t id) NN_NOEXCEPT
{
    const int count = nn::util::SNPrintf(outPath, pathSize, "%s:/data/%016llx.msgpack", mountName, id);
    NN_UNUSED(count);
    NN_SDK_ASSERT_LESS(static_cast<size_t>(count), pathSize);
}

nn::Result ReportFileManager::WriteFile(ReportDataSummary* outSummary, void* data, size_t dataSize, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSummary);
    NN_SDK_REQUIRES_NOT_NULL(data);
    NN_SDK_REQUIRES_GREATER(dataSize, 0u);
    NN_SDK_REQUIRES_GREATER(count, 0);

    NN_DETAIL_PREPO_STORAGE_SCOPED_LOCK(m_MountName);

    NN_RESULT_DO(WriteFileImpl(outSummary, data, dataSize, count));

    NN_RESULT_DO(FileSystem::Commit(m_MountName));

    NN_RESULT_SUCCESS;
}

nn::Result ReportFileManager::WriteFileImpl(ReportDataSummary* outSummary, void* data, size_t dataSize, int count) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    NN_RESULT_DO(OpenFile(outSummary, &handle, dataSize));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    nn::Bit8 header[ReportFileHeaderSize];

    msgpack::OutputStreamParam stream = {header, ReportFileHeaderSize};
    msgpack::WriteArray16(&stream, static_cast<uint16_t>(m_LastReportCount + count));

    NN_RESULT_DO(nn::fs::WriteFile(handle, 0, header, ReportFileHeaderSize, nn::fs::WriteOption::MakeValue(0)));

    NN_RESULT_DO(nn::fs::WriteFile(handle, m_LastFileSize, data, dataSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

    m_LastFileSize += dataSize;
    m_LastReportCount += count;

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    int64_t fileSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, handle));
    NN_SDK_ASSERT_EQUAL(static_cast<size_t>(fileSize), m_LastFileSize);
    NN_SDK_ASSERT_LESS_EQUAL(static_cast<size_t>(fileSize), ReportFileSizeMax);
#endif

    NN_RESULT_SUCCESS;
}

nn::Result ReportFileManager::OpenFile(ReportDataSummary* outSummary, nn::fs::FileHandle* outHandle, size_t appendSize) NN_NOEXCEPT
{
    std::memset(outSummary, 0, sizeof(*outSummary));

    NN_RESULT_TRY(OpenExistingFile(outHandle, appendSize))
        NN_RESULT_CATCH(ResultNewReportFileRequired)
        {
            NN_RESULT_DO(OpenNewFile(outSummary, outHandle));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result ReportFileManager::OpenExistingFile(nn::fs::FileHandle* outHandle, size_t appendSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_LastId != 0, ResultNewReportFileRequired()); // BCAT の起動直後の m_LastId は 0。

    NN_RESULT_THROW_UNLESS(m_LastFileSize + appendSize <= ReportFileSizeMax, ResultNewReportFileRequired()); // 追記してもファイルサイズが 128 KB を超えない。

    char path[40];
    MakeReportFilePath(path, sizeof (path), m_MountName, m_LastId);

    NN_RESULT_TRY(nn::fs::OpenFile(outHandle, path, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            // アップロードのためにファイルが移動されたとき。
            NN_RESULT_THROW(ResultNewReportFileRequired());
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result ReportFileManager::OpenNewFile(ReportDataSummary* outSummary, nn::fs::FileHandle* outHandle) NN_NOEXCEPT
{
    uint64_t id;
    NN_RESULT_DO(IssueId(&id));

    char path[40];
    MakeReportFilePath(path, sizeof (path), m_MountName, id);

    NN_RESULT_DO(FileSystem::CreateDirectoryRecursively(path, true));

    NN_RESULT_DO(DeleteOldFileIfNotEnoughSpace(outSummary, nn::prepo::detail::ReportFileSizeMax + FileSystem::FsMetaDataBlockSize));

    NN_RESULT_DO(nn::fs::CreateFile(path, 0));

    NN_DETAIL_PREPO_INFO("[prepo] I create a new report file (%s)\n", path);

    nn::fs::FileHandle handle;
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

    *outHandle = handle;
    m_LastId = id;
    m_LastFileSize = ReportFileHeaderSize;
    m_LastReportCount = 0;

    NN_RESULT_SUCCESS;
}

nn::Result ReportFileManager::IssueId(uint64_t* outId) NN_NOEXCEPT
{
    NN_DETAIL_PREPO_STORAGE_SCOPED_LOCK("prepo-sys");

    bool isCommitRequired = false;

    NN_RESULT_DO(IncrementalId::GetInstance().Issue(&isCommitRequired, outId));

    if (isCommitRequired)
    {
        NN_RESULT_DO(FileSystem::Commit("prepo-sys"));
    }

    NN_RESULT_SUCCESS;
}

nn::Result ReportFileManager::DeleteOldestFile(size_t* outSize, int* outCount) NN_NOEXCEPT
{
    char path[40];
    NN_RESULT_DO(GetOldestReportFilePath(path, sizeof(path)));

    NN_RESULT_DO(GetReportFileStatus(outSize, outCount, path));

    NN_RESULT_DO(nn::fs::DeleteFile(path));

    NN_DETAIL_PREPO_INFO("[prepo] I delete the oldest report file (%s)\n", path);

    NN_RESULT_SUCCESS;
}

nn::Result ReportFileManager::DeleteOldFileIfNotEnoughSpace(ReportDataSummary* outSummary, int64_t requiredSize) NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        int64_t freeSpaceSize;
        NN_RESULT_DO(FileSystem::GetFreeSpaceSize(&freeSpaceSize, m_MountName));

        NN_DETAIL_PREPO_INFO("[prepo] Free space size = %lld, required size = %lld.\n", freeSpaceSize, requiredSize);

        if (requiredSize <= freeSpaceSize)
        {
            NN_RESULT_SUCCESS;
        }

        size_t deletedSize;
        int deletedCount;
        NN_RESULT_DO(DeleteOldestFile(&deletedSize, &deletedCount));

        outSummary->deletedSize += deletedSize;
        outSummary->deletedCount += deletedCount;
    }
}

nn::Result ReportFileManager::GetReportFileStatus(size_t* outSize, int* outCount, const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(path);

    NN_DETAIL_PREPO_ASSERT_STORAGE_LOCK(m_MountName);

    nn::fs::FileHandle handle;
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT { nn::fs::CloseFile(handle); };

    int64_t fileSize;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

    nn::Bit8 array16Header[3];
    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, array16Header, sizeof(array16Header)));

    msgpack::InputStreamParam stream = {array16Header, sizeof(array16Header)};
    msgpack::AnyData any = {};

    NN_RESULT_THROW_UNLESS(msgpack::ReadCurrent(&any, &stream), ResultInvalidReportData());
    NN_RESULT_THROW_UNLESS(any.type == msgpack::AnyDataType_Array, ResultInvalidReportData());
    NN_RESULT_THROW_UNLESS(stream.GetRemainSize() == 0, ResultInvalidReportData()); // Array 16 なら、残りサイズは 0。

    *outSize = static_cast<size_t>(fileSize) - sizeof(array16Header);
    *outCount = any.num;

    NN_RESULT_SUCCESS;
}

nn::Result ReportFileManager::LoadUploadFile(ReportDataSummary* outSummary, Bit8* buffer, size_t bufferSize) NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        NN_RESULT_TRY(LoadUploadFileImpl(outSummary, buffer, bufferSize))
            NN_RESULT_CATCH(ResultInvalidReportData)
            {
                NN_RESULT_DO(CleanupUploadedFile());
                continue;
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }
}

nn::Result ReportFileManager::LoadUploadFileImpl(ReportDataSummary* outSummary, Bit8* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_DETAIL_PREPO_STORAGE_SCOPED_LOCK(m_MountName);

    nn::fs::FileHandle fileHandle = {};
    NN_RESULT_TRY(nn::fs::OpenFile(&fileHandle, m_UploadFilePath, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(StashFile());
            NN_RESULT_DO(nn::fs::OpenFile(&fileHandle, m_UploadFilePath, nn::fs::OpenMode_Read));
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(fileHandle);
    };

    int64_t fileSize;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, fileHandle));

    NN_RESULT_THROW_UNLESS(static_cast<size_t>(fileSize) <= bufferSize, ResultOutOfResource());

    size_t readSize;
    NN_RESULT_DO(nn::fs::ReadFile(&readSize, fileHandle, 0, buffer, bufferSize));

    NN_SDK_ASSERT_EQUAL(readSize, static_cast<size_t>(fileSize));

    NN_RESULT_THROW_UNLESS(DeleteDisagreedReport(outSummary, buffer, readSize),
        ResultInvalidReportData());

    NN_RESULT_SUCCESS;
}

nn::Result ReportFileManager::StashFile() NN_NOEXCEPT
{
    char path[64];
    NN_RESULT_DO(GetOldestReportFilePath(path, sizeof(path)));

    NN_RESULT_DO(nn::fs::RenameFile(path, m_UploadFilePath));

    NN_DETAIL_PREPO_INFO("[prepo] Stash a file to upload (%s)\n", path);

    NN_RESULT_SUCCESS;
}

nn::Result ReportFileManager::CleanupUploadedFile() NN_NOEXCEPT
{
    NN_DETAIL_PREPO_STORAGE_SCOPED_LOCK(m_MountName);

#if KEEP_UPLOADED_FILE == 0
    NN_RESULT_TRY(nn::fs::DeleteFile(m_UploadFilePath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            // LoadFile 後に、ClearStorage されているとき。
        }
    NN_RESULT_END_TRY;
#else
    while (NN_STATIC_CONDITION(true))
    {
        NN_FUNCTION_LOCAL_STATIC(uint64_t, s_Counter, = 0);

        char path[sizeof("prepo:/posted_0123456789ABCEDF.msgpack")];
        nn::util::SNPrintf(path, sizeof(path), "prepo:/posted_%016llx.msgpack", s_Counter++);

        NN_RESULT_TRY(nn::fs::RenameFile(m_UploadFilePath, path))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                // LoadFile 後に、ClearStorage されているとき。
            }
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
            {
                continue;
            }
        NN_RESULT_END_TRY;

        break;
    }
#endif

    NN_RESULT_DO(FileSystem::Commit(m_MountName));

    NN_RESULT_SUCCESS;
}

bool ReportFileManager::IsUserAgreementCheckRequired() const NN_NOEXCEPT
{
    return m_IsUserAgreementCheckRequired;
}

bool ReportFileManager::DeleteDisagreedReport(ReportDataSummary* outSummary, Bit8* data, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSummary);
    NN_SDK_REQUIRES_NOT_NULL(data);
    NN_SDK_REQUIRES_GREATER(dataSize, 0u);

    msgpack::InputStreamParam stream = {data, dataSize};
    msgpack::AnyData any = {};

    NN_DETAIL_PREPO_RETURN_IF_FALSE(msgpack::ReadCurrent(&any, &stream));

    NN_DETAIL_PREPO_RETURN_IF_FALSE(any.type == msgpack::AnyDataType_Array);

    const int reportCount = static_cast<int>(any.num);
    int deletedReportCount = 0;
    size_t deletedReportSize = 0u;

    for (int i = 0; i < reportCount; i++)
    {
        const size_t reportBegin = stream.position;

        NN_DETAIL_PREPO_RETURN_IF_FALSE(msgpack::ReadCurrent(&any, &stream));
        NN_DETAIL_PREPO_RETURN_IF_FALSE(any.type == msgpack::AnyDataType_Map);
        NN_DETAIL_PREPO_RETURN_IF_FALSE(any.num == 2);

        NN_DETAIL_PREPO_RETURN_IF_FALSE(msgpack::ReadCurrent(&any, &stream));
        NN_DETAIL_PREPO_RETURN_IF_FALSE(any.type == msgpack::AnyDataType_String);

        const char* pSysInfoKey = reinterpret_cast<const char*>(&stream.data[any.string.position]);
        NN_DETAIL_PREPO_RETURN_IF_FALSE(nn::util::Strncmp(pSysInfoKey, "sys_info", static_cast<int>(any.string.length)) == 0);

        NN_DETAIL_PREPO_RETURN_IF_FALSE(msgpack::ReadCurrent(&any, &stream));
        NN_DETAIL_PREPO_RETURN_IF_FALSE(any.type == msgpack::AnyDataType_Map);

        const int sysInfoCount = static_cast<int>(any.num);
        bool isDeleteRequried = true;

        for (int j = 0; j < sysInfoCount; j++)
        {
            // キー
            NN_DETAIL_PREPO_RETURN_IF_FALSE(msgpack::ReadCurrent(&any, &stream));
            NN_DETAIL_PREPO_RETURN_IF_FALSE(any.type == msgpack::AnyDataType_String);

            const char* pKey = reinterpret_cast<const char*>(&stream.data[any.string.position]);
            const size_t keyLength = any.string.length;

            // バリュー
            NN_DETAIL_PREPO_RETURN_IF_FALSE(msgpack::ReadCurrent(&any, &stream));

            if (nn::util::Strncmp(pKey, "nsa_id", static_cast<int>(keyLength)) == 0)
            {
                if (any.type == msgpack::AnyDataType_Extension)
                {
                    NN_DETAIL_PREPO_RETURN_IF_FALSE(any.extension.type  == MsgpackExtensionType_Any64BitId);
                    NN_DETAIL_PREPO_RETURN_IF_FALSE(any.extension.length == 8);

                    nn::account::NetworkServiceAccountId id = nn::util::LoadBigEndian(reinterpret_cast<const decltype (id) *>(&stream.data[any.extension.position]));

                    isDeleteRequried = IsUserAgreementCheckRequired()
                        ? UserAgreementChecker::GetInstance().IsUserAgreedWithAnalystics(id).IsFailure()
                        : UserAgreementChecker::GetInstance().IsUserAgreedWithNaEula(id).IsFailure();
                }
                else if (any.type == msgpack::AnyDataType_Nil)
                {
                    isDeleteRequried = IsUserAgreementCheckRequired()
                        ? UserAgreementChecker::GetInstance().IsAnyoneAgreedWithAnalystics().IsFailure()
                        : UserAgreementChecker::GetInstance().IsAnyoneAgreedWithNaEula().IsFailure();
                }
                else
                {
                    NN_RESULT_THROW(ResultInvalidReportData());
                }
            }
        }

        NN_DETAIL_PREPO_RETURN_IF_FALSE(msgpack::ReadCurrent(&any, &stream));
        NN_DETAIL_PREPO_RETURN_IF_FALSE(any.type == msgpack::AnyDataType_String);

        const char* pDataKey = reinterpret_cast<const char*>(&stream.data[any.string.position]);
        NN_DETAIL_PREPO_RETURN_IF_FALSE(nn::util::Strncmp(pDataKey, "data", static_cast<int>(any.string.length)) == 0);

        NN_DETAIL_PREPO_RETURN_IF_FALSE(msgpack::ReadCurrent(&any, &stream));
        NN_DETAIL_PREPO_RETURN_IF_FALSE(any.type == msgpack::AnyDataType_Map);

        const int dataCount = static_cast<int>(any.num);

        for (int j = 0; j < dataCount; j++)
        {
            // キー
            NN_DETAIL_PREPO_RETURN_IF_FALSE(msgpack::ReadCurrent(&any, &stream));
            NN_DETAIL_PREPO_RETURN_IF_FALSE(any.type == msgpack::AnyDataType_String);

            // バリュー
            NN_DETAIL_PREPO_RETURN_IF_FALSE(msgpack::ReadCurrent(&any, &stream));
        }

        const size_t reportEnd = stream.position;

        if (isDeleteRequried)
        {
            const size_t deleteSize = reportEnd - reportBegin;
            const size_t moveSize = stream.GetRemainSize();

            std::memmove(const_cast<nn::Bit8*>(&stream.data[reportBegin]), &stream.data[reportEnd], moveSize);

            stream.position = reportBegin;
            stream.size -= deleteSize;

            deletedReportCount++;
            deletedReportSize += deleteSize;
            msgpack::OutputStreamParam outStream = {data, stream.size};
            NN_ABORT_UNLESS(msgpack::WriteArray16(&outStream, static_cast<uint16_t>(reportCount - deletedReportCount)));
        }
    }
    NN_SDK_ASSERT_EQUAL(stream.position, stream.size);

    const size_t array16HeaderSize = 3;

    outSummary->dataSize = stream.size;
    outSummary->restSize = stream.size - array16HeaderSize;
    outSummary->restCount =  reportCount - deletedReportCount;
    outSummary->deletedSize = deletedReportSize;
    outSummary->deletedCount = deletedReportCount;

    return true;
} // NOLINT(impl/function_size)

nn::Result ReportFileManager::GetOldestReportFilePath(char* outPath, size_t pathSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outPath);

    NN_DETAIL_PREPO_ASSERT_STORAGE_LOCK(m_MountName);

    nn::fs::DirectoryHandle directoryHandle = {};

    NN_RESULT_TRY(nn::fs::OpenDirectory(&directoryHandle, m_DataDirectoryPath, nn::fs::OpenDirectoryMode_File))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_THROW(ResultReportFileNotFound());
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(directoryHandle);
    };

    uint64_t minId;

    NN_RESULT_DO(ReadReportFileId(&minId, directoryHandle));

    while (NN_STATIC_CONDITION(true))
    {
        uint64_t id;

        NN_RESULT_TRY(ReadReportFileId(&id, directoryHandle))
            NN_RESULT_CATCH(ResultReportFileNotFound)
            {
                break;
            }
        NN_RESULT_END_TRY;

        minId = std::min(id, minId);
    }

    MakeReportFilePath(outPath, pathSize, m_MountName, minId);

    NN_RESULT_SUCCESS;
}

nn::Result ReportFileManager::ReadReportFileId(uint64_t* outId, nn::fs::DirectoryHandle handle) NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        nn::fs::DirectoryEntry entry; // スタックの消費が大きいので注意。

        int64_t entryCount = 0;

        NN_RESULT_DO(nn::fs::ReadDirectory(&entryCount, &entry, handle, 1));

        NN_RESULT_THROW_UNLESS(entryCount == 1, ResultReportFileNotFound());

        unsigned long long id; // %016llx に対して uint64_t を与えると、警告が出るため。

        int result = std::sscanf(entry.name, "%016llx.msgpack", &id);

        if (result != 1)
        {
            NN_DETAIL_PREPO_WARN(
                NN_DETAIL_PREPO_STRING_WARN("Invalid name report file found: %s\n"),
                entry.name);
            continue;
        }

        *outId = id;

        NN_RESULT_SUCCESS;
    }
}

}}}}}
