﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/service/core/prepo_ReportBuffer.h>
#include <nn/prepo/detail/service/core/prepo_StatisticsManager.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

ReportBuffer::ReportBuffer() NN_NOEXCEPT :
    m_pForegroundBuffer(&m_Buffers[0]),
    m_pBackgroundBuffer(&m_Buffers[1]),
    m_AddEvent(nullptr)
{
    for (auto& buffer : m_Buffers)
    {
        buffer.filledSize = 0;
        buffer.count = 0;
    }
}

void ReportBuffer::RegisterAddEvent(nn::os::EventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(event);

    NN_UTIL_LOCK_GUARD(m_AddEventMutex);

    NN_SDK_ASSERT(!m_AddEvent);

    m_AddEvent = event;
}

nn::os::EventType* ReportBuffer::UnregisterAddEvent() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_AddEventMutex);

    NN_SDK_ASSERT(m_AddEvent);

    auto event = m_AddEvent;

    m_AddEvent = nullptr;

    return event;
}

nn::Result ReportBuffer::Add(ReportCategory category, const void* data, size_t dataSize, const void* sysInfo, size_t sysInfoSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(data);
    NN_SDK_REQUIRES_GREATER(dataSize, 0u);
    NN_SDK_REQUIRES_NOT_NULL(sysInfo);
    NN_SDK_REQUIRES_GREATER(sysInfoSize, 0u);

    NN_UTIL_LOCK_GUARD(m_ForegroundBufferMutex);

    msgpack::OutputStreamParam stream = {
        m_pForegroundBuffer->GetWritePointer(), m_pForegroundBuffer->GetRemainSize()};

    // FIXMAP + (FIXSTR + "sys_info" + sysInfoSize) + (FIXSTR + "data" + dataSize)
    const size_t totalWriteSize = 1 + (1 + 8 + sysInfoSize) + (1 + 4 + dataSize);

    if (!(true
        && msgpack::WriteSignedIntegerAutoSize(&stream, static_cast<int64_t>(category))
        && msgpack::WriteSignedIntegerAutoSize(&stream, static_cast<int64_t>(totalWriteSize))
        && totalWriteSize <= stream.GetRemainSize()))
    {
        StatisticsManager::GetInstance().AddLostByBufferShortage(category, totalWriteSize, 1);
        NN_RESULT_THROW(ResultOutOfResource());
    }

    msgpack::WriteMapAutoSize(&stream, 2);

    msgpack::WriteString(&stream, "sys_info", 8);
    std::memcpy(&(stream.buffer[stream.position]), sysInfo, sysInfoSize);
    stream.position += sysInfoSize;

    msgpack::WriteString(&stream, "data", 4);
    std::memcpy(&(stream.buffer[stream.position]), data, dataSize);
    stream.position += dataSize;

    // NN_DETAIL_PREPO_INFO("[prepo] Add a report data to the buffer (%zu Bytes)\n", stream.position);

    m_pForegroundBuffer->filledSize += stream.position;
    m_pForegroundBuffer->count++;

    {
        NN_UTIL_LOCK_GUARD(m_AddEventMutex);
        if (m_AddEvent)
        {
            nn::os::SignalEvent(m_AddEvent);
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result ReportBuffer::Flush(FlushFunction flush) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_BackgroundBufferMutex);

    while (NN_STATIC_CONDITION(true))
    {
        {
            NN_UTIL_LOCK_GUARD(m_ForegroundBufferMutex);
            if (m_pForegroundBuffer->count == 0)
            {
                NN_RESULT_SUCCESS;
            }
            std::swap(m_pForegroundBuffer, m_pBackgroundBuffer);
        }

        for (int i = 0; i < ReportCategory_Count; i++)
        {
            const auto target = static_cast<ReportCategory>(i);

            m_PickupBuffer.filledSize = 0;
            m_PickupBuffer.count = 0;

            msgpack::InputStreamParam stream = { m_pBackgroundBuffer->buffer, m_pBackgroundBuffer->filledSize };

            while (stream.GetRemainSize() > 0)
            {
                msgpack::AnyData any = {};

                msgpack::ReadCurrent(&any, &stream);
                NN_SDK_ASSERT_EQUAL(any.type, msgpack::AnyDataType_SInteger);
                const auto category = static_cast<ReportCategory>(any.si);

                msgpack::ReadCurrent(&any, &stream);
                NN_SDK_ASSERT_EQUAL(any.type, msgpack::AnyDataType_SInteger);
                const auto size = static_cast<size_t>(any.si);

                if (category == target)
                {
                    NN_SDK_ASSERT_LESS_EQUAL(size, m_PickupBuffer.GetRemainSize());
                    std::memcpy(m_PickupBuffer.GetWritePointer(), &(stream.data[stream.position]), size);
                    m_PickupBuffer.filledSize += size;
                    m_PickupBuffer.count++;
                }

                stream.position += size;
            }
            NN_SDK_ASSERT_EQUAL(stream.GetRemainSize(), 0u);

            if (m_PickupBuffer.count > 0)
            {
                NN_RESULT_DO(flush(target, m_PickupBuffer.buffer, m_PickupBuffer.filledSize, m_PickupBuffer.count));
            }
        }

        m_pBackgroundBuffer->filledSize = 0;
        m_pBackgroundBuffer->count = 0;
    }
}

}}}}}
