﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/service/core/prepo_FileSystem.h>
#include <nn/prepo/detail/service/prepo_Common.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

void FileSystem::EnableTestMode() NN_NOEXCEPT
{
    Storage::EnableTestMode();
}

nn::Result FileSystem::Mount(const char* mountName) NN_NOEXCEPT
{
    return FindStorage(mountName).Mount();
}

void FileSystem::Unmount(const char* mountName) NN_NOEXCEPT
{
    nn::fs::Unmount(mountName);
}

void FileSystem::Lock(const char* mountName) NN_NOEXCEPT
{
    FindStorage(mountName).Lock();
}

void FileSystem::Unlock(const char* mountName) NN_NOEXCEPT
{
    FindStorage(mountName).Unlock();
}

int64_t FileSystem::GetStorageUsage(const char* mountName) NN_NOEXCEPT
{
    return FindStorage(mountName).GetUsage();
}

nn::Result FileSystem::GetFreeSpaceSize(int64_t* outSize, const char* mountName) NN_NOEXCEPT
{
    *outSize = FindStorage(mountName).GetFreeSpaceSize();
    NN_RESULT_SUCCESS;
}

int64_t FileSystem::GetTotalSpaceSize(const char* mountName) NN_NOEXCEPT
{
    return FindStorage(mountName).GetTotalSpaceSize();
}

bool FileSystem::IsLockedByCurrentThread(const char* mountName) NN_NOEXCEPT
{
    return FindStorage(mountName).IsLockedByCurrentThread();
}

void FileSystem::MountAll() NN_NOEXCEPT
{
    for (auto&& storage : GetStorages())
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Mount());
    }
}

void FileSystem::UnmountAll() NN_NOEXCEPT
{
    for (auto&& storage : GetStorages())
    {
        storage.Unmount();
    }
}

nn::Result FileSystem::Commit(const char* mountName) NN_NOEXCEPT
{
    return FindStorage(mountName).Commit();
}

nn::Result FileSystem::Rollback(const char* mountName) NN_NOEXCEPT
{
    return FindStorage(mountName).Rollback();
}

nn::Result FileSystem::CreateDirectoryRecursively(const char* path, bool isFilePath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    char currentPath[nn::fs::EntryNameLengthMax + 1] = {};
    bool root = true;

    char c = path[0];

    for (int i = 0; path[i] != '\0'; i++)
    {
        c = path[i];

        if (c == '/')
        {
            if (!root)
            {
                NN_RESULT_TRY(nn::fs::CreateDirectory(currentPath))
                    NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                    {
                    }
                NN_RESULT_END_TRY;
            }
            else
            {
                root = false;
            }
        }

        currentPath[i] = c;
    }

    if (!root && !isFilePath && c != '/')
    {
        NN_RESULT_TRY(nn::fs::CreateDirectory(currentPath))
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
            {
            }
        NN_RESULT_END_TRY;
    }

    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    return nn::fs::DeleteDirectoryRecursively(path);
}

void FileSystem::ClearStorage(const char* mountName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(mountName);

    NN_DETAIL_PREPO_STORAGE_SCOPED_LOCK(mountName);

    auto&& storage = FindStorage(mountName);

    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Clear());

    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Commit());
}

Storage& FileSystem::FindStorage(const char* mountName) NN_NOEXCEPT
{
    auto&& storages = GetStorages();
    auto it = std::find_if(storages.begin(), storages.end(), [&mountName](auto&& storage) {
        return nn::util::Strncmp(mountName, storage.GetMountName(), nn::fs::MountNameLengthMax + 1) == 0; });
    NN_ABORT_UNLESS_NOT_EQUAL(it, storages.end());
    return *it;
}

}}}}}
