﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/msgpack/prepo_MessagePack.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_Endian.h>

namespace nn { namespace prepo { namespace detail { namespace msgpack {

namespace
{
    bool ReadArray(AnyData* outData, InputStreamParam* stream, nn::Bit8 dataType) NN_NOEXCEPT
    {
        outData->type = AnyDataType_Array;

        switch (dataType)
        {
        case DataType_Array16:
            {
                if (stream->GetRemainSize() < 2)
                {
                    return false;
                }
                outData->num = nn::util::LoadBigEndian(reinterpret_cast<const uint16_t*>(&stream->data[stream->position]));
                stream->position += 2;
            }
            break;
        case DataType_Array32:
            {
                if (stream->GetRemainSize() < 4)
                {
                    return false;
                }
                outData->num = nn::util::LoadBigEndian(reinterpret_cast<const uint32_t*>(&stream->data[stream->position]));
                stream->position += 4;
            }
            break;
        default:
            {
                outData->num = GetFixArrayLength(dataType);
            }
            break;
        }

        return true;
    }

    bool ReadMap(AnyData* outData, InputStreamParam* stream, nn::Bit8 dataType) NN_NOEXCEPT
    {
        outData->type = AnyDataType_Map;

        switch (dataType)
        {
        case DataType_Map16:
            {
                if (stream->GetRemainSize() < 2)
                {
                    return false;
                }
                outData->num = nn::util::LoadBigEndian(reinterpret_cast<const uint16_t*>(&stream->data[stream->position]));
                stream->position += 2;
            }
            break;
        case DataType_Map32:
            {
                if (stream->GetRemainSize() < 4)
                {
                    return false;
                }
                outData->num = nn::util::LoadBigEndian(reinterpret_cast<const uint32_t*>(&stream->data[stream->position]));
                stream->position += 4;
            }
            break;
        default:
            {
                outData->num = GetFixMapLength(dataType);
            }
            break;
        }

        return true;
    }

    bool ReadSignedInteger(AnyData* outData, InputStreamParam* stream, nn::Bit8 dataType) NN_NOEXCEPT
    {
        outData->type = AnyDataType_SInteger;

        switch (dataType)
        {
        case DataType_S8:
            {
                if (stream->GetRemainSize() < 1)
                {
                    return false;
                }
                outData->si = static_cast<int8_t>(stream->data[stream->position++]);
            }
            break;
        case DataType_S16:
            {
                if (stream->GetRemainSize() < 2)
                {
                    return false;
                }
                outData->si = nn::util::LoadBigEndian(reinterpret_cast<const int16_t*>(&stream->data[stream->position]));
                stream->position += 2;
            }
            break;
        case DataType_S32:
            {
                if (stream->GetRemainSize() < 4)
                {
                    return false;
                }
                outData->si = nn::util::LoadBigEndian(reinterpret_cast<const int32_t*>(&stream->data[stream->position]));
                stream->position += 4;
            }
            break;
        case DataType_S64:
            {
                if (stream->GetRemainSize() < 8)
                {
                    return false;
                }
                outData->si = nn::util::LoadBigEndian(reinterpret_cast<const int64_t*>(&stream->data[stream->position]));
                stream->position += 8;
            }
            break;
        default:
            {
                outData->si = static_cast<int8_t>(dataType);
            }
        }

        return true;
    }

    bool ReadUnsignedInteger(AnyData* outData, InputStreamParam* stream, nn::Bit8 dataType) NN_NOEXCEPT
    {
        outData->type = AnyDataType_UInteger;

        switch (dataType)
        {
        case DataType_U8:
            {
                if (stream->GetRemainSize() < 1)
                {
                    return false;
                }
                outData->ui = stream->data[stream->position++];
            }
            break;
        case DataType_U16:
            {
                if (stream->GetRemainSize() < 2)
                {
                    return false;
                }
                outData->ui = nn::util::LoadBigEndian(reinterpret_cast<const uint16_t*>(&stream->data[stream->position]));
                stream->position += 2;
            }
            break;
        case DataType_U32:
            {
                if (stream->GetRemainSize() < 4)
                {
                    return false;
                }
                outData->ui = nn::util::LoadBigEndian(reinterpret_cast<const uint32_t*>(&stream->data[stream->position]));
                stream->position += 4;
            }
            break;
        case DataType_U64:
            {
                if (stream->GetRemainSize() < 8)
                {
                    return false;
                }
                outData->ui = nn::util::LoadBigEndian(reinterpret_cast<const uint64_t*>(&stream->data[stream->position]));
                stream->position += 8;
            }
            break;
        default:
            {
                return false;
            }
        }

        return true;
    }

    bool ReadString(AnyData* outData, InputStreamParam* stream, nn::Bit8 dataType) NN_NOEXCEPT
    {
        outData->type = AnyDataType_String;

        switch (dataType)
        {
        case DataType_String8:
            {
                if (stream->GetRemainSize() < 1)
                {
                    return false;
                }
                outData->string.length = stream->data[stream->position++];
            }
            break;
        case DataType_String16:
            {
                if (stream->GetRemainSize() < 2)
                {
                    return false;
                }
                outData->string.length = nn::util::LoadBigEndian(reinterpret_cast<const uint16_t*>(&stream->data[stream->position]));
                stream->position += 2;
            }
            break;
        case DataType_String32:
            {
                if (stream->GetRemainSize() < 4)
                {
                    return false;
                }
                outData->string.length = nn::util::LoadBigEndian(reinterpret_cast<const uint32_t*>(&stream->data[stream->position]));
                stream->position += 4;
            }
            break;
        default:
            {
                outData->string.length = GetFixStringLength(dataType);
            }
            break;
        }

        if (stream->GetRemainSize() < outData->string.length)
        {
            return false;
        }
        outData->string.position = stream->position;

        stream->position += outData->string.length;

        return true;
    }

    bool ReadBinary(AnyData* outData, InputStreamParam* stream, nn::Bit8 dataType) NN_NOEXCEPT
    {
        outData->type = AnyDataType_Binary;

        switch (dataType)
        {
        case DataType_Binary8:
            {
                if (stream->GetRemainSize() < 1)
                {
                    return false;
                }
                outData->binary.length = stream->data[stream->position++];
            }
            break;
        case DataType_Binary16:
            {
                if (stream->GetRemainSize() < 2)
                {
                    return false;
                }
                outData->binary.length = nn::util::LoadBigEndian(reinterpret_cast<const uint16_t*>(&stream->data[stream->position]));
                stream->position += 2;
            }
            break;
        case DataType_Binary32:
            {
                if (stream->GetRemainSize() < 4)
                {
                    return false;
                }
                outData->binary.length = nn::util::LoadBigEndian(reinterpret_cast<const uint32_t*>(&stream->data[stream->position]));
                stream->position += 4;
            }
            break;
        default:
            {
                return false;
            }
        }

        if (stream->GetRemainSize() < outData->binary.length)
        {
            return false;
        }
        outData->binary.position = stream->position;

        stream->position += outData->binary.length;

        return true;
    }

    bool ReadExtension(AnyData* outData, InputStreamParam* stream, nn::Bit8 dataType) NN_NOEXCEPT
    {
        outData->type = AnyDataType_Extension;

        switch (dataType)
        {
        case DataType_Extension8:
            {
                if (stream->GetRemainSize() < 1)
                {
                    return false;
                }
                outData->extension.length = stream->data[stream->position++];
            }
            break;
        case DataType_Extension16:
            {
                if (stream->GetRemainSize() < 2)
                {
                    return false;
                }
                outData->extension.length = nn::util::LoadBigEndian(reinterpret_cast<const uint16_t*>(&stream->data[stream->position]));
                stream->position += 2;
            }
            break;
        case DataType_Extension32:
            {
                if (stream->GetRemainSize() < 4)
                {
                    return false;
                }
                outData->extension.length = nn::util::LoadBigEndian(reinterpret_cast<const uint32_t*>(&stream->data[stream->position]));
                stream->position += 4;
            }
            break;
        case DataType_FixExtension1:
            {
                outData->extension.length = 1;
            }
            break;
        case DataType_FixExtension2:
            {
                outData->extension.length = 2;
            }
            break;
        case DataType_FixExtension4:
            {
                outData->extension.length = 4;
            }
            break;
        case DataType_FixExtension8:
            {
                outData->extension.length = 8;
            }
            break;
        case DataType_FixExtension16:
            {
                outData->extension.length = 16;
            }
            break;
        default:
            {
                return false;
            }
        }

        if (stream->GetRemainSize() < 1)
        {
            return false;
        }
        outData->extension.type = stream->data[stream->position++];

        if (stream->GetRemainSize() < outData->extension.length)
        {
            return false;
        }
        outData->extension.position = stream->position;

        stream->position += outData->extension.length;

        return true;
    }
}

bool ReadCurrent(AnyData* outData, InputStreamParam* stream) NN_NOEXCEPT
{
    Bit8 dataType = stream->data[stream->position];

    if (stream->GetRemainSize() < 1)
    {
        return false;
    }
    stream->position++;

    if (IsString(dataType))
    {
        return ReadString(outData, stream, dataType);
    }

    if (IsArray(dataType))
    {
        return ReadArray(outData, stream, dataType);
    }
    if (IsMap(dataType))
    {
        return ReadMap(outData, stream, dataType);
    }

    if (IsSignedInteger(dataType))
    {
        return ReadSignedInteger(outData, stream, dataType);
    }
    if (IsUnsignedInteger(dataType))
    {
        return ReadUnsignedInteger(outData, stream, dataType);
    }

    if (dataType == DataType_Float64)
    {
        outData->type = AnyDataType_Float64;

        if (stream->GetRemainSize() < 8)
        {
            return false;
        }
        outData->f64 = nn::util::LoadBigEndian(reinterpret_cast<const double*>(&stream->data[stream->position]));
        stream->position += 8;

        return true;
    }
    if (dataType == DataType_Float32)
    {
        outData->type = AnyDataType_Float32;

        if (stream->GetRemainSize() < 4)
        {
            return false;
        }
        outData->f32 = nn::util::LoadBigEndian(reinterpret_cast<const float*>(&stream->data[stream->position]));
        stream->position += 4;

        return true;
    }

    if (dataType == DataType_Nil)
    {
        outData->type = AnyDataType_Nil;
        return true;
    }
    if (dataType == DataType_False)
    {
        outData->type = AnyDataType_Boolean;
        outData->boolean = false;
        return true;
    }
    if (dataType == DataType_True)
    {
        outData->type = AnyDataType_Boolean;
        outData->boolean = true;
        return true;
    }

    if (IsBinary(dataType))
    {
        return ReadBinary(outData, stream, dataType);
    }
    if (IsExtension(dataType))
    {
        return ReadExtension(outData, stream, dataType);
    }

    return false;
}

bool WriteFixMap(OutputStreamParam* stream, uint8_t numObjects) NN_NOEXCEPT
{
    if (numObjects > 15)
    {
        return false;
    }
    if (stream->GetRemainSize() < 1)
    {
        return false;
    }

    stream->buffer[stream->position++] = MakeFixMap(numObjects);

    return true;
}

bool WriteMap16(OutputStreamParam* stream, uint16_t numObjects) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 3)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_Map16;

    nn::util::StoreBigEndian(reinterpret_cast<uint16_t*>(&stream->buffer[stream->position]), numObjects);
    stream->position += 2;

    return true;
}

bool WriteMap32(OutputStreamParam* stream, uint32_t numObjects) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 5)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_Map32;

    nn::util::StoreBigEndian(reinterpret_cast<uint32_t*>(&stream->buffer[stream->position]), numObjects);
    stream->position += 4;

    return true;
}

bool WriteMapAutoSize(OutputStreamParam* stream, uint32_t numObjects) NN_NOEXCEPT
{
    if (numObjects <= 15)
    {
        return WriteFixMap(stream, static_cast<uint8_t>(numObjects));
    }
    if (numObjects <= 0xFFFF)
    {
        return WriteMap16(stream, static_cast<uint16_t>(numObjects));
    }

    return WriteMap32(stream, numObjects);
}

bool WriteFixArray(OutputStreamParam* stream, uint8_t numElements) NN_NOEXCEPT
{
    if (numElements > 15)
    {
        return false;
    }
    if (stream->GetRemainSize() < 1)
    {
        return false;
    }

    stream->buffer[stream->position++] = MakeFixArray(numElements);

    return true;
}

bool WriteArray16(OutputStreamParam* stream, uint16_t numElements) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 3)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_Array16;

    nn::util::StoreBigEndian(reinterpret_cast<uint16_t*>(&stream->buffer[stream->position]), numElements);
    stream->position += 2;

    return true;
}

bool WriteArray32(OutputStreamParam* stream, uint32_t numElements) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 5)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_Array32;

    nn::util::StoreBigEndian(reinterpret_cast<uint32_t*>(&stream->buffer[stream->position]), numElements);
    stream->position += 4;

    return true;
}

bool WriteArrayAutoSize(OutputStreamParam* stream, uint32_t numElements) NN_NOEXCEPT
{
    if (numElements <= 15)
    {
        return WriteFixArray(stream, static_cast<uint8_t>(numElements));
    }
    if (numElements <= 0xFFFF)
    {
        return WriteArray16(stream, static_cast<uint16_t>(numElements));
    }

    return WriteArray32(stream, numElements);
}

bool WriteString(OutputStreamParam* stream, const char* string, uint32_t length) NN_NOEXCEPT
{
    if (length <= 31)
    {
        if (stream->GetRemainSize() < 1 + static_cast<size_t>(length))
        {
            return false;
        }

        stream->buffer[stream->position++] = MakeFixString(static_cast<uint8_t>(length));
    }
    else if (length <= 0xFF)
    {
        if (stream->GetRemainSize() < 2 + static_cast<size_t>(length))
        {
            return false;
        }

        stream->buffer[stream->position++] = DataType_String8;
        stream->buffer[stream->position++] = static_cast<uint8_t>(length);
    }
    else if (length <= 0xFFFF)
    {
        if (stream->GetRemainSize() < 3 + static_cast<size_t>(length))
        {
            return false;
        }

        stream->buffer[stream->position++] = DataType_String16;

        nn::util::StoreBigEndian(reinterpret_cast<uint16_t*>(&stream->buffer[stream->position]), static_cast<uint16_t>(length));
        stream->position += 2;
    }
    else
    {
        if (stream->GetRemainSize() < 5 + static_cast<size_t>(length))
        {
            return false;
        }

        stream->buffer[stream->position++] = DataType_String32;

        nn::util::StoreBigEndian(reinterpret_cast<uint32_t*>(&stream->buffer[stream->position]), length);
        stream->position += 4;
    }

    if (length > 0)
    {
        std::memcpy(&stream->buffer[stream->position], string, length);
        stream->position += length;
    }

    return true;
}

bool WriteBinary(OutputStreamParam* stream, const void* binary, uint32_t size) NN_NOEXCEPT
{
    if (size <= 0xFF)
    {
        if (stream->GetRemainSize() < 2 + static_cast<size_t>(size))
        {
            return false;
        }

        stream->buffer[stream->position++] = DataType_Binary8;
        stream->buffer[stream->position++] = static_cast<uint8_t>(size);
    }
    else if (size <= 0xFFFF)
    {
        if (stream->GetRemainSize() < 3 + static_cast<size_t>(size))
        {
            return false;
        }

        stream->buffer[stream->position++] = DataType_Binary16;

        nn::util::StoreBigEndian(reinterpret_cast<uint16_t*>(&stream->buffer[stream->position]), static_cast<uint16_t>(size));
        stream->position += 2;
    }
    else
    {
        if (stream->GetRemainSize() < 5 + static_cast<size_t>(size))
        {
            return false;
        }

        stream->buffer[stream->position++] = DataType_Binary32;

        nn::util::StoreBigEndian(reinterpret_cast<uint32_t*>(&stream->buffer[stream->position]), size);
        stream->position += 4;
    }

    if (size > 0)
    {
        std::memcpy(&stream->buffer[stream->position], binary, size);
        stream->position += size;
    }

    return true;
}

bool WriteExtension(OutputStreamParam* stream, Bit8 type, const void* extension, uint32_t size) NN_NOEXCEPT
{
    if (size == 1)
    {
        if (stream->GetRemainSize() < 3)
        {
            return false;
        }

        stream->buffer[stream->position++] = DataType_FixExtension1;
    }
    else if (size == 2)
    {
        if (stream->GetRemainSize() < 4)
        {
            return false;
        }

        stream->buffer[stream->position++] = DataType_FixExtension2;
    }
    else if (size == 4)
    {
        if (stream->GetRemainSize() < 6)
        {
            return false;
        }

        stream->buffer[stream->position++] = DataType_FixExtension4;
    }
    else if (size == 8)
    {
        if (stream->GetRemainSize() < 10)
        {
            return false;
        }

        stream->buffer[stream->position++] = DataType_FixExtension8;
    }
    else if (size == 16)
    {
        if (stream->GetRemainSize() < 18)
        {
            return false;
        }

        stream->buffer[stream->position++] = DataType_FixExtension16;
    }
    else if (size <= 0xFF)
    {
        if (stream->GetRemainSize() < 3 + static_cast<size_t>(size))
        {
            return false;
        }

        stream->buffer[stream->position++] = DataType_Extension8;
        stream->buffer[stream->position++] = static_cast<uint8_t>(size);
    }
    else if (size <= 0xFFFF)
    {
        if (stream->GetRemainSize() < 4 + static_cast<size_t>(size))
        {
            return false;
        }

        stream->buffer[stream->position++] = DataType_Extension16;

        nn::util::StoreBigEndian(reinterpret_cast<uint16_t*>(&stream->buffer[stream->position]), static_cast<uint16_t>(size));
        stream->position += 2;
    }
    else
    {
        if (stream->GetRemainSize() < 6 + static_cast<size_t>(size))
        {
            return false;
        }

        stream->buffer[stream->position++] = DataType_Extension32;

        nn::util::StoreBigEndian(reinterpret_cast<uint32_t*>(&stream->buffer[stream->position]), size);
        stream->position += 4;
    }

    stream->buffer[stream->position++] = type;

    if (size > 0)
    {
        std::memcpy(&stream->buffer[stream->position], extension, size);
        stream->position += size;
    }

    return true;
}

bool WriteNil(OutputStreamParam* stream) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 1)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_Nil;

    return true;
}

bool WriteBool(OutputStreamParam* stream, bool value) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 1)
    {
        return false;
    }

    stream->buffer[stream->position++] = value ? DataType_True : DataType_False;

    return true;
}

bool WriteFixNum(OutputStreamParam* stream, int8_t value) NN_NOEXCEPT
{
    if (value < -31)
    {
        return false;
    }
    if (stream->GetRemainSize() < 1)
    {
        return false;
    }

    stream->buffer[stream->position++] = static_cast<Bit8>(value);

    return true;
}

bool WriteSignedInteger8(OutputStreamParam* stream, int8_t value) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 2)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_S8;
    stream->buffer[stream->position++] = static_cast<Bit8>(value);

    return true;
}

bool WriteSignedInteger16(OutputStreamParam* stream, int16_t value) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 3)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_S16;

    nn::util::StoreBigEndian(reinterpret_cast<int16_t*>(&stream->buffer[stream->position]), value);
    stream->position += 2;

    return true;
}

bool WriteSignedInteger32(OutputStreamParam* stream, int32_t value) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 5)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_S32;

    nn::util::StoreBigEndian(reinterpret_cast<int32_t*>(&stream->buffer[stream->position]), value);
    stream->position += 4;

    return true;
}

bool WriteSignedInteger64(OutputStreamParam* stream, int64_t value) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 9)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_S64;

    nn::util::StoreBigEndian(reinterpret_cast<int64_t*>(&stream->buffer[stream->position]), value);
    stream->position += 8;

    return true;
}

bool WriteSignedIntegerAutoSize(OutputStreamParam* stream, int64_t value) NN_NOEXCEPT
{
    if (value >= -31 && value <= 127)
    {
        return WriteFixNum(stream, static_cast<int8_t>(value));
    }
    if (value >= INT8_MIN && value <= INT8_MAX)
    {
        return WriteSignedInteger8(stream, static_cast<int8_t>(value));
    }
    if (value >= INT16_MIN && value <= INT16_MAX)
    {
        return WriteSignedInteger16(stream, static_cast<int16_t>(value));
    }
    if (value >= INT32_MIN && value <= INT32_MAX)
    {
        return WriteSignedInteger32(stream, static_cast<int32_t>(value));
    }

    return WriteSignedInteger64(stream, value);
}

bool WriteUnsignedInteger8(OutputStreamParam* stream, uint8_t value) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 2)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_U8;
    stream->buffer[stream->position++] = static_cast<Bit8>(value);

    return true;
}

bool WriteUnsignedInteger16(OutputStreamParam* stream, uint16_t value) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 3)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_U16;

    nn::util::StoreBigEndian(reinterpret_cast<uint16_t*>(&stream->buffer[stream->position]), value);
    stream->position += 2;

    return true;
}

bool WriteUnsignedInteger32(OutputStreamParam* stream, uint32_t value) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 5)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_U32;

    nn::util::StoreBigEndian(reinterpret_cast<uint32_t*>(&stream->buffer[stream->position]), value);
    stream->position += 4;

    return true;
}

bool WriteUnsignedInteger64(OutputStreamParam* stream, uint64_t value) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 9)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_U64;

    nn::util::StoreBigEndian(reinterpret_cast<uint64_t*>(&stream->buffer[stream->position]), value);
    stream->position += 8;

    return true;
}

bool WriteUnsignedIntegerAutoSize(OutputStreamParam* stream, uint64_t value) NN_NOEXCEPT
{
    if (value <= 127)
    {
        return WriteFixNum(stream, static_cast<int8_t>(value));
    }
    if (value <= UINT8_MAX)
    {
        return WriteUnsignedInteger8(stream, static_cast<uint8_t>(value));
    }
    if (value <= UINT16_MAX)
    {
        return WriteUnsignedInteger16(stream, static_cast<uint16_t>(value));
    }
    if (value <= UINT32_MAX)
    {
        return WriteUnsignedInteger32(stream, static_cast<uint32_t>(value));
    }

    return WriteUnsignedInteger64(stream, value);
}

bool WriteFloat32(OutputStreamParam* stream, float value) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 5)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_Float32;

    nn::util::StoreBigEndian(reinterpret_cast<float*>(&stream->buffer[stream->position]), value);
    stream->position += 4;

    return true;
}

bool WriteFloat64(OutputStreamParam* stream, double value) NN_NOEXCEPT
{
    if (stream->GetRemainSize() < 9)
    {
        return false;
    }

    stream->buffer[stream->position++] = DataType_Float64;

    nn::util::StoreBigEndian(reinterpret_cast<double*>(&stream->buffer[stream->position]), value);
    stream->position += 8;

    return true;
}

}}}}
