﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pdm/detail/pdm_Util.h>
#include <nn/pdm/detail/pdm_Log.h>
#include <nn/pdm/detail/pdm_Time.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace pdm { namespace detail {

namespace
{
    uint64_t ToUint64(uint32_t hi, uint32_t low) NN_NOEXCEPT
    {
        return (static_cast<uint64_t>(hi) << 32) | low;
    }
}

uint32_t Hi(uint64_t value) NN_NOEXCEPT
{
    return static_cast<uint32_t>((value >> 32) & 0xffffffff);
}

uint32_t Low(uint64_t value) NN_NOEXCEPT
{
    return static_cast<uint32_t>((value >> 0) & 0xffffffff);
}

bool IsApplicationEvent(const PlayEvent& playEvent) NN_NOEXCEPT
{
    return (playEvent.eventCategory == PlayEventCategory::Applet) && (playEvent.appletEventData.appletId == nn::applet::AppletId_Application);
}

bool IsApplicationBecomeForegroundEvent(const AppletEventType& type) NN_NOEXCEPT
{
    return (AppletEventType::Launch == type || AppletEventType::InFocus == type);
}

bool IsApplicationBecomeForegroundEvent(const PlayEvent& playEvent, bool statisticsMustBeAvailable) NN_NOEXCEPT
{
    if( detail::IsApplicationEvent(playEvent)
    && IsApplicationBecomeForegroundEvent(playEvent.appletEventData.eventType) )
    {
        if( statisticsMustBeAvailable )
        {
            return playEvent.appletEventData.playLogPolicy == ns::PlayLogPolicy::All;
        }
        else
        {
            return true;
        }
    }
    return false;
}

bool IsLibraryApplet(applet::AppletId id) NN_NOEXCEPT
{
    switch( id )
    {
    case applet::AppletId::AppletId_None:
    case applet::AppletId::AppletId_Application:
    case applet::AppletId::AppletId_OverlayApplet:
    case applet::AppletId::AppletId_SystemAppletMenu:
    case applet::AppletId::AppletId_SystemApplication:
        return false;
    default:
        return true;
    }
}

nn::ncm::ProgramId GetProgramId(const PlayEvent& playEvent) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(PlayEventCategory::Applet, playEvent.eventCategory);
    nn::ncm::ProgramId id{ ToUint64(playEvent.appletEventData.programIdHi, playEvent.appletEventData.programIdLow) };
    return id;
}

bool MatchProgramId(const nn::ncm::ProgramId& programId, const PlayEvent& playEvent) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(PlayEventCategory::Applet, playEvent.eventCategory);
    return programId == GetProgramId(playEvent);
}

nn::ncm::ApplicationId GetApplicationId(const PlayEvent& playEvent) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(PlayEventCategory::Applet, playEvent.eventCategory);
    nn::ncm::ApplicationId id{ ToUint64(playEvent.appletEventData.programIdHi, playEvent.appletEventData.programIdLow) };
    return id;
}

bool MatchApplicationId(const nn::ncm::ApplicationId& applicationId, const PlayEvent& playEvent) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(PlayEventCategory::Applet, playEvent.eventCategory);
    return applicationId == GetApplicationId(playEvent);
}

nn::account::Uid GetUid(const PlayEvent& playEvent) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(PlayEventCategory::UserAccount, playEvent.eventCategory);
    nn::account::Uid uid;
    uid._data[0] = ToUint64(playEvent.userAccountEventData.userId0Hi, playEvent.userAccountEventData.userId0Low);
    uid._data[1] = ToUint64(playEvent.userAccountEventData.userId1Hi, playEvent.userAccountEventData.userId1Low);
    return uid;
}

bool MatchUid(const nn::account::Uid& user, const PlayEvent& playEvent) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(PlayEventCategory::UserAccount, playEvent.eventCategory);
    return user == GetUid(playEvent);
}

bool IsTargetUidOpenEvent(const nn::account::Uid& user, const PlayEvent& playEvent) NN_NOEXCEPT
{
    return (playEvent.eventCategory == PlayEventCategory::UserAccount)
        && (playEvent.userAccountEventData.eventType == UserAccountEventType::Open)
        && (MatchUid(user, playEvent));
}

bool IsTargetUidCloseEvent(const nn::account::Uid& user, const PlayEvent& playEvent) NN_NOEXCEPT
{
    return (playEvent.eventCategory == PlayEventCategory::UserAccount)
        && (playEvent.userAccountEventData.eventType == UserAccountEventType::Close)
        && (MatchUid(user, playEvent));
}

nn::account::NetworkServiceAccountId GetNetworkServiceAccountId(const PlayEvent& playEvent) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(PlayEventCategory::UserAccount, playEvent.eventCategory);
    NN_SDK_ASSERT_EQUAL(UserAccountEventType::NetworkServiceAccountAvailable, playEvent.userAccountEventData.eventType);
    nn::account::NetworkServiceAccountId nid;
    nid.id = ToUint64(playEvent.userAccountEventData.networkServiceAccountAvailableData.networkServiceAccountIdHi,
                      playEvent.userAccountEventData.networkServiceAccountAvailableData.networkServiceAccountIdLow);
    return nid;
}

bool MatchNetworkServiceAccountId(const nn::account::NetworkServiceAccountId& networkServiceAccountId, const PlayEvent& playEvent) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(PlayEventCategory::UserAccount, playEvent.eventCategory);
    NN_SDK_ASSERT_EQUAL(UserAccountEventType::NetworkServiceAccountAvailable, playEvent.userAccountEventData.eventType);
    return networkServiceAccountId == GetNetworkServiceAccountId(playEvent);
}

bool IsPowerOffEvent(const PlayEvent& playEvent) NN_NOEXCEPT
{
    return (playEvent.eventCategory == PlayEventCategory::PowerStateChange && playEvent.powerStateChangeEventData.eventType == PowerStateChangeEventType::Off);
}

bool IsPowerOnEvent(const PlayEvent& playEvent) NN_NOEXCEPT
{
    return (playEvent.eventCategory == PlayEventCategory::PowerStateChange && playEvent.powerStateChangeEventData.eventType == PowerStateChangeEventType::On);
}

void ConvertAppletEventToApplicationEvent(ApplicationEvent* outValue, const PlayEvent& playEvent) NN_NOEXCEPT
{
    NN_SDK_ASSERT(playEvent.eventCategory == PlayEventCategory::Applet);
    NN_SDK_ASSERT(playEvent.appletEventData.appletId == nn::applet::AppletId_Application);
    outValue->applicationId = GetApplicationId(playEvent);
    outValue->eventType = playEvent.appletEventData.eventType;
    outValue->eventTime.userClockTime = detail::GetElapsedMinutesSinceInputPosixTimeMin(playEvent.userTime);
    outValue->eventTime.networkClockTime = detail::GetElapsedMinutesSinceInputPosixTimeMin(playEvent.networkTime);
}

}}}
