﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/pdm/detail/pdm_Log.h>
#include <nn/pdm/detail/pdm_EventEntryStream.h>

namespace nn { namespace pdm { namespace detail {

//!---------------------------------------------------------------------------
const char* const EventEntryFileStream::DefaultPlayEventFileName = "PlayEvent.dat";

//!---------------------------------------------------------------------------
const char* EventEntryFileStream::MakeFilePath(char* pOutValue, int capacity, const char* pFileMountedVolumeName, const char* pFileName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pFileMountedVolumeName);

    auto pActualFileName = (nullptr != pFileName) ? pFileName : DefaultPlayEventFileName;
    auto volumeTail = util::Strlcpy(pOutValue, pFileMountedVolumeName, capacity);
    volumeTail += util::Strlcpy(&pOutValue[volumeTail], ":/", capacity - volumeTail);
    volumeTail += util::Strlcpy(&pOutValue[volumeTail], pActualFileName, capacity - volumeTail);
    return pOutValue;
}

//!---------------------------------------------------------------------------
EventEntryFileStream::EventEntryFileStream(uint16_t headerSize, uint16_t entryUnitSize) NN_NOEXCEPT
    : m_EventCountMax(0)
    , m_HeaderSize(headerSize)
    , m_EntrySize(entryUnitSize)
    , m_ReadFileOpened(false)
    , m_IsStreamOpened(false)
{
    NN_SDK_REQUIRES_GREATER(entryUnitSize, 0u);
    NN_SDK_REQUIRES_GREATER_EQUAL(headerSize, sizeof(Header));

    m_FilePath[0] = '\0';
}

//!---------------------------------------------------------------------------
void EventEntryFileStream::OnHeaderReset() NN_NOEXCEPT
{
    m_Header.Reset();
}

//!---------------------------------------------------------------------------
void EventEntryFileStream::PrepareDatabaseFilePath(const char* pFileMountedVolumeName, const char* pOpenFileName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFileMountedVolumeName);
    MakeFilePath(m_FilePath, NN_ARRAY_SIZE(m_FilePath), pFileMountedVolumeName, pOpenFileName);
}

//!---------------------------------------------------------------------------
EventEntryFileStream::OpenFileResult EventEntryFileStream::OpenFileForExistOnly(fs::FileHandle* pOutHandle) NN_NOEXCEPT
{
    const int64_t expectFileSize = m_HeaderSize + m_EventCountMax * static_cast<int64_t>(m_EntrySize);
    if (fs::OpenFile(pOutHandle, m_FilePath, fs::OpenMode_Read).IsSuccess())
    {
        int64_t fileSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetFileSize(&fileSize, *pOutHandle));
        if (fileSize == expectFileSize)
        {
            //NN_DETAIL_PDM_TRACE_V1("[pdm] EventEntryFileStream::OpenFileForExistOnly: [ %s ].\n", m_FilePath);
            return OpenFileResult::Opened;
        }
        else
        {
            fs::CloseFile(*pOutHandle);
        }
    }
    OnHeaderReset();
    return OpenFileResult::Failed;
}

//!---------------------------------------------------------------------------
EventEntryFileStream::OpenFileResult EventEntryFileStream::OpenFile(fs::FileHandle* pOutHandle) NN_NOEXCEPT
{
    const int64_t expectFileSize = m_HeaderSize + m_EventCountMax * static_cast<int64_t>(m_EntrySize);
    auto result = fs::OpenFile(pOutHandle, m_FilePath, fs::OpenMode_Write | fs::OpenMode_Read);
    if (result.IsSuccess())
    {
        int64_t fileSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetFileSize(&fileSize, *pOutHandle));
        if (fileSize == expectFileSize)
        {
            //NN_DETAIL_PDM_TRACE_V1("[pdm] EventEntryFileStream::OpenFile: [ %s ].\n", m_FilePath);
            return OpenFileResult::Opened;
        }
        else
        {
            fs::CloseFile(*pOutHandle);
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::DeleteFile(m_FilePath));
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CreateFile(m_FilePath, expectFileSize));
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(pOutHandle, m_FilePath, fs::OpenMode_Write | fs::OpenMode_Read | fs::OpenMode_AllowAppend));
            NN_DETAIL_PDM_WARN("[pdm] EventEntryFileStream: The size of the existing file does not match the expected size. Deleted it and created a new file(%s).\n", m_FilePath);
            OnHeaderReset();
            return OpenFileResult::NewlyCreated;
        }
    }
    else if (fs::ResultPathNotFound::Includes(result))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CreateFile(m_FilePath, expectFileSize));
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(pOutHandle, m_FilePath, fs::OpenMode_Write | fs::OpenMode_Read));
        NN_DETAIL_PDM_TRACE_V1("[pdm] EventEntryFileStream: Created a new file(%s).\n", m_FilePath);
        OnHeaderReset();
        return OpenFileResult::NewlyCreated;
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        return OpenFileResult::Failed;
    }
}

//!---------------------------------------------------------------------------
EventEntryFileStream::OpenResult EventEntryFileStream::OpenUnsafe(uint32_t eventCountMax, bool isExistOnly) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(eventCountMax, 0u);
    NN_SDK_REQUIRES(IsPreparedUnsafe());

    if (IsOpenedUnsafe())
    {
        return OpenResult::AlreadyOpened;
    }
    m_EventCountMax = eventCountMax;

    fs::FileHandle fileHandle;
    const auto result = (isExistOnly) ? OpenFileForExistOnly(&fileHandle) : OpenFile(&fileHandle);
    NN_UTIL_SCOPE_EXIT
    {
        if (OpenFileResult::Failed != result)
        {
            fs::CloseFile(fileHandle);
        }
    };
    if (OpenFileResult::Opened == result)
    {
        if (OnHeaderRead(fileHandle, 0).IsFailure())
        {
            OnHeaderReset();
        }
        m_IsStreamOpened = true;
        return OpenResult::Opened;
    }
    else if (OpenFileResult::NewlyCreated == result)
    {
        m_IsStreamOpened = true;
        return OpenResult::NewlyCreated;
    }
    else if (false == isExistOnly)
    {
        // OpenFile 内で Abort するのでここには到達しない。
        NN_SDK_ASSERT(false);
    }
    return OpenResult::Error;
}

//!---------------------------------------------------------------------------
void EventEntryFileStream::Close() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_FileMutex);

    m_IsStreamOpened = false;
    m_EventCountMax = 0;
    OnHeaderReset();
}

//!---------------------------------------------------------------------------
void EventEntryFileStream::Discard() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_FileMutex);

    // ファイルの削除
    auto result = nn::fs::DeleteFile(m_FilePath);
    if (!(result.IsSuccess() || fs::ResultPathNotFound::Includes(result)))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    OnHeaderReset();
}

//!---------------------------------------------------------------------------
uint32_t EventEntryFileStream::GetLastIndex() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_FileMutex);
    return (IsOpenedUnsafe()) ? GetLastIndexUnsafe() : 0;
}

//!---------------------------------------------------------------------------
uint32_t EventEntryFileStream::GetStartIndex() const NN_NOEXCEPT
{
    // Out of order を考慮してレジスタ長単一変数読み込みであっても排他します。
    NN_UTIL_LOCK_GUARD(m_FileMutex);
    return (IsOpenedUnsafe()) ? GetStartIndexUnsafe() : 0;
}

//!---------------------------------------------------------------------------
uint32_t EventEntryFileStream::GetCount() const NN_NOEXCEPT
{
    // Out of order を考慮してレジスタ長単一変数読み込みであっても排他します。
    NN_UTIL_LOCK_GUARD(m_FileMutex);
    return (IsOpenedUnsafe()) ? m_Header.count : 0;
}

//!---------------------------------------------------------------------------
uint32_t EventEntryFileStream::GetCountMax() const NN_NOEXCEPT
{
    // Out of order を考慮してレジスタ長単一変数読み込みであっても排他します。
    NN_UTIL_LOCK_GUARD(m_FileMutex);
    return (IsOpenedUnsafe()) ? m_EventCountMax : 0;
}

//!---------------------------------------------------------------------------
bool EventEntryFileStream::IsOpened() const NN_NOEXCEPT
{
    // Out of order を考慮してレジスタ長単一変数読み込みであっても排他します。
    NN_UTIL_LOCK_GUARD(m_FileMutex);
    return IsOpenedUnsafe();
}

//!---------------------------------------------------------------------------
EventEntryFileStream::Header EventEntryFileStream::GetHeader() const NN_NOEXCEPT
{
    // Out of order を考慮してレジスタ長単一変数読み込みであっても排他します。
    NN_UTIL_LOCK_GUARD(m_FileMutex);
    return GetHeaderUnsafe();
}

//!---------------------------------------------------------------------------
void EventEntryFileStream::BeginReadWithForceInvalidation() NN_NOEXCEPT
{
    m_FileMutex.Lock();
    m_ReadFileOpened = false;
}

//!---------------------------------------------------------------------------
void EventEntryFileStream::BeginRead() NN_NOEXCEPT
{
    m_FileMutex.Lock();
    m_ReadFileOpened = (IsOpenedUnsafe() && fs::OpenFile(&m_ReadFileHandle, m_FilePath, fs::OpenMode_Read).IsSuccess());
    // 従来は !Success で ABORT してたが、Discard() を経由した場合、
    // イベントレコードが発生( Write() )しなければ、Query 呼ぶだけで Abort してしまうのでオープン失敗の場合は読み込み byte が 0 として変更。
}

//!---------------------------------------------------------------------------
void EventEntryFileStream::EndRead() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_FileMutex.IsLockedByCurrentThread());
    if (m_ReadFileOpened)
    {
        m_ReadFileOpened = false;
        fs::CloseFile(m_ReadFileHandle);
    }
    m_FileMutex.Unlock();
}

//!---------------------------------------------------------------------------
uint32_t EventEntryFileStream::ReadImpl(void* pOutValue, uint32_t startIndex, uint32_t outCountMax) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_FileMutex.IsLockedByCurrentThread());
    NN_SDK_REQUIRES(startIndex >= GetStartIndexUnsafe());
    NN_SDK_REQUIRES(outCountMax > 0);

    // ファイルオープンに失敗している場合は無効。
    if (!m_ReadFileOpened)
    {
        return 0u;
    }
    if (0u == m_Header.count)
    {
        return 0u;
    }
    const auto lastIndex = GetLastIndexUnsafe();
    if (startIndex > lastIndex)
    {
        return 0u;
    }

    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    auto readEventCount = std::min(outCountMax, lastIndex - startIndex + 1);

    if ((startIndex % m_EventCountMax) + readEventCount >= m_EventCountMax)
    {
        // ファイルに記録可能な最大数のインデックス（N）を跨いで [X ~ N-1] と [0 ~ Y] を読み込むパターン。
        auto startToEndCount = m_EventCountMax - (startIndex % m_EventCountMax);
        auto remainingCount = readEventCount - startToEndCount;
        auto pNextOutValue = &(static_cast<uint8_t*>(pOutValue))[startToEndCount * m_EntrySize];
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(m_ReadFileHandle, ComputeFilePosition(startIndex), pOutValue, startToEndCount * m_EntrySize));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(m_ReadFileHandle, ComputeFilePosition(startIndex + startToEndCount), pNextOutValue, remainingCount * m_EntrySize));
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(m_ReadFileHandle, ComputeFilePosition(startIndex), pOutValue, readEventCount * m_EntrySize));
    }
    return readEventCount;
}

//!---------------------------------------------------------------------------
void EventEntryFileStream::WriteUnsafe(const void* pWriteEntrySource, uint32_t writeEntryCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pWriteEntrySource);
    NN_SDK_REQUIRES(writeEntryCount > 0);
    if (false == IsOpenedUnsafe())
    {
        return;
    }
    const auto writeOption = fs::WriteOption::MakeValue(0);
    fs::FileHandle fileHandle;
    OpenFile(&fileHandle);
    NN_UTIL_SCOPE_EXIT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(OnHeaderWrite(fileHandle, 0, writeOption, nullptr));
        fs::FlushFile(fileHandle);
        fs::CloseFile(fileHandle);
    };

    if (m_Header.count + writeEntryCount <= m_EventCountMax)
    {
        NN_DETAIL_PDM_TRACE("[pdm] EventEntryFileStream: Add %d events.\n", writeEntryCount);
        // ファイルに記録可能な最大数を超えない場合。
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::WriteFile(fileHandle, ComputeFilePosition(m_Header.count), pWriteEntrySource, m_EntrySize * writeEntryCount, writeOption));
        m_Header.count += writeEntryCount;
    }
    else if (m_Header.count == m_EventCountMax)
    {
        // 既にファイルに記録可能な最大数に到達している場合。
        if (std::numeric_limits<uint32_t>::max() - m_Header.start < writeEntryCount)
        {
            // Header の start は 0 にリセットされることがないのでいつか限界に到達する。
            // uint32_t の場合、100年間毎日10万件イベントを起こしても大丈夫なので事実上は到達しないはず。
            NN_DETAIL_PDM_WARN("[pdm] EventEntryFileStream: Start index reached its upper limit. Unable to write more events.\n");
            return;
        }
        NN_DETAIL_PDM_TRACE("[pdm] EventEntryFileStream: Overwrite %d events.\n", writeEntryCount);
        // 先頭のイベントと末尾のイベントの、リングバッファ処理を考慮したファイル上での書き込み位置を比較。
        // 末尾のイベントの方がインデックスが小さければファイルの終端を跨いでいるので、分けて書き込む必要がある。
        auto firstEventWriteIndex = m_Header.start % m_EventCountMax;
        auto lastEventWriteIndex = (m_Header.start + writeEntryCount) % m_EventCountMax;
        if (firstEventWriteIndex > lastEventWriteIndex)
        {
            auto tailEventCount = m_EventCountMax - firstEventWriteIndex;
            // 末尾の上書き。
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::WriteFile(fileHandle, ComputeFilePosition(m_Header.start), pWriteEntrySource, m_EntrySize * tailEventCount, writeOption));
            // 先頭の上書き。
            auto pNextSourceEntry = &(static_cast<const uint8_t*>(pWriteEntrySource))[m_EntrySize * tailEventCount];
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::WriteFile(fileHandle, ComputeFilePosition(0), pNextSourceEntry, m_EntrySize * (writeEntryCount - tailEventCount), writeOption));
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::WriteFile(fileHandle, ComputeFilePosition(m_Header.start), pWriteEntrySource, m_EntrySize * writeEntryCount, writeOption));
        }
        m_Header.start += writeEntryCount;
    }
    else
    {
        // 今回の追加で最大数を超える場合。
        auto addCount = m_EventCountMax - m_Header.count;
        auto overwriteCount = writeEntryCount - addCount;
        NN_DETAIL_PDM_TRACE("[pdm] EventEntryFileStream: Add %d events, overwrite %d events.\n", addCount, overwriteCount);

        // 末尾への追加。
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::WriteFile(fileHandle, ComputeFilePosition(m_Header.count), pWriteEntrySource, m_EntrySize * addCount, writeOption));
        m_Header.count += addCount;
        // 先頭の上書き。
        auto pNextSourceEntry = &(static_cast<const uint8_t*>(pWriteEntrySource))[m_EntrySize * addCount];
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::WriteFile(fileHandle, ComputeFilePosition(m_Header.start), pNextSourceEntry, m_EntrySize * overwriteCount, writeOption));
        m_Header.start += overwriteCount;

        NN_SDK_ASSERT(m_Header.count == m_EventCountMax);   // 最大数まで記録なので一致するはず。
        NN_SDK_ASSERT(m_Header.start == overwriteCount);    // 初めての start の増加のはずなので一致するはず。
    }
}

//!---------------------------------------------------------------------------
Result EventEntryFileStream::OnHeaderWrite(fs::FileHandle fileHandle, int64_t position, const fs::WriteOption inheritedOption, size_t* pOutWriteSize) NN_NOEXCEPT
{
    NN_RESULT_DO(fs::WriteFile(fileHandle, position, &m_Header, sizeof(Header), inheritedOption));
    if (nullptr != pOutWriteSize)
    {
        *pOutWriteSize = sizeof(Header);
    }
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
Result EventEntryFileStream::OnHeaderRead(fs::FileHandle fileHandle, int64_t position, size_t* pOutReadSize) NN_NOEXCEPT
{
    NN_RESULT_DO(fs::ReadFile(fileHandle, position, &m_Header, sizeof(Header)));
    if (nullptr != pOutReadSize)
    {
        *pOutReadSize = sizeof(Header);
    }
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
void EventEntryFileStream::WriteHeaderOnlyUnsafe() NN_NOEXCEPT
{
    if (false == IsOpenedUnsafe())
    {
        return;
    }

    fs::FileHandle fileHandle;
    OpenFile(&fileHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(OnHeaderWrite(fileHandle, 0, fs::WriteOption::MakeValue(0), nullptr));
    fs::FlushFile(fileHandle);
    fs::CloseFile(fileHandle);
}

//!---------------------------------------------------------------------------
LockGuard EventEntryFileStream::Write(const void* pWriteEntrySource, uint32_t writeEntryCount) NN_NOEXCEPT
{
    LockGuard fileLock(m_FileMutex);
    WriteUnsafe(pWriteEntrySource, writeEntryCount);
    return fileLock;
}

}}}
