﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/detail/pctl_ResultHandler.h>
#include <nn/pctl/detail/ipc/pctl_IpcSession.h>
#include <nn/result/result_HandlingUtility.h>

#include <cstring>

namespace nn { namespace pctl {

// 本体機能(システム)向けペアコン状態チェック関数

nn::Result ConfirmLaunchApplicationPermission(nn::ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& controlProperty) NN_NOEXCEPT
{
    nn::sf::InArray<int8_t> ratingAge(controlProperty.ratingAge, std::extent<decltype(controlProperty.ratingAge)>::value);
    const bool freeCommunicationFlag =
        (controlProperty.parentalControlFlag & static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication)) != 0;

    auto result = detail::ipc::GetService()->ConfirmLaunchApplicationPermission(applicationId, ratingAge, freeCommunicationFlag);
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);

    return result;
}

nn::Result ConfirmResumeApplicationPermission(nn::ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& controlProperty) NN_NOEXCEPT
{
    nn::sf::InArray<int8_t> ratingAge(controlProperty.ratingAge, std::extent<decltype(controlProperty.ratingAge)>::value);
    const bool freeCommunicationFlag =
        (controlProperty.parentalControlFlag & static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication)) != 0;

    auto result = detail::ipc::GetService()->ConfirmResumeApplicationPermission(applicationId, ratingAge, freeCommunicationFlag);
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);

    return result;
}

void RevertRestrictionTemporaryUnlocked() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->RevertRestrictionTemporaryUnlocked());
}

void EnterRestrictedSystemSettings() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->EnterRestrictedSystemSettings());
}

void LeaveRestrictedSystemSettings() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->LeaveRestrictedSystemSettings());
}

void RevertRestrictedSystemSettingsEntered() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->RevertRestrictedSystemSettingsEntered());
}

// ペアコン設定変更関数

void SetSafetyLevel(SafetyLevel level) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(level, SafetyLevel::SafetyLevel_None, SafetyLevel::SafetyLevel_Max);

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->SetSafetyLevel(static_cast<int>(level)));
}

void GetSafetyLevelSettings(SafetyLevelSettings* pSettings, SafetyLevel level) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);
    NN_SDK_REQUIRES_RANGE(level, SafetyLevel::SafetyLevel_None, SafetyLevel::SafetyLevel_Max);

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->GetSafetyLevelSettings(pSettings, static_cast<int>(level)));
}

void SetCustomSafetyLevelSettings(const SafetyLevelSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(settings.ratingAge, 0, 255);

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->SetCustomSafetyLevelSettings(settings));
}

nn::ns::RatingOrganization GetDefaultRatingOrganization() NN_NOEXCEPT
{
    int value = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->GetDefaultRatingOrganization(&value));
    return static_cast<nn::ns::RatingOrganization>(value);
}

void SetDefaultRatingOrganization(nn::ns::RatingOrganization organization) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->SetDefaultRatingOrganization(static_cast<int>(organization)));
}

void DeleteSettings() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->DeleteSettings());
}

void DisableFeaturesForReset() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->DisableFeaturesForReset());
}

int GetFreeCommunicationApplicationListCount() NN_NOEXCEPT
{
    int count = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->GetFreeCommunicationApplicationListCount(&count));
    return count;
}

int GetFreeCommunicationApplicationList(FreeCommunicationApplicationInfo* pOutInfoArray, int offset, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutInfoArray);
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

    int outCount = 0;
    nn::sf::OutArray<FreeCommunicationApplicationInfo> outInfoArray(pOutInfoArray, count);
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->GetFreeCommunicationApplicationList(&outCount, outInfoArray, offset));
    return outCount;
}

void UpdateFreeCommunicationApplicationList(const FreeCommunicationApplicationInfo* pInfoArray, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pInfoArray);
    NN_SDK_REQUIRES_GREATER(count, 0);

    nn::sf::InArray<FreeCommunicationApplicationInfo> infoArray(pInfoArray, count);
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->UpdateFreeCommunicationApplicationList(infoArray));
}

void AddToFreeCommunicationApplicationList(nn::ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& controlProperty) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(applicationId, nn::ncm::ApplicationId::GetInvalidId());

    const bool freeCommunicationFlag =
        (controlProperty.parentalControlFlag & static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication)) != 0;
    if (freeCommunicationFlag)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->AddToFreeCommunicationApplicationList(applicationId));
    }
}

void NotifyApplicationDownloadStarted(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(applicationId, nn::ncm::ApplicationId::GetInvalidId());

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->NotifyApplicationDownloadStarted(applicationId));
}

void DeleteFromFreeCommunicationApplicationListForDebug(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(applicationId, nn::ncm::ApplicationId::GetInvalidId());

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->DeleteFromFreeCommunicationApplicationListForDebug(applicationId));
}

void ClearFreeCommunicationApplicationListForDebug() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->ClearFreeCommunicationApplicationListForDebug());
}

int GetExemptApplicationListCountForDebug() NN_NOEXCEPT
{
    int count = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->GetExemptApplicationListCountForDebug(&count));
    return count;
}

int GetExemptApplicationListForDebug(ExemptApplicationInfo* pOutInfoArray, int offset, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutInfoArray);
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

    int outCount = 0;
    nn::sf::OutArray<ExemptApplicationInfo> outInfoArray(pOutInfoArray, count);
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->GetExemptApplicationListForDebug(&outCount, outInfoArray, offset));
    return outCount;
}

void UpdateExemptApplicationListForDebug(const ExemptApplicationInfo* pInfoArray, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pInfoArray);
    NN_SDK_REQUIRES_GREATER(count, 0);

    nn::sf::InArray<ExemptApplicationInfo> infoArray(pInfoArray, count);
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->UpdateExemptApplicationListForDebug(infoArray));
}

void AddToExemptApplicationListForDebug(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(applicationId, nn::ncm::ApplicationId::GetInvalidId());

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->AddToExemptApplicationListForDebug(applicationId));
}

void DeleteFromExemptApplicationListForDebug(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(applicationId, nn::ncm::ApplicationId::GetInvalidId());

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->DeleteFromExemptApplicationListForDebug(applicationId));
}

void ClearExemptApplicationListForDebug() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->ClearExemptApplicationListForDebug());
}

}}
