﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/pctl/pctl_ApiPairing.h>
#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Npns.h>
#include <nn/pctl/detail/pctl_PairingInfoBase.h>
#include <nn/pctl/detail/pctl_ResultHandler.h>
#include <nn/pctl/detail/ipc/pctl_IpcSession.h>

#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace pctl {

namespace
{
    void WaitSystemEventPairing(nn::sf::NativeHandle& nativeHandle) NN_NOEXCEPT
    {
        nn::os::SystemEventType eventObject;
        nn::os::AttachReadableHandleToSystemEvent(&eventObject,
            nativeHandle.GetOsHandle(),
            nativeHandle.IsManaged(),
            nn::os::EventClearMode_ManualClear);
        nativeHandle.Detach();
        nn::os::WaitSystemEvent(&eventObject);
        nn::os::DestroySystemEvent(&eventObject);
    }
}

// 見守り機能サーバー連携確立処理関数

nn::Result RequestPairing(PairingInfo* pInfo, const char* code) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pInfo);
    NN_SDK_REQUIRES_NOT_NULL(code);

    bool accepted = nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId());
    NN_RESULT_THROW_UNLESS(accepted, nn::pctl::ResultConnectionNotAccepted());

    auto result = detail::PrepareNpns();
    if (result.IsFailure())
    {
        NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
        return result;
    }

    detail::AsyncData data;
    nn::sf::NativeHandle eventHandle;
    result = detail::ipc::GetServiceForWatcher()->RequestPairingAsync(
        &data, &eventHandle, nn::sf::InArray<char>(code, std::strlen(code) + 1)
        );
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_DO(result);

    WaitSystemEventPairing(eventHandle);

    result = detail::ipc::GetServiceForWatcher()->FinishRequestPairing(detail::ToHolder(pInfo), data);
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_DO(result);

    NN_RESULT_SUCCESS;
}

nn::Result AuthorizePairing(PairingInfo* pInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pInfo);

    bool accepted = nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId());
    NN_RESULT_THROW_UNLESS(accepted, nn::pctl::ResultConnectionNotAccepted());

    detail::PairingInfoHolder holderSource = *detail::ToHolder(pInfo);
    detail::AsyncData data;
    nn::sf::NativeHandle eventHandle;
    auto result = detail::ipc::GetServiceForWatcher()->AuthorizePairingAsync(
        &data, &eventHandle, holderSource);
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_DO(result);

    WaitSystemEventPairing(eventHandle);

    result = detail::ipc::GetServiceForWatcher()->FinishAuthorizePairing(detail::ToHolder(pInfo), data);
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_DO(result);

    NN_RESULT_SUCCESS;
}

}}
