﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/pctl_RetrieveSettings.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/overlay/pctl_OverlaySender.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherErrorHandler.h>
#include <nn/pctl/detail/service/watcher/dispatcher/pctl_RetrieveSettingsDispatcher.h>
#include <nn/pctl/detail/service/watcher/dispatcher/pctl_UpdateDeviceDispatcher.h>
#include <nn/pctl/detail/service/watcher/dispatcher/pctl_RequestUpdateExemptionListDispatcher.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

namespace
{
    // FreeCommunicationApplicationSettings::UpdateDataFromList の unknownTitleCallback に指定するコールバック関数
    void AppListUnknownTitleCallback(nn::ncm::ApplicationId id) NN_NOEXCEPT
    {
        // 自データにはないので改めて削除を呼び出す
        NN_DETAIL_PCTL_TRACE("AppListUnknownTitleCallback: id = 0x%016llX\n", id.value);
        g_pWatcher->GetWatcherEventStorage().AddRemoveManagedApplicationEvent(id);
    }

    // FreeCommunicationApplicationSettings::UpdateDataFromList の notTitleInListCallback に指定するコールバック関数
    void AppListNotTitleInListCallback(nn::ncm::ApplicationId id) NN_NOEXCEPT
    {
        // 自データにはあるので改めて追加を呼び出す
        NN_DETAIL_PCTL_TRACE("AppListNotTitleInListCallback: id = 0x%016llX\n", id.value);
        g_pWatcher->GetWatcherEventStorage().AddNewManagedApplicationEvent(id);
    }
}

////////////////////////////////////////////////////////////////////////////////

nn::Result RetrieveSettingsExecutor::Execute(common::NetworkBuffer& bufferInfo) NN_NOEXCEPT
{
    m_RetrieveParam.isNotModified = false;
    m_RetrieveParam.changedTypeFlags.Reset();
    m_RetrieveParam.pBufferInfo = &bufferInfo;

    auto result = ExecuteRetrieveAndStoreSettings(m_RetrieveParam);
    if (result.IsSuccess())
    {
        // 一切変更が無い(Not Modifed)場合は通知しない
        if (!m_RetrieveParam.isNotModified)
        {
            // バックグラウンドモードの場合は設定変更を通知する
            if (IsBackgroundMode())
            {
                g_pWatcher->GetWatcherEventManager().GetSynchronizationEvent()->Signal();
                // 解除コードの変更があればそのイベントも通知する
                if (m_RetrieveParam.changedTypeFlags.Test<nn::ovln::format::PctlSettingTypeFlag::PinCode>())
                {
                    g_pWatcher->GetWatcherEventManager().GetPinCodeChangedEvent()->Signal();
                }
            }

            bool isOverlayNotificationNecessary = m_IsOverlayNotificationNecessary;
            // 取得したものの結局全く変更がなかった場合はオーバーレイ通知しない
            // (強制取得の場合は isNotModified が true にならないものの、
            // 同じ Etag の設定値を取得する可能性がある)
            if (m_RetrieveParam.etagInfo.lastUpdatedAt == m_OriginalEtagInfo.lastUpdatedAt &&
                nn::util::Strncmp(
                    m_RetrieveParam.etagInfo.etag, m_OriginalEtagInfo.etag,
                    static_cast<int>(std::extent<decltype(m_OriginalEtagInfo.etag)>::value)
                    ) == 0)
            {
                NN_DETAIL_PCTL_TRACE("[pctl] Settings retrieved but etag is not changed, so skip overlay notification.\n");
                isOverlayNotificationNecessary = false;
            }

            // 制限対象外リスト以外に変更が無ければオーバーレイ通知しない
            if(m_RetrieveParam.isOnlyExemptionListUpdated)
            {
                NN_DETAIL_PCTL_TRACE("[pctl] Settings retrieved but etag is not changed except exemption list, so skip overlay notification.\n");
                isOverlayNotificationNecessary = false;
            }

            // オーバーレイ通知が必要である場合は、flags.IsAnyOn() が false (差分なし)であっても通知を行う
            // (isNotModified が true の場合別の方法 or タイミングで反映済みであるはずなので通知不要)
            if (isOverlayNotificationNecessary)
            {
                NN_DETAIL_PCTL_TRACE("[pctl] Send overlay notification for setting change.\n");
                auto r = overlay::NotifySettingChanged(m_RetrieveParam.changedTypeFlags);
                if (r.IsFailure())
                {
                    NN_DETAIL_PCTL_INFO("[pctl] Failed to send overlay notification (0x%08lX)\n", r.GetInnerValueForDebug());
                }
            }
        }
    }
    // バックグラウンド処理の場合、失敗でかつキャンセルでない場合はリトライする
    else if (IsBackgroundMode() && !nn::pctl::ResultCanceled::Includes(result))
    {
        // (リトライの場合はNPNS由来としない)
        g_pWatcher->GetWatcherEventManager().RequestRetrieveSettingsBackground(m_TryCount + 1, false);
    }
    return result;
}

//static
nn::Result RetrieveSettingsExecutor::ExecuteRetrieveAndStoreSettings(RetrieveSettingsParamBuffer& retrieveParam) NN_NOEXCEPT
{
    TokenHolder tokenHolder;
    NN_RESULT_DO(g_pWatcher->GetNetworkManager().AcquireAuthenticationToken(tokenHolder, *retrieveParam.pBufferInfo, retrieveParam.pCancelable));

    retrieveParam.isOnlyExemptionListUpdated = false;
    // Settings は一部ローカルのみの情報があるので現在の設定を取得しておく
    g_pMain->GetSettingsManager().GetSettings(retrieveParam.settings);
    // それ以外はすべてを取得することになるので 0 初期化する
    std::memset(&retrieveParam.freeCommunicationSettings, 0, sizeof(system::FreeCommunicationApplicationSettings));
    std::memset(&retrieveParam.playTimerSettings, 0, sizeof(PlayTimerSettings));

    bool isNotModified = false;
    bool isExemptionListUpdated = false;
    if (retrieveParam.exemptionSettings.GetAvailableCount() == 0)
    {
        NN_RESULT_DO(
            dispatcher::RetrieveSettingsDispatcher::Execute(
                &retrieveParam.etagInfo,
                &isNotModified,
                &retrieveParam.settings,
                &retrieveParam.freeCommunicationSettings,
                &retrieveParam.exemptionSettings,
                &retrieveParam.playTimerSettings,
                *retrieveParam.pBufferInfo,
                retrieveParam.pCancelable,
                tokenHolder.GetToken(),
                retrieveParam.deviceId)
        );
    }
    else
    {
        NN_RESULT_DO(
            dispatcher::RequestUpdateExemptionListDispatcher::Execute(
                &retrieveParam.etagInfo,
                &isNotModified,
                &retrieveParam.settings,
                &retrieveParam.freeCommunicationSettings,
                &retrieveParam.exemptionSettings,
                &retrieveParam.playTimerSettings,
                *retrieveParam.pBufferInfo,
                retrieveParam.pCancelable,
                tokenHolder.GetToken(),
                retrieveParam.deviceId)
        );
        isExemptionListUpdated = true;
    }
    retrieveParam.isNotModified = isNotModified;

    if (isNotModified)
    {
        // Etagを無効化する
        retrieveParam.etagInfo.etag[0] = 0;
    }

    // Etag をアップロード
    if (retrieveParam.etagInfo.etag[0] != 0)
    {
        NN_RESULT_DO(
            dispatcher::UpdateDeviceSynchronizationDispatcher::Execute(
                *retrieveParam.pBufferInfo,
                retrieveParam.pCancelable,
                tokenHolder.GetToken(),
                retrieveParam.deviceId,
                retrieveParam.settings.current.pinCode,
                &retrieveParam.etagInfo)
            );
    }

    if (!isNotModified)
    {
        NN_RESULT_THROW_UNLESS(
            g_pMain->GetSettingsManager().StoreAllSettings(&retrieveParam.changedTypeFlags,
                &retrieveParam.settings,
                &retrieveParam.freeCommunicationSettings,
                AppListUnknownTitleCallback, AppListNotTitleInListCallback,
                &retrieveParam.exemptionSettings),
            nn::pctl::ResultResponseFormatError()
            );
        g_pWatcher->GetNetworkManager().StoreEtagForSettings(&retrieveParam.etagInfo);
        if (g_pWatcher->GetWatcherEventManager().UpdatePlayTimerSettings(retrieveParam.playTimerSettings))
        {
            retrieveParam.changedTypeFlags.Set<nn::ovln::format::PctlSettingTypeFlag::PlayTimer>(true);
        }

        if (isExemptionListUpdated)
        {
            // RequestUpdateExemptionListDispatcher によって制限対象外リストのみが更新
            retrieveParam.changedTypeFlags.Set<nn::ovln::format::PctlSettingTypeFlag::ExemptAppList>(false);
            if (retrieveParam.changedTypeFlags.IsAllOff())
            {
                retrieveParam.isOnlyExemptionListUpdated = true;
            }
        }
    }

    NN_RESULT_SUCCESS;
}

void RetrieveSettingsExecutor::Cancel() NN_NOEXCEPT
{
    if (m_RetrieveParam.pCancelable != nullptr)
    {
        m_RetrieveParam.pCancelable->Cancel();
    }
}

}}}}}
