﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/dispatcher/pctl_MiiDownloadDispatcher.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_FileSystem.h>
#include <nn/pctl/detail/service/common/pctl_HttpRequest.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_FormatString.h>
#include <nn/util/util_IntUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include <cstdlib>

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_GCC)
#define STRTOULL strtoull
#elif defined(NN_BUILD_CONFIG_OS_WIN)
#define STRTOULL _strtoui64
#else
#define STRTOULL std::strtoull
#endif

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher { namespace dispatcher {

namespace
{
    struct MiiDownloaderCallbackData
    {
        char* contentTypeBuffer;
        uint32_t contentLength;
        nn::Result lastResult;
        bool responseHeaderEndCalled;
    };

    // 文字列リテラルと比較を行う関数
    // (NULL文字込みで比較する)
    template <typename T, size_t N>
    inline int CompareLiteralStringIgnoreCase(const T* string, const T (& stringLiteral)[N]) NN_NOEXCEPT
    {
        return nn::util::Strnicmp(string, stringLiteral, N);
    }

    static bool MiiDownloaderResponseHeaderCallback(char* label, char* value, void* param) NN_NOEXCEPT
    {
        if (CompareLiteralStringIgnoreCase(label, "Content-Length") == 0)
        {
            auto data = reinterpret_cast<MiiDownloaderCallbackData*>(param);
            char* p = nullptr;
            auto length = STRTOULL(value, &p, 10);
            if (p == nullptr || p == value || *p != 0)
            {
                return false;
            }
            if (length > MaxMiiImageDataSize)
            {
                NN_DETAIL_PCTL_WARN("Too large mii-image data: %llu\n", length);
                data->lastResult = nn::pctl::ResultInvalidMiiImageData();
                return false;
            }
            NN_STATIC_ASSERT(MaxMiiImageDataSize <= ~static_cast<uint32_t>(0));
            data->contentLength = static_cast<uint32_t>(length);
        }
        else if (CompareLiteralStringIgnoreCase(label, "Content-Type") == 0)
        {
            auto data = reinterpret_cast<MiiDownloaderCallbackData*>(param);
            nn::util::Strlcpy(data->contentTypeBuffer, value, NetworkManager::MaxMiiImageContentTypeLength);
        }
        return true;
    }

    static bool MiiDownloaderResponseHeaderEndCallback(int statusCode, void* param) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<MiiDownloaderCallbackData*>(param);
        data->responseHeaderEndCalled = true;
        data->lastResult = common::ConvertStatusCodeToResult(statusCode);
        // エラーレスポンスは見ないのでここで成否を判定する
        if (data->lastResult.IsFailure())
        {
            return false;
        }

        return true;
    }
}

nn::Result MiiDownloadDispatcher::Execute(common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable,
    const char* outputFileName, const char* miiUri, char* contentTypeBuffer) NN_NOEXCEPT
{
    // block: FileStream 処理
    {
        common::HttpRequest request;
        common::FileStream stream;
        MiiDownloaderCallbackData data;
        bool fileOpened = false;

        data.contentTypeBuffer = contentTypeBuffer;
        data.contentLength = 0;
        data.lastResult = nn::ResultSuccess();
        data.responseHeaderEndCalled = false;

        NN_RESULT_DO(request.Open("GET", miiUri));

        request.SetCancelable(pCancelable);
        request.RegisterResponseHeaderCallback(MiiDownloaderResponseHeaderCallback, &data);
        request.RegisterResponseHeaderEndCallback(MiiDownloaderResponseHeaderEndCallback, &data);

        int64_t offset = 0;
        while (NN_STATIC_CONDITION(true))
        {
            size_t read = 0;
            auto result = request.Receive(&read, bufferInfo.GetEntireBuffer(), common::NetworkBuffer::EntireBufferMemorySize);
            NN_RESULT_THROW_UNLESS(!pCancelable->IsCanceled(), ResultCanceled());
            NN_RESULT_DO(data.lastResult); // 先に判定
            NN_RESULT_DO(result);
            if (read == 0)
            {
                break;
            }
            if (data.responseHeaderEndCalled)
            {
                if (!fileOpened)
                {
                    if (data.contentLength > 0)
                    {
                        result = common::FileSystem::OpenWrite(&stream, outputFileName, static_cast<int64_t>(data.contentLength));
                    }
                    else
                    {
                        result = common::FileSystem::OpenWrite(&stream, outputFileName);
                    }
                    if (result.IsFailure())
                    {
                        NN_DETAIL_PCTL_ERROR("Unexpected: cannot open file for write mii image: result = 0x%08x, file = %s\n",
                            result.GetInnerValueForDebug(), outputFileName);
                        NN_RESULT_THROW(nn::pctl::ResultUnexpected());
                    }
                    fileOpened = true;
                }
                // Content-Length がセットされていない場合も考慮してここでも判定
                if (static_cast<size_t>(offset + static_cast<int64_t>(read)) > MaxMiiImageDataSize)
                {
                    NN_DETAIL_PCTL_WARN("Too large mii-image data: %llu\n", static_cast<uint64_t>(offset + static_cast<int64_t>(read)));
                    NN_RESULT_THROW(nn::pctl::ResultInvalidMiiImageData());
                }
                result = stream.Write(offset, bufferInfo.GetEntireBuffer(), read);
                if (result.IsFailure())
                {
                    NN_DETAIL_PCTL_ERROR("Unexpected: cannot write mii image: result = 0x%08x, file = %s\n",
                        result.GetInnerValueForDebug(), outputFileName);
                    NN_RESULT_THROW(nn::pctl::ResultUnexpected());
                }
                offset += static_cast<int64_t>(read);
            }
        }
        stream.Flush();
    } // end - block: FileStream 処理
    // Commit してすぐに使用できるようにする
    NN_RESULT_DO(common::FileSystem::Commit());

    NN_RESULT_SUCCESS;
}

}}}}}}
