﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fs/fs_Result.h>
#include <nn/os/os_Tick.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ApplicationStateManager.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_Constants.h>
#include <nn/pctl/detail/service/common/pctl_FileSystem.h>
#include <nn/pctl/detail/service/common/pctl_SystemInfo.h>
#include <nn/pctl/detail/service/system/pctl_SettingsManager.h>
#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_LockGuard.h>
#include <random>
#include <type_traits>

namespace nn { namespace pctl { namespace detail { namespace service { namespace system {

namespace
{
    static const char GeneralSettingsFileName[] = "settings.dat";
    static const char ApplicationSettingsFileName[] = "appRestrictions.dat";
    static const char ExemptionSettingsFileName[] = "appExemptions.dat";
    static const char DisableAllFeaturesFileName[] = "disableFeatures.mark";

    // ランダムで生成する解除コードの固定桁数
    static const int RandomPinCodeLength = 4;

    static std::mt19937 s_Mt;

    bool IsRestrictedByRating(uint8_t settingAge, nn::ns::RatingOrganization organization, const int8_t* ratingAgeData, size_t ratingAgeDataSize) NN_NOEXCEPT
    {
        // settingAge == 0 は設定(制限)なし
        if (settingAge == 0)
        {
            return false;
        }

        int8_t age = (static_cast<size_t>(organization) < ratingAgeDataSize) ? ratingAgeData[static_cast<size_t>(organization)] : -1;
        // GetDefaultRatingOrganization() で得られるレーティング団体が対応していない場合は
        // 最も厳しい(= 最も大きな値の)年齢値を用いる
        // (※ 「対応していない」== age が負数: nn::ns::ApplicationControlProperty::GetAge() を参照)
        if (age < 0)
        {
            // USKの場合、対応する設定が無ければ以下の判定を行わず一律「制限あり」とする(リーガル要求)
            if (organization == nn::ns::RatingOrganization::USK)
            {
                return true;
            }
            for (size_t i = 0; i < ratingAgeDataSize; ++i)
            {
                if (age < 0 || age < ratingAgeData[i])
                {
                    age = ratingAgeData[i];
                }
            }
            // 何も設定が無い(未審査である)場合は制限あり扱いにする
            // (1つでも ratingAgeData[i] >= 0 があれば age >= 0 になる)
            if (age < 0)
            {
                return true;
            }
        }

        // settingAge は「x歳より上が対象のソフトは遊べない」のxを表す
        // age は「x歳以上対象ソフト」のxを表す
        return (settingAge < static_cast<uint8_t>(age));
    }
}

SettingsManager::SettingsManager(void* threadStackBuffer, size_t stackBufferSize) NN_NOEXCEPT :
    m_TaskThread(),
    m_EventHolderExit(),
    m_EventHolderSaveGeneralSettings(),
    m_EventHolderSaveAppplicationSettings(),
    m_EventHolderSaveExemptionSettings(),
    m_EventHolderApplicationLaunched(),
    m_EventHolderApplicationTerminated(),
    m_EventHolderApplicationAddedToList(),
    m_EventHolderApplicationRemovedFromList(),
    m_EventHolderApplicationDownloadStarted(),
    m_EventHolderWrongPinCode(),
    m_IsDisabledAll(false),
    m_IsEnabledOnNextBoot(false),
    m_HasLaunchedApplicationFreeCommunication(false),
    m_IsDisabledAllForRecovery(false),
    m_ApplicationIdLaunched(nn::ncm::ApplicationId::GetInvalidId()),
    m_ApplicationIdTerminated(nn::ncm::ApplicationId::GetInvalidId()),
    m_TimestampTerminated(),
    m_ApplicationIdAddedToList(nn::ncm::ApplicationId::GetInvalidId()),
    m_ApplicationIdRemovedFromList(nn::ncm::ApplicationId::GetInvalidId())
{
    nn::os::InitializeMultiWait(&m_MultiWait);

    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderExit.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderSaveGeneralSettings.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderSaveAppplicationSettings.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderSaveExemptionSettings.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderApplicationLaunched.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderApplicationTerminated.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderApplicationAddedToList.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderApplicationRemovedFromList.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderApplicationDownloadStarted.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderWrongPinCode.m_WaitHolder);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_TaskThread, StaticThreadProc, this,
        threadStackBuffer, stackBufferSize, NN_SYSTEM_THREAD_PRIORITY(pctl, SettingsManager)));
    nn::os::SetThreadNamePointer(&m_TaskThread, NN_SYSTEM_THREAD_NAME(pctl, SettingsManager));

    // 乱数生成器を初期化(ちゃんとした乱数でなくてよいのでTickで初期化)
    s_Mt.seed(static_cast<unsigned int>(nn::os::GetSystemTick().GetInt64Value()));

    m_GeneralSettings.Reset();
    m_FreeCommunicationSettings.Clear();
    m_ExemptionSettings.Clear();

    // 0 (false) == SafetyLevel_None, 初期値 == SafetyLevel_None 相当の設定値、であることの確認
    NN_STATIC_ASSERT(0 == static_cast<int>(SizedSafetyLevel_None));
    NN_STATIC_ASSERT(0 == static_cast<int>(SafetyLevel_None));
    NN_STATIC_ASSERT(0 == static_cast<int>(nn::ns::RatingOrganization::CERO)); // CurrentSettings::ratingOrganization の初期値
    // (SafetyLevel_None (== 0) の設定値が 0 であることの確認)
    // ※ const データではあるものの static assert が使えないので便宜上ここで確認
    NN_SDK_ASSERT(common::SafetyLevelPresetSettings[SafetyLevel_None].ratingAge == 0);
    NN_SDK_ASSERT(common::SafetyLevelPresetSettings[SafetyLevel_None].freeCommunicationRestriction == false);
    NN_SDK_ASSERT(common::SafetyLevelPresetSettings[SafetyLevel_None].snsPostRestriction == false);
    NN_SDK_ASSERT(common::SafetyLevelPresetSettingsForJapan[SafetyLevel_None].ratingAge == 0);
    NN_SDK_ASSERT(common::SafetyLevelPresetSettingsForJapan[SafetyLevel_None].freeCommunicationRestriction == false);
    NN_SDK_ASSERT(common::SafetyLevelPresetSettingsForJapan[SafetyLevel_None].snsPostRestriction == false);
}

SettingsManager::~SettingsManager() NN_NOEXCEPT
{
    m_EventHolderExit.m_Event.Signal();
    //nn::os::WaitThread(&m_TaskThread);
    nn::os::DestroyThread(&m_TaskThread);

    nn::os::UnlinkAllMultiWaitHolder(&m_MultiWait);
    nn::os::FinalizeMultiWait(&m_MultiWait);
}

void SettingsManager::StartThread() NN_NOEXCEPT
{
    nn::os::StartThread(&m_TaskThread);
}

void SettingsManager::ThreadProc() NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        auto holder = nn::os::WaitAny(&m_MultiWait);
        NN_SDK_ASSERT_NOT_NULL(holder);
        if (holder == &m_EventHolderExit.m_WaitHolder)
        {
            // Clear しない
            break;
        }
        else if (holder == &m_EventHolderSaveGeneralSettings.m_WaitHolder)
        {
            NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);

            m_EventHolderSaveGeneralSettings.m_Event.Clear();
            SaveGeneralSettingsNoLock();
        }
        else if (holder == &m_EventHolderSaveAppplicationSettings.m_WaitHolder)
        {
            NN_UTIL_LOCK_GUARD(m_MutexApplicationSettings);

            m_EventHolderSaveAppplicationSettings.m_Event.Clear();
            SaveApplicationSettingsNoLock();
        }
        else if (holder == &m_EventHolderSaveExemptionSettings.m_WaitHolder)
        {
            NN_UTIL_LOCK_GUARD(m_MutexExemptionSettings);

            m_EventHolderSaveExemptionSettings.m_Event.Clear();
            SaveExemptionSettingsNoLock();
        }
        else if (holder == &m_EventHolderApplicationLaunched.m_WaitHolder)
        {
            nn::ncm::ApplicationId appId;
            bool hasFreeCommunication;
            {
                NN_UTIL_LOCK_GUARD(m_MutexMessage);

                appId = m_ApplicationIdLaunched;
                hasFreeCommunication = m_HasLaunchedApplicationFreeCommunication;
                m_ApplicationIdLaunched = nn::ncm::ApplicationId::GetInvalidId();
                m_EventHolderApplicationLaunched.m_Event.Clear();
            }
            // ロックを解除してから呼び出す
            OnApplicationLaunched(appId, hasFreeCommunication);
        }
        else if (holder == &m_EventHolderApplicationTerminated.m_WaitHolder)
        {
            nn::ncm::ApplicationId appId;
            nn::util::optional<nn::time::PosixTime> timestamp;
            {
                NN_UTIL_LOCK_GUARD(m_MutexMessage);

                appId = m_ApplicationIdTerminated;
                timestamp = m_TimestampTerminated;
                m_ApplicationIdTerminated = nn::ncm::ApplicationId::GetInvalidId();
                m_TimestampTerminated = nn::util::nullopt;
                m_EventHolderApplicationTerminated.m_Event.Clear();
            }
            OnApplicationTerminated(appId, timestamp);
        }
        else if (holder == &m_EventHolderApplicationRejected.m_WaitHolder)
        {
            nn::ncm::ApplicationId appId;
            {
                NN_UTIL_LOCK_GUARD(m_MutexMessage);

                appId = m_ApplicationIdRejected;
                m_ApplicationIdRejected = nn::ncm::ApplicationId::GetInvalidId();
                m_EventHolderApplicationRejected.m_Event.Clear();
            }
            OnApplicationRejected(appId);
        }
        else if (holder == &m_EventHolderApplicationAddedToList.m_WaitHolder)
        {
            nn::ncm::ApplicationId appId;
            {
                NN_UTIL_LOCK_GUARD(m_MutexMessage);

                appId = m_ApplicationIdAddedToList;
                m_ApplicationIdAddedToList = nn::ncm::ApplicationId::GetInvalidId();
                m_EventHolderApplicationAddedToList.m_Event.Clear();
            }
            OnApplicationAddedToList(appId);
        }
        else if (holder == &m_EventHolderApplicationRemovedFromList.m_WaitHolder)
        {
            nn::ncm::ApplicationId appId;
            {
                NN_UTIL_LOCK_GUARD(m_MutexMessage);

                appId = m_ApplicationIdRemovedFromList;
                m_ApplicationIdRemovedFromList = nn::ncm::ApplicationId::GetInvalidId();
                m_EventHolderApplicationRemovedFromList.m_Event.Clear();
            }
            OnApplicationRemovedFromList(appId);
        }
        else if (holder == &m_EventHolderApplicationDownloadStarted.m_WaitHolder)
        {
            nn::ncm::ApplicationId appId;
            {
                NN_UTIL_LOCK_GUARD(m_MutexMessage);

                appId = m_ApplicationIdDownloadStarted;
                m_ApplicationIdDownloadStarted = nn::ncm::ApplicationId::GetInvalidId();
                m_EventHolderApplicationDownloadStarted.m_Event.Clear();
            }
            OnApplicationDownloadStarted(appId);
        }
        else if (holder == &m_EventHolderWrongPinCode.m_WaitHolder)
        {
            {
                NN_UTIL_LOCK_GUARD(m_MutexMessage);
                m_EventHolderWrongPinCode.m_Event.Clear();
            }
            OnWrongPinCode();
        }
    }
} // NOLINT(impl/function_size)

void SettingsManager::OnApplicationLaunched(nn::ncm::ApplicationId id, bool hasFreeCommunication) NN_NOEXCEPT
{
    if (IsWatcherAvailable())
    {
        ApplicationStateManager::RecordPlayDataBasedEvents();
        ApplicationStateManager::AddApplicationLaunchEvent(id);
    }

    // UGCリストの更新
    if (hasFreeCommunication)
    {
        AddFreeCommunicationApplicationList(id);
    }
    else
    {
        DeleteFreeCommunicationApplicationList(id);
    }
}

void SettingsManager::OnApplicationTerminated(nn::ncm::ApplicationId id, nn::util::optional<nn::time::PosixTime> timestamp) NN_NOEXCEPT
{
    if (IsWatcherAvailable())
    {
        // Suspend/Resumeイベントの消化
        ApplicationStateManager::RecordPlayDataBasedEvents();
        // 終了イベントを発行
        ApplicationStateManager::AddApplicationTerminateEvent(id, timestamp);
    }
}

void SettingsManager::OnApplicationRejected(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    if (IsWatcherAvailable())
    {
        ApplicationStateManager::AddApplicationRejectEvent(id);
    }
}

void SettingsManager::OnApplicationAddedToList(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    if (IsWatcherAvailable())
    {
        // リスト追加イベントを発行
        g_pWatcher->GetWatcherEventStorage().AddNewManagedApplicationEvent(id);
    }
}

void SettingsManager::OnApplicationRemovedFromList(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    if (IsWatcherAvailable())
    {
        // リスト削除イベントを発行
        g_pWatcher->GetWatcherEventStorage().AddRemoveManagedApplicationEvent(id);
    }
}

void SettingsManager::OnApplicationDownloadStarted(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    if (IsWatcherAvailable())
    {
        // ダウンロード開始イベントを発行
        g_pWatcher->GetWatcherEventStorage().AddDownloadStartedEvent(id);
    }
}

void SettingsManager::OnWrongPinCode() NN_NOEXCEPT
{
    if (IsWatcherAvailable())
    {
        // 解除コード間違いイベントを発行
        g_pWatcher->GetWatcherEventStorage().AddWrongPinCodeEvent();
    }
}


nn::Result SettingsManager::ConfirmFreeCommunicationRestriction(const ParentalControlStates& states, nn::ncm::ApplicationId applicationId, bool freeCommunicationFlag) const NN_NOEXCEPT
{
    // 一時解除していれば制限しない
    if (states.temporaryUnlocked)
    {
        NN_RESULT_SUCCESS;
    }
    // アプリ自身に利用フラグが無ければ制限しない
    if (!freeCommunicationFlag)
    {
        NN_RESULT_SUCCESS;
    }

    {
        NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
        if (m_GeneralSettings.current.pinCode[0] == 0)
        {
            NN_RESULT_SUCCESS;
        }
        // 既定値がfalseであれば制限なし
        if (!m_GeneralSettings.current.isFreeCommunicationRestrictedByDefault)
        {
            NN_RESULT_SUCCESS;
        }
    }

    // 制限対象外リストにあれば制限なし
    {
        NN_UTIL_LOCK_GUARD(m_MutexExemptionSettings);
        bool exemptValue;
        if (m_ExemptionSettings.GetExemptedValueByApplicationId(&exemptValue, applicationId)
            && exemptValue)
        {
            NN_RESULT_SUCCESS;
        }
    }

    // アプリ別制限値、または既定の制限値を確認してその結果を返す
    // (個別設定が見つからなければ制限ありとする)
    {
        NN_UTIL_LOCK_GUARD(m_MutexApplicationSettings);
        bool value;
        if (!m_FreeCommunicationSettings.GetRestrictedValueByApplicationId(&value, applicationId))
        {
            value = true;
        }
        if (value)
        {
            NN_RESULT_THROW(nn::pctl::ResultFreeCommunicationRestricted());
        }
    }
    // 制限が無ければSuccess
    NN_RESULT_SUCCESS;
}

nn::Result SettingsManager::ConfirmLaunchApplicationPermission(const ParentalControlStates& states, nn::ncm::ApplicationId applicationId, const int8_t* ratingAgeData, size_t ratingAgeDataSize, bool freeCommunicationFlag) const NN_NOEXCEPT
{
    // 一時解除していれば制限しない
    if (states.temporaryUnlocked)
    {
        NN_RESULT_SUCCESS;
    }

    // レーティングによる制限がある場合のみブロックする
    // (UGCのブロックは実際に使用するタイミングで行い、ここでは行わない)
    NN_UNUSED(freeCommunicationFlag);

    uint8_t settingAge;
    nn::ns::RatingOrganization organization;
    {
        NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
        if (m_GeneralSettings.current.pinCode[0] == 0)
        {
            NN_RESULT_SUCCESS;
        }
        // 制限対象外リストにあれば制限なし
        {
            bool exemptValue;
            if (m_ExemptionSettings.GetExemptedValueByApplicationId(&exemptValue, applicationId)
                && exemptValue)
            {
                NN_RESULT_SUCCESS;
            }
        }

        settingAge = m_GeneralSettings.current.ratingAge;
        if (m_GeneralSettings.current.isDefaultRatingOrganizationSet != 0)
        {
            organization = m_GeneralSettings.current.ratingOrganization;
        }
        else
        {
            // 設定されていない場合は既定値を用いる
            organization = common::GetDefaultRatingOrganizationFromSystemSettings();
        }
    }

    if (IsRestrictedByRating(settingAge, organization, ratingAgeData, ratingAgeDataSize))
    {
        NN_RESULT_THROW(nn::pctl::ResultRestrictedByRating());
    }
    NN_RESULT_SUCCESS;
}

nn::Result SettingsManager::ConfirmResumeApplicationPermission(const ParentalControlStates& states, nn::ncm::ApplicationId applicationId, const int8_t* ratingAgeData, size_t ratingAgeDataSize, bool freeCommunicationFlag) const NN_NOEXCEPT
{
    // 一時解除状態になっている(改めてセットされている場合も含む)の場合は制限しない
    if (states.temporaryUnlocked)
    {
        NN_RESULT_SUCCESS;
    }

    uint8_t settingAge;
    nn::ns::RatingOrganization organization;
    bool isFreeCommunicationRestrictedByDefault;
    bool isStereoVisionRestricted;
    {
        NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
        // ペアコン未設定の場合は成功
        if (m_GeneralSettings.current.pinCode[0] == 0)
        {
            NN_RESULT_SUCCESS;
        }
        settingAge = m_GeneralSettings.current.ratingAge;
        if (m_GeneralSettings.current.isDefaultRatingOrganizationSet != 0)
        {
            organization = m_GeneralSettings.current.ratingOrganization;
        }
        else
        {
            // 設定されていない場合は既定値を用いる
            organization = common::GetDefaultRatingOrganizationFromSystemSettings();
        }
        isFreeCommunicationRestrictedByDefault = m_GeneralSettings.current.isFreeCommunicationRestrictedByDefault;
        isStereoVisionRestricted = m_GeneralSettings.current.isStereoVisionRestricted;
    }

    // 制限対象外リストにあれば制限なし
    {
        NN_UTIL_LOCK_GUARD(m_MutexExemptionSettings);
        bool exemptValue;
        if (m_ExemptionSettings.GetExemptedValueByApplicationId(&exemptValue, applicationId)
            && exemptValue)
        {
            NN_RESULT_SUCCESS;
        }
    }

    // 返す制限値の優先順位はレーティング→UGCとするため
    // 先にレーティングをチェックする
    if (IsRestrictedByRating(settingAge, organization, ratingAgeData, ratingAgeDataSize))
    {
        NN_RESULT_THROW(nn::pctl::ResultRestrictedByRating());
    }

    // UGCの場合、まだ一度もUGCを利用していなければ制限を行わない
    if (states.freeCommunicationUsed)
    {
        // (アプリケーションにフラグがセットされている場合のみ制限が起きる)
        if (freeCommunicationFlag && isFreeCommunicationRestrictedByDefault)
        {
            NN_UTIL_LOCK_GUARD(m_MutexApplicationSettings);
            bool value;
            if (!m_FreeCommunicationSettings.GetRestrictedValueByApplicationId(&value, applicationId))
            {
                value = true;
            }
            if (value)
            {
                NN_RESULT_THROW(nn::pctl::ResultFreeCommunicationRestricted());
            }
        }
    }

    // StereoVisionは一度でも利用していれば制限を行う
    if (states.stereoVisionUsed)
    {
        NN_RESULT_THROW_UNLESS(!isStereoVisionRestricted,
            nn::pctl::ResultStereoVisionRestricted());
    }

    // いずれの制限もかかっていないor不要であるのでSuccessを返す
    NN_RESULT_SUCCESS;
}

nn::Result SettingsManager::ConfirmSnsPostPermission(const ParentalControlStates& states) const NN_NOEXCEPT
{
    if (states.temporaryUnlocked)
    {
        NN_RESULT_SUCCESS;
    }
    {
        NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
        // ペアコン未設定の場合は成功
        if (m_GeneralSettings.current.pinCode[0] == 0)
        {
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_THROW_UNLESS(m_GeneralSettings.current.isSnsPostRestricted == 0,
            nn::pctl::ResultSnsPostRestricted());
    }
    NN_RESULT_SUCCESS;
}

nn::Result SettingsManager::ConfirmStereoVisionPermission(const ParentalControlStates& states) const NN_NOEXCEPT
{
    // 一時解除していれば制限しない
    if (states.temporaryUnlocked)
    {
        NN_RESULT_SUCCESS;
    }
    {
        NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
        // ペアコン未設定の場合は成功
        if (m_GeneralSettings.current.pinCode[0] == 0)
        {
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_THROW_UNLESS(m_GeneralSettings.current.isStereoVisionRestricted == 0,
            nn::pctl::ResultStereoVisionRestricted());
    }
    NN_RESULT_SUCCESS;
}

nn::Result SettingsManager::ConfirmPlayableApplicationVideo(const ParentalControlStates& states, nn::ncm::ApplicationId applicationId, const int8_t* ratingAgeData, size_t ratingAgeDataSize) const NN_NOEXCEPT
{
    // 一時解除していれば制限しない
    if (states.temporaryUnlocked)
    {
        NN_RESULT_SUCCESS;
    }

    uint8_t settingAge;
    nn::ns::RatingOrganization organization;
    {
        NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
        if (m_GeneralSettings.current.pinCode[0] == 0)
        {
            NN_RESULT_SUCCESS;
        }
        settingAge = m_GeneralSettings.current.ratingAge;
        if (m_GeneralSettings.current.isDefaultRatingOrganizationSet != 0)
        {
            organization = m_GeneralSettings.current.ratingOrganization;
        }
        else
        {
            // 設定されていない場合は既定値を用いる
            organization = common::GetDefaultRatingOrganizationFromSystemSettings();
        }
    }

    // 制限対象外リストにあれば制限なし
    {
        NN_UTIL_LOCK_GUARD(m_MutexExemptionSettings);
        bool exemptValue;
        if (m_ExemptionSettings.GetExemptedValueByApplicationId(&exemptValue, applicationId)
            && exemptValue)
        {
            NN_RESULT_SUCCESS;
        }
    }

    if (IsRestrictedByRating(settingAge, organization, ratingAgeData, ratingAgeDataSize))
    {
        NN_RESULT_THROW(nn::pctl::ResultPlayingApplicationVideoRestricted());
    }
    NN_RESULT_SUCCESS;
}

nn::Result SettingsManager::ConfirmSystemSettingsPermission() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    // ペアコン未設定の場合は成功
    if (m_GeneralSettings.current.pinCode[0] == 0)
    {
        NN_RESULT_SUCCESS;
    }
    // 設定されている場合は常に制限あり
    NN_RESULT_THROW(nn::pctl::ResultRestricted());
}

nn::Result SettingsManager::ConfirmStereoVisionRestrictionConfigurable() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    // ペアコン未設定の場合は設定不可
    if (m_GeneralSettings.current.pinCode[0] == 0)
    {
        NN_RESULT_THROW(nn::pctl::ResultRestrictionNotEnabled());
    }
    // NOTE: 将来別の理由ができた場合は nn::pctl::ResultConfigurationNotAllowed の子Resultを返す
    NN_RESULT_SUCCESS;
}


nn::Result SettingsManager::CheckPinCode(const char* pinCode, size_t pinCodeWithNullCharLength) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);

    NN_RESULT_THROW_UNLESS(m_GeneralSettings.current.pinCode[0] != 0, nn::pctl::ResultInvalidState());
    NN_RESULT_THROW_UNLESS(IsEqualPinCodeNoLock(pinCode, pinCodeWithNullCharLength),
        nn::pctl::ResultWrongPinCode());
    NN_RESULT_SUCCESS;
}


int SettingsManager::GetRestrictedFeatures() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    int feature = Feature::Feature_None;
    if (m_GeneralSettings.current.pinCode[0] != 0)
    {
        if (m_GeneralSettings.current.ratingAge != 0)
        {
            feature |= Feature::Feature_LaunchApplicationByRating;
        }
        if (m_GeneralSettings.current.isSnsPostRestricted)
        {
            feature |= Feature::Feature_SnsPost;
        }
        // 既定値が制限ありであれば有効として扱う
        if (m_GeneralSettings.current.isFreeCommunicationRestrictedByDefault)
        {
            feature |= Feature::Feature_FreeCommunication;
        }
    }
    return feature;
}

nn::ns::RatingOrganization SettingsManager::GetDefaultRatingOrganization() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    if (m_GeneralSettings.current.isDefaultRatingOrganizationSet != 0)
    {
        return m_GeneralSettings.current.ratingOrganization;
    }
    else
    {
        // 設定されていない場合は既定値を用いるが、ここではまだ保存しない
        return common::GetDefaultRatingOrganizationFromSystemSettings();
    }
}

void SettingsManager::SetDefaultRatingOrganization(nn::ns::RatingOrganization organization) NN_NOEXCEPT
{
    // 無効化中は設定変更を無視する
    if (m_IsDisabledAll)
    {
        return;
    }

    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    m_GeneralSettings.current.ratingOrganization = organization;
    m_GeneralSettings.current.isDefaultRatingOrganizationSet = 1;
    PostSaveGeneralSettingsNoLock();
}

void SettingsManager::SetSafetyLevel(SafetyLevel safetyLevel) NN_NOEXCEPT
{
    // 無効化中は設定変更を無視する
    if (m_IsDisabledAll)
    {
        return;
    }

    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    InitializeDefaultRatingOrganizationIfNeeded();

    m_GeneralSettings.current.safetyLevel = static_cast<SizedSafetyLevel>(safetyLevel);
    if (m_GeneralSettings.current.safetyLevel == SizedSafetyLevel_Custom)
    {
        // 安心レベルがカスタムになる場合は現在値に反映させる
        m_GeneralSettings.current.ratingAge = m_GeneralSettings.custom.ratingAge;
        m_GeneralSettings.current.isSnsPostRestricted = m_GeneralSettings.custom.isSnsPostRestricted;
        m_GeneralSettings.current.isFreeCommunicationRestrictedByDefault = m_GeneralSettings.custom.isFreeCommunicationRestrictedByDefault;
    }
    else
    {
        // 安心レベルがカスタムではない場合は現在値にプリセット値を反映させる
        // プリセット値を用いる際のリージョンはこの時点のものを用いる
        // (リージョン変更があっても値を追随させない)
        nn::settings::system::RegionCode region;
        nn::settings::system::GetRegionCode(&region);
        auto& settings = (region == nn::settings::system::RegionCode::RegionCode_Japan ?
            common::SafetyLevelPresetSettingsForJapan[safetyLevel] :
            common::SafetyLevelPresetSettings[safetyLevel]);
        m_GeneralSettings.current.ratingAge = settings.ratingAge;
        m_GeneralSettings.current.isSnsPostRestricted = settings.snsPostRestriction ? 1 : 0;
        m_GeneralSettings.current.isFreeCommunicationRestrictedByDefault = settings.freeCommunicationRestriction ? 1 : 0;
    }
    PostSaveGeneralSettingsNoLock();
}

void SettingsManager::SetCustomSafetyLevelSettings(const SafetyLevelSettings& settings) NN_NOEXCEPT
{
    // 無効化中は設定変更を無視する
    if (m_IsDisabledAll)
    {
        return;
    }

    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    m_GeneralSettings.custom.ratingAge = settings.ratingAge;
    m_GeneralSettings.custom.isSnsPostRestricted = settings.snsPostRestriction ? 1 : 0;
    m_GeneralSettings.custom.isFreeCommunicationRestrictedByDefault = settings.freeCommunicationRestriction ? 1 : 0;
    // 安心レベルがカスタムになっている場合は現在値にも反映させる
    if (m_GeneralSettings.current.safetyLevel == SizedSafetyLevel_Custom)
    {
        InitializeDefaultRatingOrganizationIfNeeded();

        m_GeneralSettings.current.ratingAge = m_GeneralSettings.custom.ratingAge;
        m_GeneralSettings.current.isSnsPostRestricted = m_GeneralSettings.custom.isSnsPostRestricted;
        m_GeneralSettings.current.isFreeCommunicationRestrictedByDefault = m_GeneralSettings.custom.isFreeCommunicationRestrictedByDefault;
    }
    PostSaveGeneralSettingsNoLock();
}

void SettingsManager::SetStereoVisionRestriction(bool restricted) NN_NOEXCEPT
{
    // 無効化中は設定変更を無視する
    if (m_IsDisabledAll)
    {
        return;
    }
    // 暗証番号設定が無い場合は無視する
    if (m_GeneralSettings.current.pinCode[0] == 0)
    {
        return;
    }

    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    m_GeneralSettings.current.isStereoVisionRestricted = restricted ? 1 : 0;

    PostSaveGeneralSettingsNoLock();
}

void SettingsManager::DeleteSettings() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);

    // レーティング団体含めて全て初期化する(current / custom)
    m_GeneralSettings.Reset();
    // リストはそのままとする
    m_FreeCommunicationSettings.ResetValues();
    m_ExemptionSettings.ResetValues();

    PostSaveGeneralSettingsNoLock();
}

nn::Result SettingsManager::SetPinCode(const char* pinCode, size_t pinCodeWithNullCharLength) NN_NOEXCEPT
{
    // 無効化中は設定変更を無視する
    if (m_IsDisabledAll)
    {
        NN_RESULT_SUCCESS;
    }

    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    if (pinCodeWithNullCharLength == 0)
    {
        bool hasPinCode = (m_GeneralSettings.current.pinCode[0] != 0);
        // 未設定状態
        // ※ 他の設定値はそのままとする
        m_GeneralSettings.current.pinCode[0] = 0;

        if (hasPinCode)
        {
            // 解除コード削除時にシグナルも落とす
            watcher::NetworkManager::ClearUnlinkedFlag();
            // 通常は未連携時は影響がないが、デバッグ処理でプレイタイマーを稼働させている場合があることも考慮して対応
            watcher::WatcherEventManager::ClearPlayTimerSettingsFromSaveData();
        }
    }
    else
    {
        NN_RESULT_THROW_UNLESS(IsValidPinCode(pinCode, pinCodeWithNullCharLength - 1), nn::pctl::ResultInvalidValue());

        // MaxPinCodeLength未満の場合はNULL文字込みでコピーするので -1 しない
        if (pinCodeWithNullCharLength > MaxPinCodeLength)
        {
            pinCodeWithNullCharLength = MaxPinCodeLength;
        }
        std::memcpy(m_GeneralSettings.current.pinCode, pinCode, sizeof(char) * pinCodeWithNullCharLength);
    }
    PostSaveGeneralSettingsNoLock();
    OnChangedPinCode();
    NN_RESULT_SUCCESS;
}

size_t SettingsManager::GetPinCode(char* outPinCode, size_t maxLengthWithNull) NN_NOEXCEPT
{
    if (!maxLengthWithNull)
    {
        return 0;
    }

    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    auto length = static_cast<size_t>(nn::util::Strnlen(m_GeneralSettings.current.pinCode, static_cast<int>(MaxPinCodeLength)));
    if (length == 0)
    {
        outPinCode[0] = 0;
        return 0;
    }
    if (length > maxLengthWithNull - 1)
    {
        length = maxLengthWithNull - 1;
    }
    std::memcpy(outPinCode, m_GeneralSettings.current.pinCode, sizeof(char) * length);
    outPinCode[length] = 0;
    return length;
}

void SettingsManager::GenerateRandomPinCodeAndNoRestriction(Settings& outSettings) NN_NOEXCEPT
{
    // 乱数で生成
    char newCode[MaxPinCodeLength];
    while (NN_STATIC_CONDITION(true))
    {
        // 桁数(固定)
        int length = RandomPinCodeLength;
        // 8桁にする場合は「newCode[length] = 0」の行が不要
        NN_STATIC_ASSERT(RandomPinCodeLength < MaxPinCodeLength);
        newCode[length] = 0;
        bool isAllSame = true;
        char prevChar = 0;
        // 各桁の数字を生成
        std::uniform_int_distribution<int> dist('0', '9');
        while (length--)
        {
            newCode[length] = static_cast<char>(dist(s_Mt));
            // すべての桁が同じ数字にならないようにチェック
            isAllSame = isAllSame && (prevChar == 0 || prevChar == newCode[length]);
            prevChar = newCode[length];
        }
        if (!isAllSame)
        {
            break;
        }
    }

    // 「制限なし」プリセット値を反映させる
    // (「制限なし」はリージョン別で変わることはないはず)
    auto& settings = common::SafetyLevelPresetSettings[SafetyLevel::SafetyLevel_None];
    outSettings.Reset();
    outSettings.current.ratingAge = settings.ratingAge;
    outSettings.current.isSnsPostRestricted = settings.snsPostRestriction ? 1 : 0;
    outSettings.current.isFreeCommunicationRestrictedByDefault = settings.freeCommunicationRestriction ? 1 : 0;
    NN_STATIC_ASSERT(sizeof(outSettings.current.pinCode) == sizeof(newCode));
    std::memcpy(outSettings.current.pinCode, newCode, sizeof(newCode));
    // レーティング団体は既定値を用いる
    outSettings.current.ratingOrganization = common::GetDefaultRatingOrganizationFromSystemSettings();
    outSettings.current.isDefaultRatingOrganizationSet = 1;
}

void SettingsManager::GetSettings(Settings& outSettings) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    std::memcpy(&outSettings, &m_GeneralSettings, sizeof(m_GeneralSettings));

    // 設定されていない場合は既定値を取得するが、ここでは保存しない
    if (m_GeneralSettings.current.isDefaultRatingOrganizationSet == 0)
    {
        outSettings.current.ratingOrganization = common::GetDefaultRatingOrganizationFromSystemSettings();
        outSettings.current.isDefaultRatingOrganizationSet = 1;
    }
}

void SettingsManager::SetSettings(const Settings& settings) NN_NOEXCEPT
{
    // 無効化中は設定変更を無視する
    if (m_IsDisabledAll)
    {
        return;
    }

    NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
    NN_STATIC_ASSERT((std::is_same<decltype(m_GeneralSettings), Settings>::value));
    std::memcpy(&m_GeneralSettings, &settings, sizeof(settings));
    PostSaveGeneralSettingsNoLock();
    OnChangedPinCode();
}

void SettingsManager::UpdateFreeCommunicationApplicationList(const nn::pctl::FreeCommunicationApplicationInfo* pTitleInfoArray, size_t count) NN_NOEXCEPT
{
    if (count == 0)
    {
        return;
    }
    // 無効化中は設定変更を無視する
    if (m_IsDisabledAll)
    {
        return;
    }

    NN_UTIL_LOCK_GUARD(m_MutexApplicationSettings);
    for (size_t i = 0; i < count; ++i)
    {
        m_FreeCommunicationSettings.SetRestrictedValueByApplicationId(pTitleInfoArray[i]);
    }
    PostSaveApplicationSettingsNoLock();
}

void SettingsManager::AddFreeCommunicationApplicationList(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    // 無効化中は設定変更を無視する
    if (m_IsDisabledAll)
    {
        return;
    }

    {
        NN_UTIL_LOCK_GUARD(m_MutexApplicationSettings);
        // 初期設定値は true
        if (!m_FreeCommunicationSettings.AddRestrictedValue(id, true))
        {
            return;
        }
        PostSaveApplicationSettingsNoLock();
        NN_DETAIL_PCTL_TRACE("AddFreeCommunicationApplicationList: title added (0x%016llX)\n",
            id.value);
    }

    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        m_ApplicationIdAddedToList = id;
        m_EventHolderApplicationAddedToList.m_Event.Signal();
    }
}

void SettingsManager::DeleteFreeCommunicationApplicationList(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_MutexApplicationSettings);
        if (!m_FreeCommunicationSettings.DeleteRestrictedValueByApplicationId(id))
        {
            return;
        }
        PostSaveApplicationSettingsNoLock();
    }

    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        m_ApplicationIdRemovedFromList = id;
        m_EventHolderApplicationRemovedFromList.m_Event.Signal();
    }
}

void SettingsManager::ClearFreeCommunicationApplicationList() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexApplicationSettings);
    m_FreeCommunicationSettings.Clear();
    PostSaveApplicationSettingsNoLock();
}

void SettingsManager::UpdateExemptApplicationList(const nn::pctl::ExemptApplicationInfo* pTitleInfoArray, size_t count) NN_NOEXCEPT
{
    if (count == 0)
    {
        return;
    }
    // 無効化中は設定変更を無視する
    if (m_IsDisabledAll)
    {
        return;
    }

    NN_UTIL_LOCK_GUARD(m_MutexExemptionSettings);
    for (size_t i = 0; i < count; ++i)
    {
        m_ExemptionSettings.SetExemptedValueByApplicationId(pTitleInfoArray[i]);
    }
    PostSaveExemptionSettingsNoLock();
}

void SettingsManager::AddExemptApplicationList(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    // 無効化中は設定変更を無視する
    if (m_IsDisabledAll)
    {
        return;
    }

    {
        NN_UTIL_LOCK_GUARD(m_MutexExemptionSettings);
        // 初期設定値は false
        if (!m_ExemptionSettings.AddExemptedValue(id, false))
        {
            return;
        }
        PostSaveExemptionSettingsNoLock();
        NN_DETAIL_PCTL_TRACE("AddExemptApplicationList: title added (0x%016llX)\n",
            id.value);
    }

    // イベントログ追記はしない
}

void SettingsManager::DeleteExemptApplicationList(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_MutexExemptionSettings);
        if (!m_ExemptionSettings.DeleteExemptedValueByApplicationId(id))
        {
            return;
        }
        PostSaveExemptionSettingsNoLock();
    }

    // イベントログ追記はしない
}

void SettingsManager::ClearExemptApplicationList() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexExemptionSettings);
    m_ExemptionSettings.Clear();
    PostSaveExemptionSettingsNoLock();
}

void SettingsManager::PostApplicationLaunched(nn::ncm::ApplicationId id, bool hasFreeCommunication) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    m_HasLaunchedApplicationFreeCommunication = hasFreeCommunication;
    m_ApplicationIdLaunched = id;
    m_EventHolderApplicationLaunched.m_Event.Signal();
}

void SettingsManager::PostApplicationTerminated(nn::ncm::ApplicationId id, nn::util::optional<nn::time::PosixTime> timestamp) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    m_ApplicationIdTerminated = id;
    m_TimestampTerminated = timestamp;
    m_EventHolderApplicationTerminated.m_Event.Signal();
}

void SettingsManager::PostApplicationRejected(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    m_ApplicationIdRejected = id;
    m_EventHolderApplicationRejected.m_Event.Signal();
}

void SettingsManager::PostApplicationDownloadStarted(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    m_ApplicationIdDownloadStarted = id;
    m_EventHolderApplicationDownloadStarted.m_Event.Signal();
}

void SettingsManager::PostWrongPinCode() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    m_EventHolderWrongPinCode.m_Event.Signal();
}

bool SettingsManager::StoreAllSettings(nn::ovln::format::PctlSettingTypeFlagSet* outChangedType,
    const Settings* pSettings,
    const FreeCommunicationApplicationSettings* pFreeCommunicationSettings,
    void(*unknownTitleCallback)(nn::ncm::ApplicationId id),
    void(*notTitleInListCallback)(nn::ncm::ApplicationId id),
    const ExemptApplicationSettings* pExemptionSettings) NN_NOEXCEPT
{
    if (m_IsDisabledAll)
    {
        return false;
    }
    if (pSettings->current.pinCode[0] == 0)
    {
        return false;
    }
    size_t newPinCodeLength = static_cast<size_t>(nn::util::Strnlen(pSettings->current.pinCode, MaxPinCodeLength));
    if (!IsValidPinCode(pSettings->current.pinCode, newPinCodeLength))
    {
        return false;
    }

    nn::settings::system::RegionCode region;
    nn::settings::system::GetRegionCode(&region);

    bool isPinCodeChanged = false;
    {
        NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
        // 各種設定値が変更されるかどうかのチェック
        if (static_cast<size_t>(nn::util::Strnlen(m_GeneralSettings.current.pinCode, MaxPinCodeLength)) != newPinCodeLength ||
            std::strncmp(m_GeneralSettings.current.pinCode, pSettings->current.pinCode, newPinCodeLength) != 0)
        {
            isPinCodeChanged = true;
            outChangedType->Set<nn::ovln::format::PctlSettingTypeFlag::PinCode>(true);
        }
        if (m_GeneralSettings.current.isDefaultRatingOrganizationSet == 0 ||
            m_GeneralSettings.current.ratingOrganization != pSettings->current.ratingOrganization)
        {
            outChangedType->Set<nn::ovln::format::PctlSettingTypeFlag::Rating>(true);
        }
        if (m_GeneralSettings.current.safetyLevel != pSettings->current.safetyLevel)
        {
            outChangedType->Set<nn::ovln::format::PctlSettingTypeFlag::SafetyLevel>(true);
        }
        if (pSettings->current.safetyLevel == SizedSafetyLevel_Custom)
        {
            if (m_GeneralSettings.current.ratingAge != pSettings->custom.ratingAge)
            {
                outChangedType->Set<nn::ovln::format::PctlSettingTypeFlag::Rating>(true);
            }
            if (m_GeneralSettings.current.isSnsPostRestricted != pSettings->custom.isSnsPostRestricted)
            {
                outChangedType->Set<nn::ovln::format::PctlSettingTypeFlag::SnsPost>(true);
            }
            if (m_GeneralSettings.current.isFreeCommunicationRestrictedByDefault != pSettings->custom.isFreeCommunicationRestrictedByDefault)
            {
                outChangedType->Set<nn::ovln::format::PctlSettingTypeFlag::FreeCommunication>(true);
            }
        }
        else
        {
            // 安心レベルがカスタムではない場合は現在値にプリセット値を反映させる
            auto& settings = (region == nn::settings::system::RegionCode::RegionCode_Japan ?
                common::SafetyLevelPresetSettingsForJapan[pSettings->current.safetyLevel] :
                common::SafetyLevelPresetSettings[pSettings->current.safetyLevel]);
            if (m_GeneralSettings.current.ratingAge != settings.ratingAge)
            {
                outChangedType->Set<nn::ovln::format::PctlSettingTypeFlag::Rating>(true);
            }
            if (m_GeneralSettings.current.isSnsPostRestricted != (settings.snsPostRestriction ? 1 : 0))
            {
                outChangedType->Set<nn::ovln::format::PctlSettingTypeFlag::SnsPost>(true);
            }
            if (m_GeneralSettings.current.isFreeCommunicationRestrictedByDefault != (settings.freeCommunicationRestriction ? 1 : 0))
            {
                outChangedType->Set<nn::ovln::format::PctlSettingTypeFlag::FreeCommunication>(true);
            }
        }
        std::memcpy(&m_GeneralSettings, pSettings, sizeof(Settings));

        // 安心レベルから現在の設定値を更新する
        if (m_GeneralSettings.current.safetyLevel == SizedSafetyLevel_Custom)
        {
            m_GeneralSettings.current.ratingAge = m_GeneralSettings.custom.ratingAge;
            m_GeneralSettings.current.isSnsPostRestricted = m_GeneralSettings.custom.isSnsPostRestricted;
            m_GeneralSettings.current.isFreeCommunicationRestrictedByDefault = m_GeneralSettings.custom.isFreeCommunicationRestrictedByDefault;
        }
        else
        {
            // 安心レベルがカスタムではない場合は現在値にプリセット値を反映させる
            auto& settings = (region == nn::settings::system::RegionCode::RegionCode_Japan ?
                common::SafetyLevelPresetSettingsForJapan[pSettings->current.safetyLevel] :
                common::SafetyLevelPresetSettings[pSettings->current.safetyLevel]);
            m_GeneralSettings.current.ratingAge = settings.ratingAge;
            m_GeneralSettings.current.isSnsPostRestricted = settings.snsPostRestriction ? 1 : 0;
            m_GeneralSettings.current.isFreeCommunicationRestrictedByDefault = settings.freeCommunicationRestriction ? 1 : 0;
        }

        SaveGeneralSettingsNoLock();
    }

    {
        NN_UTIL_LOCK_GUARD(m_MutexApplicationSettings);
        if (m_FreeCommunicationSettings.UpdateDataFromList(*pFreeCommunicationSettings,
            unknownTitleCallback, notTitleInListCallback))
        {
            outChangedType->Set<nn::ovln::format::PctlSettingTypeFlag::FreeCommunication>(true);
        }
        std::memcpy(&m_FreeCommunicationSettings, pFreeCommunicationSettings, sizeof(FreeCommunicationApplicationSettings));
        SaveApplicationSettingsNoLock();
    }

    {
        NN_UTIL_LOCK_GUARD(m_MutexExemptionSettings);
        if (m_ExemptionSettings.UpdateDataFromList(*pExemptionSettings, nullptr, nullptr))
        {
            outChangedType->Set<nn::ovln::format::PctlSettingTypeFlag::ExemptAppList>(true);
        }
        std::memcpy(&m_ExemptionSettings, pExemptionSettings, sizeof(ExemptApplicationSettings));
        SaveExemptionSettingsNoLock();
    }

    if (isPinCodeChanged)
    {
        OnChangedPinCode();
    }
    return true;
} // NOLINT(impl/function_size)

bool SettingsManager::DisableAllFeatures(bool isRecoveryMode) NN_NOEXCEPT
{
    if (m_IsDisabledAll)
    {
        // 非Recoveryモードからの要求であるか、以前の無効化がRecoveryモードであれば
        // (既に無効化されている場合は)なにもせず false を返す
        if (!isRecoveryMode || m_IsDisabledAllForRecovery)
        {
            return false;
        }
    }
    m_IsDisabledAll = true;
    m_IsDisabledAllForRecovery = isRecoveryMode;
    // Recoveryモードの場合は再起動後も無効化されるようにする
    if (isRecoveryMode)
    {
        // ファイル作成を同期処理で行う
        {
            common::FileStream stream;
            NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::OpenWrite(&stream, DisableAllFeaturesFileName, 0));
            // 0バイトで作成したので何もせずにそのまま閉じる
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    }
    // 設定を無いものとして扱う
    {
        NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
        m_EventHolderSaveGeneralSettings.m_Event.Clear();
        m_GeneralSettings.Reset();
    }
    {
        NN_UTIL_LOCK_GUARD(m_MutexApplicationSettings);
        m_EventHolderSaveAppplicationSettings.m_Event.Clear();
        m_FreeCommunicationSettings.Clear();
    }
    {
        NN_UTIL_LOCK_GUARD(m_MutexExemptionSettings);
        m_EventHolderSaveExemptionSettings.m_Event.Clear();
        m_ExemptionSettings.Clear();
    }
    return true;
}

bool SettingsManager::PostEnableAllFeatures() NN_NOEXCEPT
{
    if (!m_IsDisabledAll || m_IsEnabledOnNextBoot)
    {
        return false;
    }
    m_IsEnabledOnNextBoot = true;
    // ファイルを削除する(エラーは無視する)
    if (common::FileSystem::Delete(DisableAllFeaturesFileName).IsSuccess())
    {
        common::FileSystem::Commit();
    }
    return true;
}

void SettingsManager::LoadSettings() NN_NOEXCEPT
{
    // DisableAllFeatures の読み込み
    {
        common::FileStream stream;
        auto result = common::FileSystem::OpenRead(&stream, DisableAllFeaturesFileName);
        NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result),
            "Unexpected result for file open: %08x", result.GetInnerValueForDebug());
        // ファイルがあるかどうかで判断
        m_IsDisabledAll = result.IsSuccess();
        // 無効化状態である場合はRecoveryモードとする
        m_IsDisabledAllForRecovery = m_IsDisabledAll;
    }
    // 無効化されている場合は設定を使わない
    if (m_IsDisabledAll)
    {
        return;
    }

    // GeneralSettings の読み込み
    {
        NN_UTIL_LOCK_GUARD(m_MutexGeneralSettings);
        Settings data = {}; // 警告対策も兼ねて初期化する
        size_t size = 0;
        {
            common::FileStream stream;
            auto result = common::FileSystem::OpenRead(&stream, GeneralSettingsFileName);
            NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result),
                "Unexpected result for file open: %08x", result.GetInnerValueForDebug());
            if (result.IsSuccess())
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Read(&size, 0, &data, sizeof(data)));
            }
        }
        if (size == sizeof(old::SettingsVer1))
        {
            auto& oldData = *reinterpret_cast<old::SettingsVer1*>(&data);
            if (IsValidPinCode(oldData.current.pinCode, static_cast<size_t>(nn::util::Strnlen(oldData.current.pinCode, MaxPinCodeLength))))
            {
                NN_STATIC_ASSERT(sizeof(m_GeneralSettings.current.pinCode) == sizeof(oldData.current.pinCode));
                std::memcpy(m_GeneralSettings.current.pinCode, oldData.current.pinCode, sizeof(m_GeneralSettings.current.pinCode));
                m_GeneralSettings.current.safetyLevel = oldData.current.safetyLevel;
                m_GeneralSettings.current.ratingAge = oldData.current.ratingAge;
                m_GeneralSettings.current.ratingOrganization = oldData.current.ratingOrganization;
                m_GeneralSettings.current.isDefaultRatingOrganizationSet = oldData.current.isDefaultRatingOrganizationSet;
                m_GeneralSettings.current.isSnsPostRestricted = oldData.current.isSnsPostRestricted;
                m_GeneralSettings.current.isFreeCommunicationRestrictedByDefault = oldData.current.isFreeCommunicationRestrictedByDefault;

                m_GeneralSettings.custom.ratingAge = oldData.custom.ratingAge;
                m_GeneralSettings.custom.isSnsPostRestricted = oldData.custom.isSnsPostRestricted;
                m_GeneralSettings.custom.isFreeCommunicationRestrictedByDefault = oldData.custom.isFreeCommunicationRestrictedByDefault;

                m_GeneralSettings.current.isStereoVisionRestricted = 0;
                m_GeneralSettings.current._padding1 = 0;
                m_GeneralSettings.current.dataVersion = SettingsDataVersionCurrent;
                std::memset(m_GeneralSettings.current._padding2, 0, sizeof(m_GeneralSettings.current._padding2));
            }
        }
        else if (size == sizeof(data))
        {
            if (data.current.dataVersion == SettingsDataVersionCurrent &&
                IsValidPinCode(data.current.pinCode, static_cast<size_t>(nn::util::Strnlen(data.current.pinCode, MaxPinCodeLength))))
            {
                std::memcpy(&m_GeneralSettings, &data, sizeof(m_GeneralSettings));
            }
        }
        // (適切に読み込めなかった場合は初期値のままとする)
    }
    // ApplicationSettings の読み込み
    {
        NN_UTIL_LOCK_GUARD(m_MutexApplicationSettings);
        size_t size = 0;
        // 一時変数を使わず直接読み込む
        {
            common::FileStream stream;
            auto result = common::FileSystem::OpenRead(&stream, ApplicationSettingsFileName);
            NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result),
                "Unexpected result for file open: %08x", result.GetInnerValueForDebug());
            if (result.IsSuccess())
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Read(&size, 0,
                    &m_FreeCommunicationSettings, sizeof(m_FreeCommunicationSettings)));
            }
            // (適切に読み込めなかった場合は初期値のままとする)
        }
    }
    // ExemptionSettings の読み込み
    {
        NN_UTIL_LOCK_GUARD(m_MutexExemptionSettings);
        size_t size = 0;
        {
            common::FileStream stream;
            auto result = common::FileSystem::OpenRead(&stream, ExemptionSettingsFileName);
            NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result),
                "Unexpected result for file open: %08x", result.GetInnerValueForDebug());
            if (result.IsSuccess())
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Read(&size, 0,
                    &m_ExemptionSettings, sizeof(m_ExemptionSettings)));
            }
            // (適切に読み込めなかった場合は初期値のままとする)
        }
    }
}

void SettingsManager::InitializeDefaultRatingOrganizationIfNeeded() NN_NOEXCEPT
{
    // 無効化中は設定変更しない
    if (!m_IsDisabledAll && m_GeneralSettings.current.isDefaultRatingOrganizationSet == 0)
    {
        m_GeneralSettings.current.ratingOrganization = common::GetDefaultRatingOrganizationFromSystemSettings();
        m_GeneralSettings.current.isDefaultRatingOrganizationSet = 1;
        PostSaveGeneralSettingsNoLock();
    }
}

void SettingsManager::SaveGeneralSettingsNoLock() NN_NOEXCEPT
{
    // 無効化中は保存を行わない
    if (m_IsDisabledAll)
    {
        return;
    }

    m_GeneralSettings.current.dataVersion = SettingsDataVersionCurrent;
    {
        common::FileStream stream;
        NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::OpenWrite(&stream, GeneralSettingsFileName, sizeof(m_GeneralSettings)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Write(0, &m_GeneralSettings, sizeof(m_GeneralSettings)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Flush());
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
}

void SettingsManager::SaveApplicationSettingsNoLock() NN_NOEXCEPT
{
    // 無効化中は保存を行わない
    if (m_IsDisabledAll)
    {
        return;
    }

    {
        common::FileStream stream;
        NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::OpenWrite(&stream, ApplicationSettingsFileName, sizeof(m_FreeCommunicationSettings)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Write(0, &m_FreeCommunicationSettings, sizeof(m_FreeCommunicationSettings)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Flush());
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
}

void SettingsManager::SaveExemptionSettingsNoLock() NN_NOEXCEPT
{
    // 無効化中は保存を行わない
    if (m_IsDisabledAll)
    {
        return;
    }

    {
        common::FileStream stream;
        NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::OpenWrite(&stream, ExemptionSettingsFileName, sizeof(m_ExemptionSettings)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Write(0, &m_ExemptionSettings, sizeof(m_ExemptionSettings)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Flush());
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
}

}}}}}
