﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/json/pctl_JsonWebApi.h>
#include <nn/pctl/detail/service/json/pctl_JsonDataHandlerProxy.h>
#include <nn/pctl/detail/service/json/pctl_JsonErrorHandler.h>
#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_IntUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace json {

namespace
{
    struct ResponseHeaderEndCallbackData
    {
        JsonDataHandlerProxy* pProxy;
        JsonDataHandler* pHandler;
        JsonWebErrorDataHandler* pErrorDataHandler;
        int statusCode;
    };

    struct SendDataCallbackData
    {
        JsonStructuredWriter* pWriter;
        nn::Result result;
    };

    bool ResponseHeaderEndCallback(int statusCode, void* param) NN_NOEXCEPT
    {
        ResponseHeaderEndCallbackData* pData = static_cast<ResponseHeaderEndCallbackData*>(param);
        pData->statusCode = statusCode;
        if (statusCode / 100 == 2)
        {
            pData->pProxy->SetHandler(pData->pHandler);
        }
        else
        {
            pData->pProxy->SetHandler(pData->pErrorDataHandler);
        }
        return true;
    }

    size_t SendDataCallbackWithWriter(char* buffer, size_t size, size_t count, void* param)
    {
        SendDataCallbackData* pData = reinterpret_cast<SendDataCallbackData*>(param);
        size_t totalWritten = 0;
        size *= count;
        count = 1;
        while (NN_STATIC_CONDITION(true))
        {
            size_t written;
            auto result = pData->pWriter->FillData(&written, buffer, size);
            switch (result)
            {
                case JsonStructuredWriter::WriteResult::Succeeded:
#if 0 // POSTされるJSONデータの確認用
                    NN_DETAIL_PCTL_TRACE("[pctl][SendData] %.*s\n", written, buffer);
#endif
                    totalWritten += written;
                    buffer += written;
                    size -= written;
                    break;
                case JsonStructuredWriter::WriteResult::Error:
                    pData->result = nn::pctl::ResultHttpErrorWriteError();
                    return 0;
                case JsonStructuredWriter::WriteResult::NoMoreData:
                    return totalWritten;
                case JsonStructuredWriter::WriteResult::NeedMoreBuffer:
                    if (totalWritten == 0)
                    {
                        NN_SDK_ASSERT(false, "Too small buffer size (maybe the JSON data is too large)");
                        pData->result = nn::pctl::ResultHttpErrorWriteError();
                        return 0;
                    }
                    return totalWritten;
                default:
                    NN_UNEXPECTED_DEFAULT;
            }
        }
    }
}

nn::Result JsonWebErrorDataHandler::CallErrorFunction(int statusCode, nn::Result parseResult) NN_NOEXCEPT
{
    nn::Result defaultResult = common::ConvertStatusCodeToResult(statusCode);
    if (defaultResult.IsSuccess())
    {
        defaultResult = parseResult;
    }

    if (m_ErrorCallback != nullptr)
    {
        return m_ErrorCallback(GetParam(), statusCode, defaultResult);
    }
    return defaultResult;
}

nn::Result ParseWebStream(JsonDataHandler* pHandler,
    JsonWebErrorDataHandler* pErrorDataHandler,
    JsonHttpInputStream* pStream,
    void* jsonValueBuffer,
    size_t jsonValueBufferSize,
    void* streamWorkBuffer,
    size_t streamWorkBufferSize,
    const common::Cancelable* pCancelable) NN_NOEXCEPT
{
    JsonDataHandlerProxy proxy;

    proxy.SetHandler(pHandler);
    proxy.SetCancelable(pCancelable);
    proxy.SetOverflowDetector(pStream->GetOverflowDetector());

    ResponseHeaderEndCallbackData data = {
        &proxy,
        pHandler,
        pErrorDataHandler,
        0
    };

    pStream->SetBuffer(static_cast<JsonInputStream::Ch*>(jsonValueBuffer), jsonValueBufferSize,
        static_cast<JsonInputStream::Ch*>(streamWorkBuffer), streamWorkBufferSize);
    pStream->SetCancelable(pCancelable);
    pStream->GetRequest().RegisterResponseHeaderEndCallback(ResponseHeaderEndCallback, &data);
    pStream->GetRequest().AddRequestHeader("Accept: application/json");

    NullAllocator allocator;

    const unsigned int ParseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;

    RAPIDJSON_NAMESPACE::ParseResult result = JsonReader(&allocator, 1).Parse<ParseFlags>(*pStream, proxy);

    NN_RESULT_THROW_UNLESS(!pCancelable->IsCanceled(), ResultCanceled());

    NN_RESULT_DO(pStream->GetLastError());

    auto parseResult = detail::service::json::HandleJsonError(result);
    if (pErrorDataHandler != nullptr)
    {
        NN_RESULT_DO(pErrorDataHandler->CallErrorFunction(data.statusCode, parseResult));
    }
    else
    {
        NN_RESULT_DO(common::ConvertStatusCodeToResult(data.statusCode));
        NN_RESULT_DO(parseResult);
    }

    NN_RESULT_SUCCESS;
}

nn::Result ParseWebStreamWithPostJSON(JsonStructuredWriter* pWriter,
    JsonDataHandler* pHandler,
    JsonWebErrorDataHandler* pErrorDataHandler,
    JsonHttpInputStream* pStream,
    void* jsonValueBuffer,
    size_t jsonValueBufferSize,
    void* streamWorkBuffer,
    size_t streamWorkBufferSize,
    const common::Cancelable* pCancelable) NN_NOEXCEPT
{
    JsonDataHandlerProxy proxy;

    proxy.SetHandler(pHandler);
    proxy.SetCancelable(pCancelable);
    proxy.SetOverflowDetector(pStream->GetOverflowDetector());

    ResponseHeaderEndCallbackData data = {
        &proxy,
        pHandler,
        pErrorDataHandler,
        0
    };

    SendDataCallbackData sendData = {
        pWriter,
        nn::ResultSuccess()
    };

    size_t contentLength = 0;
    {
        char contentLengthHeader[27]; // strlen("Content-Length: ") + strlen("4294967296") + 1 (4294967296 == 0xFFFFFFFF + 1)
        NN_RESULT_THROW_UNLESS(pWriter->CalculateTotalSize(&contentLength) == JsonStructuredWriter::WriteResult::Succeeded,
            nn::pctl::ResultInvalidOperation());
        NN_RESULT_THROW_UNLESS(nn::util::IsIntValueRepresentable<uint32_t>(contentLength),
            nn::pctl::ResultInvalidOperation());
        nn::util::SNPrintf(contentLengthHeader, std::extent<decltype(contentLengthHeader)>::value,
            "Content-Length: %lu", static_cast<uint32_t>(contentLength));
        pStream->GetRequest().AddRequestHeader(contentLengthHeader);
    }
    pStream->SetBuffer(static_cast<JsonInputStream::Ch*>(jsonValueBuffer), jsonValueBufferSize,
        static_cast<JsonInputStream::Ch*>(streamWorkBuffer), streamWorkBufferSize);
    pStream->SetCancelable(pCancelable);
    pStream->GetRequest().RegisterResponseHeaderEndCallback(ResponseHeaderEndCallback, &data);

    pWriter->InitializeHandler();
    NN_RESULT_DO(pStream->GetRequest().RegisterSendDataCallback(SendDataCallbackWithWriter, &sendData, contentLength));

    NullAllocator allocator;

    const unsigned int ParseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;

    RAPIDJSON_NAMESPACE::ParseResult result = JsonReader(&allocator, 1).Parse<ParseFlags>(*pStream, proxy);

    NN_RESULT_THROW_UNLESS(!pCancelable->IsCanceled(), ResultCanceled());

    // 書き込みでエラーがあればそれを先に返す
    NN_RESULT_DO(sendData.result);

    NN_RESULT_DO(pStream->GetLastError());

    auto parseResult = detail::service::json::HandleJsonError(result);
    if (pErrorDataHandler != nullptr)
    {
        NN_RESULT_DO(pErrorDataHandler->CallErrorFunction(data.statusCode, parseResult));
    }
    else
    {
        NN_RESULT_DO(common::ConvertStatusCodeToResult(data.statusCode));
        NN_RESULT_DO(parseResult);
    }

    NN_RESULT_SUCCESS;
}

}}}}}
