﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/json/pctl_JsonDataHandlerProxy.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace json {

JsonDataHandlerProxy::JsonDataHandlerProxy() NN_NOEXCEPT :
    m_Handler(nullptr),
    m_Cancelable(nullptr),
    m_OverflowDetector(nullptr)
{
}

void JsonDataHandlerProxy::SetHandler(JsonDataHandler* handler) NN_NOEXCEPT
{
    m_Handler = handler;
}

void JsonDataHandlerProxy::SetCancelable(const common::Cancelable* cancelable) NN_NOEXCEPT
{
    m_Cancelable = cancelable;
}

void JsonDataHandlerProxy::SetOverflowDetector(const std::atomic<bool>* overflowDetector) NN_NOEXCEPT
{
    m_OverflowDetector = overflowDetector;
}

bool JsonDataHandlerProxy::StartObject() NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->StartObject();
    }

    return true;
}

bool JsonDataHandlerProxy::EndObject(RAPIDJSON_NAMESPACE::SizeType numObjects) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->EndObject(static_cast<size_t>(numObjects));
    }

    return true;
}

bool JsonDataHandlerProxy::StartArray() NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->StartArray();
    }

    return true;
}

bool JsonDataHandlerProxy::EndArray(RAPIDJSON_NAMESPACE::SizeType numElements) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->EndArray(static_cast<size_t>(numElements));
    }

    return true;
}

bool JsonDataHandlerProxy::Key(const char* key, RAPIDJSON_NAMESPACE::SizeType length, bool copy) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->Key(key, length, copy);
    }

    return true;
}

bool JsonDataHandlerProxy::Null() NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->Null();
    }

    return true;
}

bool JsonDataHandlerProxy::Bool(bool value) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->Bool(value);
    }

    return true;
}

bool JsonDataHandlerProxy::Int(int32_t value) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->Int(value);
    }

    return true;
}

bool JsonDataHandlerProxy::Uint(uint32_t value) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->Uint(value);
    }

    return true;
}

bool JsonDataHandlerProxy::Int64(int64_t value) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->Int64(value);
    }

    return true;
}

bool JsonDataHandlerProxy::Uint64(uint64_t value) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->Uint64(value);
    }

    return true;
}

bool JsonDataHandlerProxy::Double(double value) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->Double(value);
    }

    return true;
}

bool JsonDataHandlerProxy::String(const char* value, RAPIDJSON_NAMESPACE::SizeType length, bool copy) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->String(value, static_cast<size_t>(length), copy);
    }

    return true;
}

bool JsonDataHandlerProxy::IsCanceled() const NN_NOEXCEPT
{
    if (m_Cancelable)
    {
        return m_Cancelable->IsCanceled();
    }

    return false;
}

bool JsonDataHandlerProxy::IsOverflowed() const NN_NOEXCEPT
{
    if (m_OverflowDetector)
    {
        return *m_OverflowDetector;
    }

    return false;
}

}}}}}
