﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/common/pctl_ClientCertificate.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/result/result_HandlingUtility.h>

// NOTE: hfio経由で読み込む場合は以下を修正
//#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
//#define NN_DETAIL_PCTL_CONFIG_LOAD_CERT_FROM_HOST_FS  1
//#endif

#if defined(NN_DETAIL_PCTL_CONFIG_LOAD_CERT_FROM_HOST_FS)
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_Mount.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_LockGuard.h>
#endif

namespace nn { namespace pctl { namespace detail { namespace service { namespace common {

namespace
{
#if defined(NN_DETAIL_PCTL_CONFIG_LOAD_CERT_FROM_HOST_FS)
    static const char MountName[] = "certvol";
    static const char MountPoint[] = "C:\\siglo_ssl";
    static const char CertPath[] = "certvol:/client_cert.p12";
    static const char PasswordPath[] = "certvol:/client_cert.key";

    static nn::Result LoadCertificateFromHostFs(nn::ssl::CertStoreId* pCertStoreId, nn::ssl::Context* pContext, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(nn::os::SdkMutexType, s_Lock, =NN_OS_SDK_MUTEX_INITIALIZER());
        NN_UTIL_LOCK_GUARD(s_Lock);

        NN_RESULT_DO(fs::MountHost(MountName, MountPoint));
        NN_UTIL_SCOPE_EXIT
        {
            fs::Unmount(MountName);
        };

        fs::FileHandle certFile;
        int64_t certSize;
        fs::FileHandle passwordFile;
        int64_t passwordSize;

        /* ファイル長の検査　*/
        NN_RESULT_DO(fs::OpenFile(&certFile, CertPath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT { nn::fs::CloseFile(certFile); };
        NN_RESULT_DO(fs::GetFileSize(&certSize, certFile));

        NN_RESULT_DO(fs::OpenFile(&passwordFile, PasswordPath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT { nn::fs::CloseFile(passwordFile); };
        NN_RESULT_DO(fs::GetFileSize(&passwordSize, passwordFile));

        NN_RESULT_THROW_UNLESS(static_cast<size_t>(certSize + passwordSize + 2) <= bufferSize, nn::pctl::ResultUnexpected());

        /* 読み込み　*/
        char* cert = static_cast<char*>(buffer);
        char* password = cert + certSize + 1;
        NN_RESULT_DO(fs::ReadFile(certFile, 0, cert, static_cast<size_t>(certSize)));
        NN_RESULT_DO(fs::ReadFile(passwordFile, 0, password, static_cast<size_t>(passwordSize)));
        cert[certSize] = password[passwordSize] = '\0';

        NN_RESULT_DO(
            pContext->ImportClientPki(
                pCertStoreId, cert, password,
                static_cast<uint32_t>(certSize), static_cast<uint32_t>(passwordSize))
            );

        NN_RESULT_SUCCESS;
    }
#endif // defined(NN_DETAIL_PCTL_CONFIG_LOAD_CERT_FROM_HOST_FS)
}

nn::Result InitializeSslContext(nn::ssl::Context* pContext) NN_NOEXCEPT
{
    NN_RESULT_DO(pContext->Create(nn::ssl::Context::SslVersion_Auto));

    ssl::CertStoreId certStoreId;
    auto result = pContext->RegisterInternalPki(&certStoreId, ssl::Context::InternalPki_DeviceClientCertDefault);
    if (result.IsFailure())
    {
#if defined(NN_DETAIL_PCTL_CONFIG_LOAD_CERT_FROM_HOST_FS)
        static const size_t BufferSizeForCert = 4096;
        void* certBuffer = AllocateMemoryBlock(BufferSizeForCert);
        NN_RESULT_THROW_UNLESS(certBuffer != nullptr, result);
        NN_UTIL_SCOPE_EXIT { FreeMemoryBlock(certBuffer); };
        NN_RESULT_THROW_UNLESS(LoadCertificateFromHostFs(&certStoreId, pContext, certBuffer, BufferSizeForCert).IsSuccess(), result);
#else
        NN_RESULT_THROW(result);
#endif
    }
    NN_UNUSED(certStoreId);

    NN_RESULT_SUCCESS;
}

}}}}}
