﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/pctl/detail/ipc/pctl_IpcSession.h>
#include <nn/pctl/detail/ipc/pctl_IParentalControlServiceFactory.sfdl.h>

#if NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS == NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_HIPC

#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/sm/sm_Result.h>

namespace nn { namespace pctl { namespace detail { namespace ipc {

namespace
{
    bool g_IsSessionInitialized = false;
    typedef std::aligned_storage<sizeof(nn::sf::HipcSimpleClientSessionManager), NN_ALIGNOF(nn::sf::HipcSimpleClientSessionManager)>::type
        SessionManagerStorage;
    nn::sf::ShimLibraryObjectHolder<IParentalControlServiceFactory>
        g_Holder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
    SessionManagerStorage g_SessionManagerStorage;
    nn::sf::ProxyObjectAllocator<SessionCountPerProcess>
        g_Allocator = NN_SF_PROXY_OBJECT_ALLOCATOR_INITIALIZER;
    IParentalControlService* g_pActiveService = nullptr;

    nn::Result InitializeDomainForAnyPort() NN_NOEXCEPT
    {
        bool isInitialized = false;
        bool isSucceeded = false;
        const char* ServiceNameList[] = {
            ServiceNameForGeneral,
            ServiceNameForSystem,
            ServiceNameForAuthentication,
            ServiceNameForRecovery
        };
        NN_STATIC_ASSERT(
            static_cast<int>(std::extent<decltype(ServiceNameList)>::value) == static_cast<int>(ServiceIndex_Count)
        );
        SessionManagerStorage tempSessionManagerStorage;
        NN_STATIC_ASSERT(sizeof(tempSessionManagerStorage) <= 1024); // それほどスタックを消費しないはず

        g_Allocator.Initialize();
        auto pManager = new (static_cast<void*>(&g_SessionManagerStorage)) nn::sf::HipcSimpleClientSessionManager();

        NN_UTIL_SCOPE_EXIT
        {
            if (!isSucceeded)
            {
                if (isInitialized)
                {
                    nn::sf::ReleaseSharedObject(g_pActiveService);
                    g_pActiveService = nullptr;
                    g_Holder.FinalizeHolder();
                    pManager->Finalize();
                }
                pManager->~HipcSimpleClientSessionManager();
                g_Allocator.Finalize();
            }
        };

        // すべてのポートに対して初期化を試みる
        for (auto name : ServiceNameList)
        {
            nn::sf::HipcSimpleClientSessionManager* pCurrentManager;
            // 既に1つ初期化済みの場合は一時的な SessionManager を新たに作る
            // (すぐに破棄する)
            if (isInitialized)
            {
                pCurrentManager = new (static_cast<void*>(&tempSessionManagerStorage)) nn::sf::HipcSimpleClientSessionManager();
            }
            else
            {
                pCurrentManager = pManager;
            }
            NN_UTIL_SCOPE_EXIT
            {
                if (pCurrentManager != nullptr)
                {
                    pCurrentManager->Finalize();
                    if (pCurrentManager != pManager)
                    {
                        pCurrentManager->~HipcSimpleClientSessionManager();
                    }
                }
            };
            nn::sf::SharedPointer<IParentalControlServiceFactory> pFactory;
            auto r = pCurrentManager->InitializeByName<IParentalControlServiceFactory, nn::sf::MemoryResourceAllocationPolicy>(
                &pFactory, g_Allocator.GetMemoryResource(), name
                );
            if (r.IsSuccess())
            {
                nn::sf::SharedPointer<IParentalControlService> pService;
                NN_RESULT_DO(pFactory->CreateServiceWithoutInitialize(&pService, 0));
                // まだ g_Holder にオブジェクトを割り当てていない場合は割り当てる
                // すでに g_Holder に割り当て済みの場合は既にサービスオブジェクトを生成済みであり、
                // その場合プロセス側で capability のマージが行われるので2つ目以降を破棄する
                if (!isInitialized)
                {
                    NN_SDK_ASSERT(g_pActiveService == nullptr);
                    g_Holder.InitializeHolderDirectly(std::move(pFactory));
                    g_pActiveService = pService.Detach();
                    isInitialized = true;
                    // このセッションは使い続けるので破棄しないようにする
                    NN_SDK_ASSERT(pCurrentManager == pManager);
                    pCurrentManager = nullptr;
                }
                // pService が Detach されていなければここで破棄される
            }
            // pFactory が move されていなければここで破棄される
        }

        NN_RESULT_THROW_UNLESS(isInitialized, nn::sm::ResultNotPermitted());

        NN_SDK_ASSERT_NOT_NULL(g_pActiveService);
        NN_RESULT_DO(g_pActiveService->Initialize());

        isSucceeded = true;

        NN_RESULT_SUCCESS;
    }
}

// Finalize を行う場合は以下の処理
#if defined(NN_DETAIL_PCTL_CONFIG_USE_FINALIZE_SESSION)
void FinalizeIpcSessionImpl() NN_NOEXCEPT
{
    if (!g_IsSessionInitialized)
    {
        return;
    }

    if (g_pActiveService != nullptr)
    {
        nn::sf::ReleaseSharedObject(g_pActiveService);
        g_pActiveService = nullptr;
    }

    g_Holder.FinalizeHolder();
    reinterpret_cast<nn::sf::HipcSimpleClientSessionManager*>(&g_SessionManagerStorage)->~HipcSimpleClientSessionManager();
    g_Allocator.Finalize();

    g_IsSessionInitialized = false;
}
#endif

// NOTE: nninitInitializeSdkModule -> nn::pctl::Initialize から呼び出されるため、
// 扱う変数が static initializer を利用しないようにする
nn::Result InitializeIpcSessionIfNeeded() NN_NOEXCEPT
{
    if (g_IsSessionInitialized)
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(InitializeDomainForAnyPort());

    g_IsSessionInitialized = true;

    NN_RESULT_SUCCESS;
}

nn::Result InitializeWatcherIpcSessionIfNeeded() NN_NOEXCEPT
{
    // do nothing
    NN_RESULT_SUCCESS;
}

IParentalControlService* GetServiceImpl() NN_NOEXCEPT
{
    return g_pActiveService;
}

}}}}

#endif // NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS == NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_HIPC
