﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    pci_Tegra124RootComplex.cpp
 * @brief   TK1 Platform PCIe Root Complex
 */
#include "pcie_Tegra.h"

namespace nn { namespace pcie { namespace driver { namespace detail {

#define NN_PCI_TEGRA124_ROOTCOMPLEX_CFG     \
    {                                       \
        BusSpeed_EGen2,                     \
        2,                                  \
                                            \
        {                                   \
            {                               \
                true,                       \
                false,                      \
                2                           \
            },                              \
            {                               \
                true,                       \
                false,                      \
                1                           \
            }                               \
        }                                   \
    }

class Driver;

class Tegra124RootComplex : public RootComplex
{
public:
    NN_IMPLICIT Tegra124RootComplex(Driver *pDriver, Config *pCfg);
    ~Tegra124RootComplex();
    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;

private:
    enum ResetType
    {
        PCIEXCLK_RST,
        AFI_RST,
        PCIE_RST
    };
    enum PortOperation
    {
        PORT_OP_DISABLE,
        PORT_OP_ENABLE,
        PORT_OP_INIT,
        PORT_OP_RESET,
        PORT_OP_GET_STATE
    };
    enum InterruptEvent
    {
        InterruptEvent_Msi,
        InterruptEvent_Int,
        InterruptEvent_Wake,
        InterruptEvent_Max,
    };
    enum Params
    {
        Params_ConfigSpaceSize   = 0x04000000,
        Params_IoSpaceSize       = 0x00010000,
        Params_PrefetchMemSize   = 0x02000000,
        Params_NoPrefetchMemSize = 0x02000000,
        Params_MsiBufferSize     = 0x10000,
    };


    // Waiting for PLL
    static const uint32_t MaxPllWaitTimeInUs  = 1000000;
    static const uint32_t PllPollIntervalInUs = MaxPllWaitTimeInUs / 10;

    // Interrupt vectors
    static const os::InterruptName WakeInterruptName = 32 + 100;
    static const os::InterruptName MsiInterruptName = 32 + 99;
    static const os::InterruptName MainInterruptName = 32 + 98;

    // IO VA
    static const nn::dd::DeviceVirtualAddress IoVaHeapBase = 0x80000000;
    static const nn::dd::DeviceVirtualAddress IoVaHeapSize = 0x60000000;
    static const nn::dd::DeviceVirtualAddress IoVaInternalBase = 0xF0000000;

    nn::os::InterruptEventType  m_InterruptEvent[InterruptEvent_Max];

    // MSI DMA buffer
    void       *m_pMsiBuffer;

    // Register virtual base addresses
    uintptr_t  m_PcieA1Base;
    uintptr_t  m_PcieA2Base;
    uintptr_t  m_PcieA3Base;
    uintptr_t  m_PcieCtrl0CfgBase;
    uintptr_t  m_PcieCtrl1CfgBase;
    uintptr_t  m_PciePca0Base;
    uintptr_t  m_PciePca1Base;
    uintptr_t  m_PciePadsBase;
    uintptr_t  m_PcieAfiBase;
    uintptr_t  m_ClkRstBase;
    uintptr_t  m_RtcBase;
    uintptr_t  m_XusbPadctlBase;
    uintptr_t  m_PmcBase;

    //Config Space(within A2)
    ResourceAddr m_ConfigSpaceStart;
    ResourceAddr m_ConfigSpaceEnd;
    uintptr_t    m_ConfigSpaceStartPhys;

    //I/O Space(within A2)
    ResourceAddr m_IoSpaceStart;
    uintptr_t    m_IoSpaceStartPhys;

    //Prefetchable Memory(within A3)
    ResourceAddr m_PrefetchMemStart;
    uintptr_t    m_PrefetchMemStartPhys;

    // Non-Prefetchable Memory(within A3)
    ResourceAddr m_NoPrefetchMemStart;
    uintptr_t    m_NoPrefetchMemStartPhys;

    // IRQ Managers
    static const int32_t msiIrqMgrPoolId = 0;
    static const int32_t intPort0IrqMgrBasePoolId = 0x10;
    static const int32_t intPort1IrqMgrBasePoolId = 0x10 + AFI_INT_TOTAL_VECTORS;
    typedef IrqManager<MaxFunctionsPerIrq, 1, EndpointFunction *, 0> PortIntxIrqMgrType;
    PortIntxIrqMgrType m_port0IntIrqMgrs[AFI_INT_TOTAL_VECTORS];
    PortIntxIrqMgrType m_port1IntIrqMgrs[AFI_INT_TOTAL_VECTORS];
    IrqManager<AFI_MSI_TOTAL_VECTORS, nn::pcie::MaxIrqVectorsPerDeviceFunction,
        EndpointFunction *, msiIrqMgrPoolId> m_msiIrqMgr;

    Result SetReset(ResetType t, bool trueToPutIntoReset);
    Result SetPowerGate(int32_t bitpos, bool state);
    Result InitPCIePll();
    Result SetPhyEnable(bool enabled);
    Result EnablePortFeatures(DeviceNumber portNum);
    Result SetAspmEnable(EndpointFunction *pEf, bool isEnabled);
    Result GetDmaBusAddressRange(BusAddress *pOutBase, BusAddress *pOutSize);
    Result PortControl(DeviceNumber devNum, PortCommand portCommand);
    Result InitAFIBars();
    Result InitAFIConfigReg();
    Result EnableAFI();
    Result InitMSI();
    void   ResolveMemMap();
    Result InitInterruptVectors();
    uint32_t GetEndpointFunctionEnableMask(int32_t baseResourceIndex);

    volatile uint32_t* ComputeConfigSpaceAddress(BusNumber busNum, DeviceNumber devNum,
                                                 FunctionNumber funcNum, int32_t where);
    Result DoPortOperation(DeviceNumber portNum, PortOperation op);

    // Implement RootComplex's API
    Result Initialize();
    Result Finalize();
    Result ReadConfigSpace(BusNumber busNum, DeviceNumber devNum, FunctionNumber funcNum,
                           int32_t where, uint32_t size, void *pVal);
    Result WriteConfigSpace(BusNumber busNum, DeviceNumber devNum, FunctionNumber funcNum,
                            int32_t where, uint32_t size, uint32_t val);
    void GetRegions(Region *pIo, Region *pPrefetchMem, Region *pNoPrefetchMem);
    Result AcquireIrq(EndpointFunction *pEp, IrqOptions *pOptions, IrqProfile *pRetProfile);
    Result ReleaseIrq(IrqHandle irqHandle);
    Result SetIrqEnable(IrqHandle irqHandle, int32_t index, bool irqEnabled);
    Result GetIrqProfile(IrqHandle irqHandle, IrqProfile *pRetProfile);
    void HandleInterrupt(IntVector intVector);
    void SetErrorReportingEnable(bool enable);

    void HandleINTInterrupt();
    void DispatchIntx(PortIntxIrqMgrType *pMgr);
    void HandleSidebandMessageInterrupt(uint32_t sigRegVal);
    void HandleMSIInterrupt();

};



} // end of namespace detail
} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn



