﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "pcie_PrivateIncludes.h"

namespace nn { namespace pcie { namespace driver { namespace detail {

Result GetMemoryBusAddressess(void *address, size_t size, BusAddress *pRetBusAddress)
{
    nn::dd::PhysicalMemoryInfo physInfo;
    Result result;
    BusAddress retBusAddress = 0;
    nn::dd::InitializePhysicalMemoryInfo(&physInfo, address, size);
    if ((result = nn::dd::QueryNextPhysicalMemoryInfo(&physInfo)).IsSuccess())
    {
        if (nn::dd::GetSize(&physInfo) >= size)
        {
            retBusAddress = nn::dd::GetPhysicalAddress(&physInfo);
        }
        else
        {
            result = ResultMemoryNotContiguous();
        }
    }
    if (pRetBusAddress!=NULL)
    {
        *pRetBusAddress = retBusAddress;
    }
    return result;
}

bool IsPowerOfTwo(int32_t value)
{
    // zero is not considered a power of two
    return (value != 0 && ((value & (value - 1)) == 0));
}

// For example: Passing 0x80000001 returns 32
int32_t FindLastSetBit(int32_t mask32)
{
    int32_t bit;
    if (mask32 == 0)
    {
        return 0;
    }
    for (bit = 1; mask32 != 1; bit++)
    {
        mask32 = (uint32_t)mask32 >> 1;
    }
    return (bit);
}

// For example: Passing 0x80000001 returns 1
int32_t FindFirstSetBit(int32_t mask32)
{
    int32_t bit;
    if (mask32 == 0)
    {
        return 0;
    }
    for (bit = 1; !(mask32 & 1); bit++)
    {
        mask32 = (unsigned int)mask32 >> 1;
    }
    return bit;
}

size_t RoundupSize(size_t inputSize, size_t powerOfTwoRoundedSize)
{
    size_t size;
    size  = (inputSize + (powerOfTwoRoundedSize - 1)) &  ~(powerOfTwoRoundedSize - 1);
    return size;
}

const char* GetBusSpeedDescription(BusSpeed speed)
{
    const char * speedDescription = "GEN?";
    switch (speed)
    {
    case BusSpeed_EGen1:
        speedDescription = "GEN1";
        break;
    case BusSpeed_EGen2:
        speedDescription = "GEN2";
        break;
    default:
        break;
    }
    return speedDescription;
}

const char* GetPmStateDescription(PmState pmState)
{
    const char* pmStateDescription = "PmState?";
    switch(pmState)
    {
    case PmState_D0:
        pmStateDescription = "PmState_D0";
        break;
    case PmState_D1:
        pmStateDescription = "PmState_D1";
        break;
    case PmState_D2:
        pmStateDescription = "PmState_D2";
        break;
    case PmState_D3Hot:
        pmStateDescription = "PmState_D3Hot";
        break;
    case PmState_D3Off:
        pmStateDescription = "PmState_D3Off";
        break;
    default:
        pmStateDescription = "PmState_Invalid";
    }
    return pmStateDescription;
}


uint32_t AddSaturated(uint32_t a, uint32_t b)
{
    uint32_t c = a + b;
    if (c < a)
    {
        c = -1;
    }
    return c;
}

void IncrementSaturated(uint32_t & value)
{
    value = AddSaturated(value, 1);
}

} // end of namespace detail
} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn




