﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "pcie_PrivateIncludes.h"

namespace nn { namespace pcie { namespace driver { namespace detail {

RootComplex::RootComplex(Driver *pDriver, Config *pCfg)
    : m_IsInitialized(false)
    , m_IsResumed(false)
    , m_pDriver(pDriver)
    , m_Config(*pCfg)
    , m_ControllerType(ControllerType_Invalid)
    , m_pBus(nullptr)
{

}

RootComplex::~RootComplex()
{

}

Result RootComplex::Initialize()
{
    Result result = ResultSuccess();
    m_IsInitialized = true;
    return result;
}

Result RootComplex::Finalize()
{
    Result result = ResultSuccess();
    m_IsInitialized = false;
    return result;
}

Result RootComplex::Resume()
{
    Result result = ResultMemAllocFailure();

    if ((m_pBus = new Bus(m_pDriver, NULL, NULL, 0)) != NULL)
    {
        result = ResultSuccess();
        result = m_pBus->Initialize();
        m_pBus->SetRootComplex(this);
        m_IsResumed = true;
    }

    return result;
}

Result RootComplex::Suspend()
{
    Result result = ResultSuccess();

    if (m_pBus != NULL)
    {
        m_pBus->Finalize();
        delete m_pBus;
        m_pBus = NULL;
        m_IsResumed = false;
    }

    return result;
}

bool RootComplex::IsInitialized()
{
    return m_IsInitialized;
}

bool RootComplex::IsResumed()
{
    return m_IsResumed;
}

Result RootComplex::SetIrqEnable(IrqHandle irqHandle, bool irqEnabled)
{
    return SetIrqEnable(irqHandle, 0, irqEnabled);
}

Bus* RootComplex::GetBus()
{
    return m_pBus;
}

Result RootComplex::WriteConfigSpaceMasked32(BusNumber busNum, DeviceNumber devNum, FunctionNumber funcNum,
                                             int32_t where, uint32_t clearMask, uint32_t setMask)
{
    Result result;
    uint32_t val32 = 0;
    NN_PCIE_RETURN_UPON_ERROR(ReadConfigSpace(busNum, devNum, funcNum, where, sizeof(uint32_t), &val32));
    val32 &= ~clearMask;
    val32 |= setMask;
    result = WriteConfigSpace(busNum, devNum, funcNum, where, sizeof(uint32_t), val32);
    return result;
}

// When polarity=true, we are waiting for all mask specified bits to be cleared
// When polarity=false, we are waiting for all mask specified bits to be set
Result RootComplex::WaitPolledRegister32(volatile uint32_t *pReg, uint32_t mask,
                                         bool polarity, int64_t retryIntervalInMicroseconds,
                                         uint32_t maxWaitTimeInMicroseconds, const char *subject)
{
    Result result = ResultSuccess();
    os::Tick timeoutTick = nn::os::ConvertToTick(nn::TimeSpan::FromMicroSeconds(maxWaitTimeInMicroseconds)) +
        nn::os::GetSystemTick();
    NN_UNUSED(subject);

    while ( (polarity) ? ( ((*pReg) & mask) != 0 ) : ( ((*pReg) & mask) != mask ) )
    {
        if (nn::os::GetSystemTick() > timeoutTick)
        {
            result = ResultOperationTimeout();
            NN_PCIE_LOG_WARN("RootComplex::WaitPolledRegister32() timed out on %s.\n",
                             subject);
            break;
        }
        else
        {
            if (retryIntervalInMicroseconds > 0)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(retryIntervalInMicroseconds));
            }
        }
    }

    return result;
}

BusSpeed RootComplex::GetBusSpeedLimitPolicy()
{
    return m_Config.speedLimit;
}

Result RootComplex::CreateDmaMap(BusAddress *pReturnedBusAddressBase, int32_t procVaIndex, nn::dd::ProcessHandle procHandle,
                                 uint64_t procVa, size_t size, nn::dd::MemoryPermission cpuPermission,
                                 nn::dd::MemoryPermission devicePermission, int32_t tag)
{
    Result result = ResultSuccess();
    NN_UNUSED(cpuPermission);
    if(!(m_IoVaManager.LookupByProcVa(procVaIndex, procVa, NULL).IsSuccess()))
    {
        BusAddress busAddr = m_IoVaManager.Allocate(size, nn::pcie::MinimumDmaAddressAlignment, procVaIndex, procVa, procHandle, tag);
        if(busAddr != 0)
        {
            if((result = nn::dd::MapDeviceAddressSpaceNotAligned(&m_DeviceAddressSpace, procHandle, procVa,
                                                                 size, busAddr, devicePermission)).IsSuccess())
            {
                *pReturnedBusAddressBase = busAddr;
            }
            else
            {
                m_IoVaManager.Free(busAddr);
            }
        }
        else
        {
            result = ResultFailedDmaMappingAllocation();
        }
    }
    else
    {
        result = ResultOverlappedDmaMapping();
    }
    NN_PCIE_LOG_VERBOSE("RootComplex::CreateDmaMap(procVaIndex=%d, procHandle=%p, procVa=%p, size=%d, busAddr=%p, tag=%d) = %d:%d\n",
                        procVaIndex, reinterpret_cast<void*>(procHandle), reinterpret_cast<void*>(procVa),
                        size, reinterpret_cast<void*>(*pReturnedBusAddressBase), tag, result.GetModule(), result.GetDescription());
    return result;
}

Result RootComplex::DestroyDmaMapByBusAddress(BusAddress busAddr)
{
    Result result;
    IoVaAllocationSummary summary;
    if((result=m_IoVaManager.LookupByIoVa(busAddr, &summary)).IsSuccess())
    {
        NN_PCIE_ABORT_UPON_ERROR(m_IoVaManager.Free(busAddr));
        nn::dd::UnmapDeviceAddressSpace(&m_DeviceAddressSpace,
                                        static_cast<nn::dd::ProcessHandle>(summary.context),
                                        summary.procVa, summary.requestedSize, summary.ioVa);
    }
    NN_PCIE_LOG_VERBOSE("RootComplex::DestroyDmaMapByBusAddress(busAddr=%p) = %d:%d\n",
                        reinterpret_cast<void*>(busAddr), result.GetModule(), result.GetDescription());
    return result;
}

Result RootComplex::DestroyDmaMapByProcAddr(int32_t procVaIndex, uint64_t procVa)
{
    Result result;
    IoVaAllocationSummary summary;
    if((result=m_IoVaManager.LookupByProcVa(procVaIndex, procVa, &summary)).IsSuccess())
    {
        NN_PCIE_ABORT_UPON_ERROR(m_IoVaManager.Free(summary.ioVa));
        nn::dd::UnmapDeviceAddressSpace(&m_DeviceAddressSpace,
                                        static_cast<nn::dd::ProcessHandle>(summary.context),
                                        summary.procVa, summary.requestedSize, summary.ioVa);
    }
    NN_PCIE_LOG_VERBOSE("RootComplex::DestroyDmaMapByBusAddress(procVaIndex=%d, procVa=%p) = %d:%d\n",
                        procVaIndex, reinterpret_cast<void*>(procVa),
                        result.GetModule(), result.GetDescription());
    return result;
}

Result RootComplex::DestroyDmaMapsByTag(int32_t tag)
{
    return m_IoVaManager.FreeByTag(tag, &m_DeviceAddressSpace, DestroyByTagCallout);
}

Result RootComplex::GetDmaBusAddress(BusAddress* pReturnedBusAddress, int32_t procVaIndex, uint64_t procVa)
{
    Result result;
    result = m_IoVaManager.GetIoVa(procVaIndex, procVa, pReturnedBusAddress);
    NN_PCIE_LOG_VERBOSE("RootComplex::GetDmaBusAddress(procVaIndex=%d, procVa=%p, busAddr=%p) = %d:%d\n",
                        procVaIndex, reinterpret_cast<void*>(procVa), reinterpret_cast<void*>(*pReturnedBusAddress),
                        result.GetModule(), result.GetDescription());
    return result;
}

void RootComplex::DestroyByTagCallout(void* calloutContext, IoVaAllocationSummary& summary)
{
    NN_PCIE_LOG_VERBOSE("RootComplex::DestroyByTagCallout() tag=%d, procIndex=%d, procVa=0x%llx, requestedSize=%d\n",
                        summary.tag, summary.procVaIndex, summary.procVa, summary.requestedSize);
    nn::dd::UnmapDeviceAddressSpace(reinterpret_cast<nn::dd::DeviceAddressSpaceType*>(calloutContext),
                                    static_cast<nn::dd::ProcessHandle>(summary.context),
                                    summary.procVa, summary.requestedSize, summary.ioVa);
}

void* RootComplex::operator new(size_t size) NN_NOEXCEPT
{
    return nn::pcie::detail::MemoryAllocAligned(size, MinDataAlignmentSize, "RootComplex");
}

void RootComplex::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::pcie::detail::MemoryFree(p, "EndpointFunction");
}


} // end of namespace detail
} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn
