﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief Client class
 *
 * @details
 * - Every independent user of PCI driver is considered to be a
 *   "client".
 * - There can be zero to many active clients in a system.
 * - A client is allowed to acquire (own) zero to many PCI
 *   functions.
 */

namespace nn { namespace pcie { namespace driver { namespace detail {

class Driver;

// Main client class
class Client
{
public:
    struct IrqParameters
    {
        IrqCallback      callback;
        uintptr_t        context;
        IrqType          type;
        int32_t          numberOfVectors;
    };
    static const IrqParameters InvalidIrqParameters;

    struct EndpointFunctionProfile
    {
        FunctionState    state;
        IrqParameters    irqParameters;
        bool             isResetUponResumeEnable;
        uintptr_t        clientContext;
    };

    static const EndpointFunctionProfile InvalidEndpointFunctionProfile;

    enum SecurityPolicy
    {
        SecurityPolicy_Open,
        SecurityPolicy_OpenIfNotAcquired,
        SecurityPolicy_MustBeAcquired,
    };

    NN_IMPLICIT Client(Driver *pDriver, const ClassDriverConfig *pConfig,
                       RegistrationCallback regIndCallback,
                       PmStateCallback pmStateCallback);
    ~Client();

    Result Initialize(ClassDriverHandle *pReturnedHandl);
    void Finalize();

    ClassDriverHandle GetHandle();
    Result AcquireFunction(FunctionHandle funcHandle, uintptr_t clientContext);
    Result ReleaseFunction(FunctionHandle funcHandle);
    Result GetFunctionState(FunctionState* pOutput, FunctionHandle functionHandle);
    static Result GetEndpointFunction(detail::EndpointFunction** ppEpFunc, detail::Client** ppClient,
                                      detail::Driver* pDriver, ClassDriverHandle classDriverHandle,
                                      FunctionHandle functionHandle, SecurityPolicy policy);
    int QueryFunctions(FunctionState* pOutput, size_t queryBufferSize);
    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;
    static int32_t GetNormalizedHandleIndex(Driver *pDriver, ClassDriverHandle handle);
    static Client* GetClient(Driver *pDriver, ClassDriverHandle handle);
    static void UpdateClientFunctionState(Driver *pDriver,
                                          EndpointFunction *pEpFunc,
                                          ClassDriverHandle handle);
    static Result LookupReservedEndpointFunction(ClassDriverHandle* pOutClassDriverHandle,
                                                 FunctionHandle* pOutFunctionHandle,
                                                 IrqParameters* pOutIrqParameters,
                                                 Driver *pDriver, BusNumber busNumber, DeviceNumber deviceNumber,
                                                 FunctionNumber functionNumber, FullClassCode fullClassCode,
                                                 uint16_t vendorId, uint16_t deviceId);
    Result LookupReservedEndpointFunction(ClassDriverHandle* pOutClassDriverHandle,
                                          FunctionHandle* pOutFunctionHandle,
                                          IrqParameters* pOutIrqParameters,
                                          BusNumber busNumber, DeviceNumber deviceNumber,
                                          FunctionNumber functionNumber, FullClassCode fullClassCode,
                                          uint16_t vendorId, uint16_t deviceId);
    Result SetResetUponResumeEnable(FunctionHandle functionHandle, bool isEnabled);
    Result AcquireIrq(FunctionHandle functionHandle, IrqCallback callback,
                      uintptr_t context, IrqType irqType, int32_t numberOfVectors);
    Result ReleaseIrq(FunctionHandle functionHandle);

    nn::util::IntrusiveListNode m_ListNode;

private:
    void DoRegistrationCallback();
    void DoPmStateCallback(EndpointFunction *pEpFunc);
    void EvaluateRegistration();
    EndpointFunctionProfile* GetEndpointFunctionProfile(FunctionHandle funcHandle);

     // Constructor initialized
    Driver*                    m_pDriver;
    ClassDriverConfig          m_Config;
    RegistrationCallback       m_RegistrationCallback;
    PmStateCallback            m_PmStateCallback;
    ClassDriverHandle          m_Handle;

     // Functions owned by this client
    EndpointFunctionProfile    m_Functions[MaxDeviceFunctionsPerClient];

};

} // namespace detail
} // namespace driver
} // namespace pcie
} // namespace nn

