﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/util/util_Optional.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/lbl/lbl_Switch.h>
#include <nn/vi/vi_Display.private.h>
#include <nn/vi/vi_DisplayMode.private.h>
#include <nn/vi/vi_DisplayConfig.h>
#include <nn/vi/vi_DisplayEvents.h>
#include <nn/vi/vi_ResultPrivate.h>
#include <nn/cec/cec_Api.h>
#include <nn/omm/omm_Result.h>
#include <nn/omm/detail/omm_Log.h>
#include <nn/omm/srv/omm_OperationModeManager.h>
#include <nn/omm/srv/omm_DisplayManager.h>
#include <nn/am/service/am_DisplayLayerControl.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_Tv.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/audioctrl/audioctrl_AudioControllerShim.h>

#ifdef NN_OMM_AS_VI_MANAGER
#define NN_OMM_CALL_VI_FUNCTION(Function,Parameters) (::nn::am::service:: Function Parameters)
#else
#define NN_OMM_CALL_VI_FUNCTION(Function,Parameters) (::nn::vi:: Function Parameters)
#endif

namespace nn { namespace omm { namespace srv {
    namespace {

        bool    IsResultEqual(Result result1, Result result2)
        {
            return(result1.GetInnerValueForDebug() == result2.GetInnerValueForDebug());
        }

        enum TvPowerState
        {
            TvPowerState_On,
            TvPowerState_GoingOn,
            TvPowerState_Off,
            TvPowerState_GoingOff,
            TvPowerState_NoAnswer,
            TvPowerState_Error
        };

        TvPowerState    ReportTvPowerState() NN_NOEXCEPT
        {
            cec::PowerState tvPowerState;
            auto result = cec::GetTvPowerState(&tvPowerState);
            if(result.IsSuccess())
            {
                switch(tvPowerState)
                {
                    case cec::PowerState_On:
                        return TvPowerState_On;
                    case cec::PowerState_GoingOn:
                        return TvPowerState_GoingOn;
                    case cec::PowerState_Standby:
                        return TvPowerState_Off;
                    case cec::PowerState_GoingStandby:
                        return TvPowerState_GoingOff;
                    default:
                        break;
                }
            }
            else if((IsResultEqual(result, cec::ResultCommandExecutionFailed())) ||
                    (IsResultEqual(result, cec::ResultTimeout())))
            {
                return TvPowerState_NoAnswer;
            }
            else if(IsResultEqual(result, cec::ResultFeatureAbort()))
            {
                return TvPowerState_On;
            }
            return TvPowerState_Error;
        }

        const int   NumberOfOtpRetries = 5;
        const int   CecOtpRetryPeriodInMilliseconds = 2500;
        const int   CecOtpImageViewOnMaximumRetries = 5;
        const int   CecOtpActiveSourceWaitInMilliseconds = 200;
        const int   CecOtpActiveSourceRetries = 2;
        const int   CecOtpHpdHighTimeoutInMilliseconds = 15000;

        void SetDisplayFade(DisplayManager::DisplayType display, float alpha) NN_NOEXCEPT
        {
            auto result = NN_OMM_CALL_VI_FUNCTION(SetDisplayAlpha, (display, alpha));
            if (result.IsFailure())
            {
                NN_DETAIL_OMM_TRACE("[DisplayManager] Failed to set display alpha 0x%08x\n", result.GetInnerValueForDebug());
            }
        }

        void FadeDisplay(DisplayManager::DisplayType display, TimeSpan timeSpan, bool isIn) NN_NOEXCEPT
        {
            auto begin = os::GetSystemTick().ToTimeSpan();

            while (NN_STATIC_CONDITION(true))
            {
                auto delta = os::GetSystemTick().ToTimeSpan() - begin;
                if (delta >= timeSpan)
                {
                    break;
                }

                static const TimeSpan OneFrameTimeSpan = TimeSpan::FromMicroSeconds(16666);

                auto alpha = isIn ? static_cast<float>(delta.GetMicroSeconds()) / static_cast<float>(timeSpan.GetMicroSeconds()) :
                    static_cast<float>((timeSpan - delta).GetMicroSeconds()) / static_cast<float>(timeSpan.GetMicroSeconds());
                SetDisplayFade(display, alpha);
                os::SleepThread(OneFrameTimeSpan);
            }

            SetDisplayFade(display, isIn ? 1.f : 0.f);
        }
    }

    void DisplayManager::Initialize(OperationModeManager* pOperationModeManager, ConsoleStyle consoleStyle) NN_NOEXCEPT
    {
        m_ConsoleStyle = consoleStyle;
        m_CecController.Initialize(consoleStyle);
        m_HdcpManager.Initialize(pOperationModeManager);
        if (IsConsoleSupported(m_ConsoleStyle))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_OMM_CALL_VI_FUNCTION(OpenDisplay, (&m_Tv, "External")));
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_OMM_CALL_VI_FUNCTION(GetDisplayHotplugEvent, (&m_HotplugEvent, m_Tv)));
            m_StopHotplugHandlerEvent.Signal();
        }
        if (IsHandheldSupported(m_ConsoleStyle))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_OMM_CALL_VI_FUNCTION(OpenDisplay, (&m_Lcd, "Internal")));
        }
    }

    void DisplayManager::SetLcdEnabled(bool isOn) NN_NOEXCEPT
    {
        if (IsHandheldSupported(m_ConsoleStyle))
        {
            NN_DETAIL_OMM_TRACE("[DisplayManager] Set LCD enabled %d\n", isOn);

            // workaround for SIGLO-33313
            m_IsLcdEnabled = isOn;
        }
    }

    void DisplayManager::SetTvEnabled(bool isOn) NN_NOEXCEPT
    {
        if (IsConsoleSupported(m_ConsoleStyle))
        {
            NN_DETAIL_OMM_TRACE("[DisplayManager] Set TV enabled %d\n", isOn);

            m_IsTvEnabled = isOn;
        }
    }

    void DisplayManager::EnableHandheldDisplay() NN_NOEXCEPT
    {
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            this->SetTvEnabled(false);
            this->SetLcdEnabled(true);
        }
        this->UpdatePowerState();
    }

    void DisplayManager::EnableConsoleDisplay() NN_NOEXCEPT
    {
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            this->SetLcdEnabled(false);
            this->SetTvEnabled(true);
        }
        this->UpdatePowerState();
    }

    void DisplayManager::DisableAllDisplay() NN_NOEXCEPT
    {
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            this->SetTvEnabled(false);
            this->SetLcdEnabled(false);
        }
        this->UpdatePowerState();
    }

    void DisplayManager::SwitchBacklightOn() NN_NOEXCEPT
    {
        if (IsHandheldSupported(m_ConsoleStyle))
        {
            lbl::SwitchBacklightOn(0);
        }
    }

    void DisplayManager::SwitchBacklightOff() NN_NOEXCEPT
    {
        if (IsHandheldSupported(m_ConsoleStyle))
        {
            lbl::SwitchBacklightOff(0);
        }
    }

    nn::Result DisplayManager::UpdatePowerState() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        NN_DETAIL_OMM_TRACE("[DisplayManager] Update power state TV %d LCD %d\n", m_IsTvEnabled, m_IsLcdEnabled);

        if (IsConsoleSupported(m_ConsoleStyle) && m_IsTvEnabled)
        {
            NN_DETAIL_OMM_TRACE("[DisplayManager] Setting default display to TV.\n");

            auto result = NN_OMM_CALL_VI_FUNCTION(SetDefaultDisplay, (m_Tv));

            NN_RESULT_TRY(result)
                NN_RESULT_CATCH(nn::vi::ResultDisplayInterfaceFailure)
                {
                    NN_DETAIL_OMM_TRACE("[DisplayManager] Failed to enable display interface\n");
                }
                NN_RESULT_CATCH(nn::vi::ResultPowerStateAlreadyRequested)
                {
                    NN_DETAIL_OMM_TRACE("[DisplayManager] Display interface previously enabled\n");
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                }
            NN_RESULT_END_TRY

            vi::HotplugState hotplugState;
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_OMM_CALL_VI_FUNCTION(GetDisplayHotplugState, (&hotplugState, m_Tv)));

            if(hotplugState == vi::HotplugState_Connected)
            {
                // 外部ディスプレイの電源が入っていないとフェードを操作できない
                // ここで設定すると、hotplug とスリープ遷移が重なった時に微妙な見た目になるが
                // めったに起きないし、起きても一瞬なので許容する
                FadeTv(true, 0);
                UpdateDefaultDisplayResolution(m_Tv);
                m_HdcpManager.NotifyHdmiHotplugConnected();
            }
            else
            {
                m_HdcpManager.NotifyHdmiHotplugDisconnected();
            }
        }
        else
        {
            if (IsHandheldSupported(m_ConsoleStyle) && m_IsLcdEnabled)
            {
                NN_DETAIL_OMM_TRACE("[DisplayManager] Setting default display to LCD.\n");
                NN_ABORT_UNLESS_RESULT_SUCCESS(NN_OMM_CALL_VI_FUNCTION(SetDefaultDisplay, (m_Lcd)));
                UpdateDefaultDisplayResolution(m_Lcd);
            }
        }

        if (IsHandheldSupported(m_ConsoleStyle) && !m_IsLcdEnabled)
        {
            NN_DETAIL_OMM_TRACE("[DisplayManager] Disabling LCD.\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_OMM_CALL_VI_FUNCTION(SetDisplayPowerState, (m_Lcd, vi::PowerState_Off)));
        }

        if (IsConsoleSupported(m_ConsoleStyle) && !m_IsTvEnabled)
        {
            NN_DETAIL_OMM_TRACE("[DisplayManager] Disabling TV.\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_OMM_CALL_VI_FUNCTION(SetDisplayPowerState, (m_Tv, vi::PowerState_Off)));
        }

        NN_RESULT_SUCCESS;
    }

    void DisplayManager::StartDrawingLayer() NN_NOEXCEPT
    {
        am::service::BeginOperationModeDisplay();
    }

    void DisplayManager::StopDrawingLayer() NN_NOEXCEPT
    {
        am::service::EndOperationModeDisplay();
    }

    void DisplayManager::SetDrawingLayerVisible(bool isOn) NN_NOEXCEPT
    {
        if(isOn)
        {
            am::service::ShowOperationModeTexture();
        }
        else
        {
            am::service::HideOperationModeTexture();
        }
    }

    void DisplayManager::SetDrawingLayerTexture(int x, int y, int width, int height, const void* buffer, size_t size) NN_NOEXCEPT
    {
        am::service::SetOperationModeTexture(x, y, width, height, buffer, size);
    }

    void DisplayManager::SetStartupLogoLayerVisible(bool isOn) NN_NOEXCEPT
    {
        if(isOn)
        {
            am::service::ShowOperationModeStartupLogoTexture();
        }
        else
        {
            am::service::HideOperationModeTexture();
        }
    }

    void DisplayManager::FadeLcdAndBacklight(bool isIn, TimeSpan span) NN_NOEXCEPT
    {
        if (IsHandheldSupported(m_ConsoleStyle))
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            NN_DETAIL_OMM_TRACE("[DisplayManager] Fade LCD and backlight %d %lld ms\n", isIn, span.GetMilliSeconds());

            if (isIn)
            {
                lbl::SwitchBacklightOn(span);
            }
            else
            {
                lbl::SwitchBacklightOff(span);
            }
            FadeDisplay(m_Lcd, span, isIn);
        }
    }

    void DisplayManager::FadeLcd(bool isIn, TimeSpan span) NN_NOEXCEPT
    {
        if (IsHandheldSupported(m_ConsoleStyle))
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            NN_DETAIL_OMM_TRACE("[DisplayManager] Fade LCD %d %lld ms\n", isIn, span.GetMilliSeconds());

            FadeDisplay(m_Lcd, span, isIn);
        }
    }

    void DisplayManager::FadeBacklight(bool isIn, TimeSpan span) NN_NOEXCEPT
    {
        if (IsHandheldSupported(m_ConsoleStyle))
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            NN_DETAIL_OMM_TRACE("[DisplayManager] Fade backlight %d %lld ms\n", isIn, span.GetMilliSeconds());

            if (isIn)
            {
                lbl::SwitchBacklightOn(span);
            }
            else
            {
                lbl::SwitchBacklightOff(span);
            }
            os::SleepThread(span);
        }
    }

    void DisplayManager::FadeTv(bool isIn, TimeSpan span) NN_NOEXCEPT
    {
        if (IsConsoleSupported(m_ConsoleStyle))
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            NN_DETAIL_OMM_TRACE("[DisplayManager] Fade TV %d %lld ms\n", isIn, span.GetMilliSeconds());

            FadeDisplay(m_Tv, span, isIn);
        }
    }

    void DisplayManager::StartHotplugHandler() NN_NOEXCEPT
    {
        if (IsConsoleSupported(m_ConsoleStyle))
        {
            NN_ABORT_UNLESS(m_StopHotplugHandlerEvent.TryWait());
            m_StopHotplugHandlerEvent.Clear();

            static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[16 * 1024];
            NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_HotplugHandleThread, [](void* p) {
                auto self = reinterpret_cast<DisplayManager*>(p);
                self->HotplugHandler();
            }, this, s_Stack, sizeof(s_Stack), NN_SYSTEM_THREAD_PRIORITY(omm, HotplugHandler)));
            os::SetThreadNamePointer(&m_HotplugHandleThread, NN_SYSTEM_THREAD_NAME(omm, HotplugHandler));
            os::StartThread(&m_HotplugHandleThread);
        }
    }

    void DisplayManager::StopHotplugHandler() NN_NOEXCEPT
    {
        if (IsConsoleSupported(m_ConsoleStyle))
        {
            NN_ABORT_UNLESS(!m_StopHotplugHandlerEvent.TryWait());
            m_StopHotplugHandlerEvent.Signal();

            os::WaitThread(&m_HotplugHandleThread);
            os::DestroyThread(&m_HotplugHandleThread);
        }
    }

    void DisplayManager::HotplugHandler() NN_NOEXCEPT
    {
        while (NN_STATIC_CONDITION(true))
        {
            auto index = os::WaitAny(&m_HotplugEvent, m_StopHotplugHandlerEvent.GetBase());
            switch (index)
            {
            case 0:
                {
                    os::ClearSystemEvent(&m_HotplugEvent);

                    NN_DETAIL_OMM_TRACE("[DisplayManager] Hotplug event signaled\n");
                    UpdatePowerState();
                }
                break;
            case 1:
                {
                    return;
                }
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    void DisplayManager::GetDefaultDisplayResolution(int* pWidth, int* pHeight) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);
        *pWidth = m_DefaultDisplayResolutionWidth;
        *pHeight = m_DefaultDisplayResolutionHeight;
    }

    os::SystemEvent* DisplayManager::GetDefaultDisplayResolutionChangeEvent() NN_NOEXCEPT
    {
        return &m_DefaultDisplayResolutionChangeEvent;
    }

    void DisplayManager::UpdateDefaultDisplayResolution(DisplayType display) NN_NOEXCEPT
    {
        bool isChanged = false;
        int width, height;
        std::lock_guard<os::Mutex> guard(m_Mutex);
        auto result = NN_OMM_CALL_VI_FUNCTION(GetDisplayResolution, (&width, &height, display));
        if (result.IsFailure())
        {
            NN_DETAIL_OMM_TRACE("[DisplayManager] GetDisplayResolution failed(0x%08x)\n", result.GetInnerValueForDebug());
            return;
        }
        if (width != m_DefaultDisplayResolutionWidth)
        {
            isChanged = true;
            m_DefaultDisplayResolutionWidth = width;
        }
        if (height != m_DefaultDisplayResolutionHeight)
        {
            isChanged = true;
            m_DefaultDisplayResolutionHeight = height;
        }
        if (isChanged)
        {
            m_DefaultDisplayResolutionChangeEvent.Signal();
        }
    }

    void DisplayManager::UpdateDefaultDisplayResolution(bool isLcd) NN_NOEXCEPT
    {
        if (isLcd)
        {
            UpdateDefaultDisplayResolution( m_Lcd );
        }
        else
        {
            UpdateDefaultDisplayResolution( m_Tv );
        }
    }

    // DisplayManager::CecController

    Result DisplayManager::CecController::PerformOneTouchPlayImpl(bool* pOut) NN_NOEXCEPT
    {
        if(m_IsConsoleOnly)
        {
            return(PerformOneTouchPlayConsoleOnlyImpl(pOut));
        }
        else
        {
            return(PerformOneTouchPlayDefaultImpl(pOut));
        }
    }

    bool DisplayManager::CecController::ShouldBeActive() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
        return m_IsEnabled && m_IsResumed && m_IsSettingsEnabled;
    }

    Result DisplayManager::CecController::PerformOneTouchPlayConsoleOnlyImpl(bool* pOut) NN_NOEXCEPT
    {
        NN_DETAIL_OMM_TRACE("[DisplayManager][cec][ConsoleOnly] Start\n");
        *pOut = false;
        NN_RESULT_DO(cec::SetActiveSourceState());
        {
            TvPowerState    tvPowerState = TvPowerState_Error;
            Result          result = cec::ResultOperationFailed();
            int             retries = 0;

            while (retries < CecOtpImageViewOnMaximumRetries)
            {
                std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
                if(!(result.IsSuccess()) ||
                   (tvPowerState == TvPowerState_Off) ||
                   (tvPowerState == TvPowerState_GoingOff))
                {
                    result = cec::PerformImageViewOn();
                    if(!((result.IsSuccess()) || (IsResultEqual(result, cec::ResultCommandExecutionFailed()))))
                    {
                        break;
                    }
                }
                m_Condition.TimedWait(m_Mutex, TimeSpan::FromMilliSeconds(CecOtpRetryPeriodInMilliseconds));
                if (!ShouldBeActive())
                {
                    NN_RESULT_SUCCESS;
                }
                tvPowerState = ReportTvPowerState();
                if (tvPowerState == TvPowerState_Error)
                {
                    return cec::ResultOperationFailed();
                }
                else if((tvPowerState == TvPowerState_On) || (tvPowerState == TvPowerState_GoingOn))
                {
                    result = ResultSuccess();
                    break;
                }
                retries++;
            }
            if(!((result.IsSuccess()) && ((tvPowerState == TvPowerState_On) || (tvPowerState == TvPowerState_GoingOn))))
            {
                return((result.IsSuccess()) ? cec::ResultOperationFailed() : result);
            }
        }
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            os::WaitTimerEvent(&m_CecHpdEvent);
            if ((m_NeedToStop) || (!ShouldBeActive()))
            {
                NN_RESULT_SUCCESS;
            }
        }
        {
            bool    isActiveSource;
            Result  result;
            int     retries = 0;

            while(retries < CecOtpActiveSourceRetries)
            {
                std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
                result = cec::PerformActiveSource();
                if(result.IsSuccess())
                {
                    m_Condition.TimedWait(m_Mutex, TimeSpan::FromMilliSeconds(CecOtpActiveSourceWaitInMilliseconds));
                    if (!ShouldBeActive())
                    {
                        NN_RESULT_SUCCESS;
                    }
                    NN_RESULT_DO(cec::IsActiveSource(&isActiveSource));
                    if (isActiveSource)
                    {
                        if( os::TryWaitTimerEvent(&m_CecHpdEvent))
                        {
                            *pOut = true;
                            break;
                        }
                        os::WaitTimerEvent(&m_CecHpdEvent);
                        if ((m_NeedToStop) || (!ShouldBeActive()))
                        {
                            NN_RESULT_SUCCESS;
                        }
                    }
                }
                else if(!(IsResultEqual(result, cec::ResultCommandExecutionFailed())))
                {
                    return result;
                }
                else
                {
                    m_Condition.TimedWait(m_Mutex, TimeSpan::FromMilliSeconds(CecOtpRetryPeriodInMilliseconds));
                    if (!ShouldBeActive())
                    {
                        NN_RESULT_SUCCESS;
                    }
                }
                retries++;
            }
        }
        NN_RESULT_SUCCESS;
    }

    Result DisplayManager::CecController::PerformOneTouchPlayDefaultImpl(bool* pOut) NN_NOEXCEPT
    {
        NN_DETAIL_OMM_TRACE("[DisplayManager][cec][Default] Start\n");
        // OneTouchPlay を試行
        NN_RESULT_DO(cec::PerformOneTouchPlay());
        // 一定時間待機
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            m_Condition.TimedWait(m_Mutex, TimeSpan::FromMilliSeconds(CecOtpRetryPeriodInMilliseconds));
            if (!ShouldBeActive())
            {
                *pOut = false;
                NN_RESULT_SUCCESS;
            }
        }
        // TV の電源 ON を確認
        if (!(ReportTvPowerState() == TvPowerState_On))
        {
            NN_DETAIL_OMM_TRACE("[DisplayManager][cec] OTP: TV power is not yet On.\n");
            *pOut = false;
            NN_RESULT_SUCCESS;
        }
        // TV のソースが設定されていることを確認
        bool isActiveSource;
        NN_RESULT_DO(cec::IsActiveSource(&isActiveSource));
        if (!isActiveSource)
        {
            NN_DETAIL_OMM_TRACE("[DisplayManager][cec] OTP: Not active device yet.\n");
            *pOut = false;
            NN_RESULT_SUCCESS;
        }

        *pOut = true;
        NN_RESULT_SUCCESS;
    }

    bool DisplayManager::CecController::TryToRestart() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
        // 現在のシーケンス番号が、エラーの起きたシーケンス番号より新しい場合、再スタートの価値がある。
        return m_CurrentConnectionSequenceNumber > m_FaliedConnectionSequenceNumber;
    }

    void DisplayManager::CecController::MarkFailed(uint64_t sequenceNumber) NN_NOEXCEPT
    {
        this->m_FaliedConnectionSequenceNumber = sequenceNumber;
    }

    void DisplayManager::CecController::MarkNewConnection() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
        ++this->m_CurrentConnectionSequenceNumber;
    }

    Result DisplayManager::CecController::RunOneImpl(uint64_t sequenceNumber) NN_NOEXCEPT
    {
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            this->m_NeedsCancel = true;
        }
        NN_UTIL_SCOPE_EXIT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            this->m_NeedsCancel = false;
        };

        // workaround for SIGLO-53470(IAAA-4313,NXGLOPS-561)
        // STDP FW がロード、起動中に DPCD レジスタにアクセスして STDP(クレードル) が予期しない状態になっている可能性があるため、
        // cec::RestartManager() に実行を遅らせて DPCD レジスタアクセスを遅らせる
        // SoC から STDP を起こす命令を発行してからリセット解除処理まで 150ms
        // リセット解除から FW 起動までは 250ms、計400ms 待つ
        os::SleepThread(TimeSpan::FromMilliSeconds(400));

        // cec 有効化
        NN_RESULT_DO(cec::RestartManager());
        NN_UTIL_SCOPE_EXIT
        {
            cec::SuspendManager();
        };

        // TV への接続を確認
        {
            bool isTvResponsive;
            NN_RESULT_DO(cec::IsTvResponsive(&isTvResponsive));
            if (!isTvResponsive)
            {
                MarkFailed(sequenceNumber);
                NN_DETAIL_OMM_TRACE("[DisplayManager][cec] OTP: TV cannot be reached by CEC.\n");
                NN_RESULT_SUCCESS;
            }
        }

        // ワンタッチプレイを起動
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        for (;;)
        {
            NN_SDK_ASSERT(lk.owns_lock());
            if (!ShouldBeActive())
            {
                // すでに ShouldBeActive() であったら抜ける
                NN_DETAIL_OMM_TRACE("[DisplayManager][cec] One Touch Play canceled.\n");
                NN_RESULT_SUCCESS;
            }
            else if (m_OneTouchPlayRetryCount > 0)
            {
                // ワンタッチプレイを一回試行する
                NN_DETAIL_OMM_TRACE("[DisplayManager][cec] Try One Touch Play (rest count of trial = %d).\n", m_OneTouchPlayRetryCount);
                --this->m_OneTouchPlayRetryCount;
                lk.unlock();
                bool done;
                NN_RESULT_DO(PerformOneTouchPlayImpl(&done));
                lk.lock();
                if (done)
                {
                    NN_DETAIL_OMM_TRACE("[DisplayManager][cec] OTP: Successed.\n");
                    this->m_OneTouchPlayRetryCount = 0;
                }
                else
                {
                    if (m_OneTouchPlayRetryCount == 0)
                    {
                        NN_DETAIL_OMM_TRACE("[DisplayManager][cec] OTP: Give up.\n");
                    }
                }
            }
            else
            {
                m_Condition.Wait(m_Mutex);
            }
        }
    }

    void DisplayManager::CecController::RunImpl() NN_NOEXCEPT
    {
        for (;;)
        {
            // ShouldBeActive() && TryToRestart() を満たすまで待機
            uint64_t sequenceNumber;
            {
                // 待機区間中のみ m_SuspendedEvent をシグナルする
                m_SuspendedEvent.Signal();
                NN_UTIL_SCOPE_EXIT
                {
                    m_SuspendedEvent.Clear();
                };
                std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
                while (!(ShouldBeActive() && TryToRestart()))
                {
                    m_Condition.Wait(m_Mutex);
                }
                sequenceNumber = m_CurrentConnectionSequenceNumber;
            }
            NN_DETAIL_OMM_TRACE("[DisplayManager][cec] Start CEC Manager()\n");
            auto result = RunOneImpl(sequenceNumber);
            if (result.IsFailure())
            {
                MarkFailed(sequenceNumber);
                NN_DETAIL_OMM_TRACE("[DisplayManager][cec] Failed: CEC manager 0x%08X\n",
                                    result.GetInnerValueForDebug());
            }
            NN_DETAIL_OMM_TRACE("[DisplayManager][cec] Suspend CEC Manager()\n");
        }
    }

    void DisplayManager::CecController::ManageCecTimerEvent() NN_NOEXCEPT
    {
        cec::GetHpdState(&m_HpdState);
        NN_DETAIL_OMM_TRACE("[DisplayManager][cec][hpd] %d\n", m_HpdState);
        if(m_HpdState)
        {
            os::SignalTimerEvent(&m_CecHpdEvent);
        }
        else
        {
            os::ClearTimerEvent(&m_CecHpdEvent);
            os::StartOneShotTimerEvent(&m_CecHpdEvent, TimeSpan::FromMilliSeconds(CecOtpHpdHighTimeoutInMilliseconds));
        }
    }

    void DisplayManager::CecController::CecEventMonitor() NN_NOEXCEPT
    {
        while (NN_STATIC_CONDITION(true))
        {
            cec::BusEventType   busEvent;

            os::WaitSystemEvent(&m_CecSystemEvent);
            cec::GetBusEventType(&busEvent);
            switch(busEvent)
            {
                case cec::BusEventType_Started:
                    m_NeedToStop = false;
                    ManageCecTimerEvent();
                    break;
                case cec::BusEventType_ActiveSourceChangedToActive:
                    m_NeedToStop = false;
                    break;
                case cec::BusEventType_Suspending:
                case cec::BusEventType_ActiveSourceChangedToInactive:
                case cec::BusEventType_GoStandby:
                case cec::BusEventType_FeatureAbortOneTouchPlay:
                    m_NeedToStop = true;
                    os::SignalTimerEvent(&m_CecHpdEvent);
                    break;
                case cec::BusEventType_ConnectionChange:
                    ManageCecTimerEvent();
                    break;
                default:
                    break;
            }
        }
    }

    void DisplayManager::CecController::Initialize(ConsoleStyle consoleStyle) NN_NOEXCEPT
    {
        this->m_IsSettingsEnabled = IsCurrentlyCecEnabled();
        this->m_IsEnabled = false;
        this->m_IsResumed = true;
        static NN_ALIGNAS(4096) char s_Stack[8 * 1024];
        static NN_ALIGNAS(4096) char s_CecMonitorStack[8 * 1024];
        NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_CecThread, [](void* this_)
        {
            reinterpret_cast<CecController*>(this_)->RunImpl();
        }, this, s_Stack, sizeof(s_Stack), NN_SYSTEM_THREAD_PRIORITY(omm, PerformOneTouchPlay)));
        os::SetThreadNamePointer(&m_CecThread, NN_SYSTEM_THREAD_NAME(omm, PerformOneTouchPlay));
        os::StartThread(&m_CecThread);
        os::InitializeTimerEvent(&m_CecHpdEvent, os::EventClearMode_ManualClear);
        m_IsConsoleOnly = IsConsoleSupported(consoleStyle) && !IsHandheldSupported(consoleStyle);
        cec::Initialize(&m_CecSystemEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_CecMonitor, [](void* this_)
        {
            reinterpret_cast<CecController*>(this_)->CecEventMonitor();
        }, this, s_CecMonitorStack, sizeof(s_CecMonitorStack), NN_SYSTEM_THREAD_PRIORITY(omm, CecMonitorThread)));
        os::SetThreadNamePointer(&m_CecMonitor, NN_SYSTEM_THREAD_NAME(omm, CecMonitorThread));
        os::StartThread(&m_CecMonitor);
    }

    bool DisplayManager::CecController::ReadCecSettings() NN_NOEXCEPT
    {
        nn::settings::system::TvSettings settings;
        nn::settings::system::GetTvSettings(&settings);
        return settings.flags.Test<nn::settings::system::TvFlag::AllowsCec>();
    }

    bool DisplayManager::CecController::IsCurrentlyCecEnabled() NN_NOEXCEPT
    {
        if (m_ApplicationPowerStateMatchingMode == TvPowerStateMatchingMode_Disabled)
        {
            // この if 文が成立するのは、アプリがインフォーカス状態かつ、
            // API によりアプリが CEC を無効にしている場合のみ。
            return false;
        }
        return ReadCecSettings();
    }

    void DisplayManager::CecController::NotifyCecSettingsChanged() NN_NOEXCEPT
    {
        auto isSettingsEnabled = IsCurrentlyCecEnabled();
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_IsSettingsEnabled = isSettingsEnabled;
        this->m_OneTouchPlayRetryCount = 0;
        m_Condition.Signal();
    }

    void DisplayManager::CecController::SetApplicationCecSettingsAndNotifyChanged(TvPowerStateMatchingMode mode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( IsValidTvPowerStateMatchingMode(mode) );

        // この関数はアプリによる API 呼出時もしくは、
        // アプリ <=> アプレット間遷移時に都度呼ばれる。
        this->m_ApplicationPowerStateMatchingMode = mode;
        this->NotifyCecSettingsChanged();
    }

    void DisplayManager::CecController::CancelImpl(std::unique_lock<os::Mutex> lk) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(lk.owns_lock());
        m_Condition.Signal();
        if (m_NeedsCancel)
        {
            NN_DETAIL_OMM_TRACE("[DisplayManager][cec] Trigger cancel CEC\n");
            lk.unlock();
            bool dummy;
            os::SignalTimerEvent(&m_CecHpdEvent);
            cec::CancelCurrentApiCall(&dummy);
        }
    }

    void DisplayManager::CecController::Enable() NN_NOEXCEPT
    {
        auto isSettingsEnabled = IsCurrentlyCecEnabled();
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_IsSettingsEnabled = isSettingsEnabled;
        this->m_IsEnabled = true;
        MarkNewConnection();
        this->m_OneTouchPlayRetryCount = 0;
        m_Condition.Signal();
    }

    void DisplayManager::CecController::Disable() NN_NOEXCEPT
    {
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        this->m_IsEnabled = false;
        this->m_OneTouchPlayRetryCount = 0;
        CancelImpl(std::move(lk));
    }

    void DisplayManager::CecController::BeforeSleep() NN_NOEXCEPT
    {
        {
            std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
            this->m_IsResumed = false;
            this->m_OneTouchPlayRetryCount = 0;
            CancelImpl(std::move(lk));
        }
        // cec の停止が完了していることを待機
        m_SuspendedEvent.Wait();
    }

    void DisplayManager::CecController::AfterSleep() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_IsResumed = true;
        MarkNewConnection();
        this->m_OneTouchPlayRetryCount = 0;
        m_Condition.Signal();
    }

    void DisplayManager::CecController::TriggerOneTouchPlay() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_OneTouchPlayRetryCount = NumberOfOtpRetries;
        m_Condition.Signal();
    }

    // DisplayManager::HdcpManager

    void DisplayManager::HdcpManager::Initialize(OperationModeManager* pOperationModeManager) NN_NOEXCEPT
    {
        m_pOperationModeManager = pOperationModeManager;
        m_HdcpState = HdcpState::Unauthenticated;
        m_TriggerStatus = { OperationMode::Handheld, hdcp::HdcpAuthenticationState_Unauthenticated, false, false, false };
        os::InitializeEvent( &m_TriggerStatusChangeEvent, false, os::EventClearMode_ManualClear );
        NN_ABORT_UNLESS_RESULT_SUCCESS(hdcp::GetHdcpStateTransitionEvent( &m_HdcpAuthenticationTransitionEvent ));

        m_IsHdcpFailedEmulationEnabled = false;
        settings::fwdbg::GetSettingsItemValue(&m_IsHdcpFailedEmulationAllowed, sizeof(m_IsHdcpFailedEmulationAllowed), "omm", "hdcp_failed_emulation");

        static NN_ALIGNAS(4096) char s_Stack[8 * 1024];
        NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_HdcpThread, [](void* this_)
        {
            reinterpret_cast<HdcpManager*>(this_)->RunImpl();
        }, this, s_Stack, sizeof(s_Stack), NN_SYSTEM_THREAD_PRIORITY(omm, HdcpHandler)));
        os::SetThreadNamePointer(&m_HdcpThread, NN_SYSTEM_THREAD_NAME(omm, HdcpHandler));
        os::StartThread(&m_HdcpThread);
    }

    os::SystemEvent* DisplayManager::HdcpManager::GetHdcpAuthenticationFailedEvent() NN_NOEXCEPT
    {
        return &m_HdcpAuthenticationFailedEvent;
    }

    HdcpState DisplayManager::HdcpManager::GetHdcpState() NN_NOEXCEPT
    {
        return m_HdcpState;
    }

    os::SystemEvent* DisplayManager::HdcpManager::GetHdcpStateChangeEvent() NN_NOEXCEPT
    {
        return &m_HdcpAuthenticationStateChangeEvent;
    }

    void DisplayManager::HdcpManager::NotifyHdcpApplicationExecutionStarted() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_TriggerStatusMutex);
        m_TriggerStatus.isHdcpApplicationRunning = true;
        os::SignalEvent( &m_TriggerStatusChangeEvent );
    }

    void DisplayManager::HdcpManager::NotifyHdcpApplicationExecutionFinished() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_TriggerStatusMutex);
        m_TriggerStatus.isHdcpApplicationRunning = false;
        os::SignalEvent( &m_TriggerStatusChangeEvent );
    }

    void DisplayManager::HdcpManager::NotifyHdcpApplicationDrawingStarted() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_TriggerStatusMutex);
        m_TriggerStatus.isHdcpApplicationForeground = true;
        os::SignalEvent( &m_TriggerStatusChangeEvent );
    }

    void DisplayManager::HdcpManager::NotifyHdcpApplicationDrawingFinished() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_TriggerStatusMutex);
        m_TriggerStatus.isHdcpApplicationForeground = false;
        os::SignalEvent( &m_TriggerStatusChangeEvent );
    }

    void DisplayManager::HdcpManager::NotifyOperationModeChanged() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_TriggerStatusMutex);
        m_TriggerStatus.operationMode = m_pOperationModeManager->GetOperationMode();
        os::SignalEvent( &m_TriggerStatusChangeEvent );
    }

    void DisplayManager::HdcpManager::NotifyHdmiHotplugConnected() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_TriggerStatusMutex);
        m_TriggerStatus.isHdmiConneted = true;
        os::SignalEvent( &m_TriggerStatusChangeEvent );
    }

    void DisplayManager::HdcpManager::NotifyHdmiHotplugDisconnected() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_TriggerStatusMutex);
        m_TriggerStatus.isHdmiConneted = false;
        os::SignalEvent( &m_TriggerStatusChangeEvent );
    }

    void DisplayManager::HdcpManager::NotifyHdcpAuthenticationStateChanged() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_TriggerStatusMutex);
        hdcp::HdcpAuthenticationState currentHdcpAuthenticationState;
        NN_ABORT_UNLESS_RESULT_SUCCESS(hdcp::GetHdcpAuthenticationState( &currentHdcpAuthenticationState ));

        // HDCP Failed Emulation
        if ( currentHdcpAuthenticationState == hdcp::HdcpAuthenticationState_Authenticated &&
            m_IsHdcpFailedEmulationEnabled == true )
        {
            currentHdcpAuthenticationState = hdcp::HdcpAuthenticationState_Timeout;
        }

        m_TriggerStatus.hdcpAuthenticationState = currentHdcpAuthenticationState;

        // workaround for NSBG-7386
        // 解像度変更したときに Timeout が返ってきてしまうので HdcpState が Processing でなかったら書き換える
        if ( m_HdcpState != HdcpState::ProcessingForeground &&
            m_HdcpState != HdcpState::ProcessingBackground )
        {
            m_TriggerStatus.hdcpAuthenticationState = hdcp::HdcpAuthenticationState_Unauthenticated;
        }

        if ( IsHdcpAuthenticationFailed(m_TriggerStatus.hdcpAuthenticationState) )
        {
            m_HdcpAuthenticationFailedEvent.Signal();
        }

        os::SignalEvent( &m_TriggerStatusChangeEvent );
    }

    bool DisplayManager::HdcpManager::GetHdcpAuthenticationFailedEmulationEnabled() NN_NOEXCEPT
    {
        return m_IsHdcpFailedEmulationEnabled;
    }

    Result DisplayManager::HdcpManager::SetHdcpAuthenticationFailedEmulation(bool enable) NN_NOEXCEPT
    {
        if (enable && !m_IsHdcpFailedEmulationAllowed)
        {
            return omm::ResultNotPermittedOnProduction();
        }
        m_IsHdcpFailedEmulationEnabled = enable;

        NN_RESULT_SUCCESS;
    }

    bool DisplayManager::HdcpManager::IsHdcpAuthenticationFailed(hdcp::HdcpAuthenticationState state) const NN_NOEXCEPT
    {
        return state == hdcp::HdcpAuthenticationState_Timeout;
    }

    HdcpState DisplayManager::HdcpManager::GetNextHdcpState() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_TriggerStatusMutex);
        const TriggerStatus& ts = m_TriggerStatus;

        NN_DETAIL_OMM_TRACE("[DisplayManager][hdcp] TriggerStatus={ %d, %d, %d, %d, %d }\n",
            ts.isHdcpApplicationRunning, ts.isHdcpApplicationForeground,
            ts.operationMode, ts.isHdmiConneted, ts.hdcpAuthenticationState );

        // HDCP 認証不要もしくは未認証
        if ( ! (ts.isHdcpApplicationRunning == true &&
            ts.operationMode == OperationMode::Console &&
            ts.isHdmiConneted == true) )
        {
            return HdcpState::Unauthenticated;
        }
        // HDCP 認証開始または認証中 (Foreground)
        else if ( ts.isHdcpApplicationForeground == true &&
            (ts.hdcpAuthenticationState == hdcp::HdcpAuthenticationState_Unauthenticated ||
            ts.hdcpAuthenticationState == hdcp::HdcpAuthenticationState_Processing ||
            ts.hdcpAuthenticationState == hdcp::HdcpAuthenticationState_NotPlugged) )
        {
            return HdcpState::ProcessingForeground;
        }
        // HDCP 認証中 (Background)
        else if ( ts.isHdcpApplicationForeground == false &&
            ts.hdcpAuthenticationState == hdcp::HdcpAuthenticationState_Processing )
        {
            return HdcpState::ProcessingBackground;
        }
        // HDCP 認証失敗
        else if ( IsHdcpAuthenticationFailed(ts.hdcpAuthenticationState) )
        {
            if ( ts.isHdcpApplicationForeground == true )
            {
                return HdcpState::AuthenticationFailed;
            }
            return HdcpState::Unauthenticated;
        }
        // HDCP 認証成功
        else if ( ts.hdcpAuthenticationState == hdcp::HdcpAuthenticationState_Authenticated )
        {
            return HdcpState::Authenticated;
        }
        // HDCP 認証不要または未認証
        return HdcpState::Unauthenticated;
    }

    void DisplayManager::HdcpManager::UpdateHdcpState() NN_NOEXCEPT
    {
        HdcpState nextHdcpState = GetNextHdcpState();
        NN_DETAIL_OMM_TRACE("[DisplayManager][hdcp] CurrentHdcpState=%d, NextHdcpState=%d\n",
            static_cast<int>(m_HdcpState), static_cast<int>(nextHdcpState) );
        if (nextHdcpState == m_HdcpState)
        {
            return;
        }

        switch (nextHdcpState)
        {
        case HdcpState::Unauthenticated:
            NN_ABORT_UNLESS_RESULT_SUCCESS(hdcp::SetCurrentHdcpMode( hdcp::HdcpMode_Disabled ));
            {
                std::lock_guard<os::Mutex> guard(m_TriggerStatusMutex);
                m_TriggerStatus.hdcpAuthenticationState = hdcp::HdcpAuthenticationState_Unauthenticated;
            }
            audioctrl::SetSystemOutputMasterVolume( audioctrl::SystemOutputMasterVolumeMax );
            m_HdcpAuthenticationStateChangeEvent.Signal();
            break;

        case HdcpState::ProcessingForeground:
            audioctrl::SetSystemOutputMasterVolume( audioctrl::SystemOutputMasterVolumeMin );
            NN_ABORT_UNLESS_RESULT_SUCCESS(hdcp::SetCurrentHdcpMode( hdcp::HdcpMode_Enabled ));
            m_HdcpAuthenticationStateChangeEvent.Signal();
            break;

        case HdcpState::ProcessingBackground:
            audioctrl::SetSystemOutputMasterVolume( audioctrl::SystemOutputMasterVolumeMin );
            break;

        case HdcpState::AuthenticationFailed:
            audioctrl::SetSystemOutputMasterVolume( audioctrl::SystemOutputMasterVolumeMin );
            break;

        case HdcpState::Authenticated:
            audioctrl::SetSystemOutputMasterVolume( audioctrl::SystemOutputMasterVolumeMax );
            m_HdcpAuthenticationStateChangeEvent.Signal();
            break;

        default:
            NN_ABORT("Unexpected State\n");
        }
        m_HdcpState = nextHdcpState;
    }

    void DisplayManager::HdcpManager::RunImpl() NN_NOEXCEPT
    {
        for (;;)
        {
            int signal = os::WaitAny(
                            &m_HdcpAuthenticationTransitionEvent,
                            &m_TriggerStatusChangeEvent
                            );
            if (signal == 0)
            {
                os::ClearSystemEvent( &m_HdcpAuthenticationTransitionEvent );
                NotifyHdcpAuthenticationStateChanged();
            }
            else if (signal == 1)
            {
                os::ClearEvent( &m_TriggerStatusChangeEvent );
                UpdateHdcpState();
            }
        }
    }

}}}
