﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataTransferType.h>
#include <nn/http/http_Result.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/olsc/srv/olsc_TransferTask.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveUpload.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveDownload.h>
#include <nn/olsc/srv/transfer/olsc_ComponentFileUpload.h>
#include <nn/olsc/srv/transfer/olsc_ComponentFileDownload.h>
#include <nn/olsc/srv/transfer/olsc_TransferHelper.h>
#include <nn/olsc/srv/util/olsc_Account.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Execution.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc { namespace srv { namespace transfer {

namespace {

} // namespace

Result TransferHelper::StartWholeUpload(
    SaveDataArchiveInfo* outSaveDataArchiveInfo,
    const account::Uid& uid,
    const ApplicationId appId,
    const SeriesInfo& seriesInfo,
    const NsaIdToken& nsaIdToken,
    const fs::SaveDataInfo& saveDataInfo,
    CURL* curlHandle,
    void* workBuffer, size_t workBufferSize, nn::util::Cancelable& cancelable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(Buffer));
    NN_SDK_REQUIRES_EQUAL(reinterpret_cast<uintptr_t>(workBuffer) % static_cast<uint64_t>(std::alignment_of<Buffer>::value), 0);

    auto buffer = reinterpret_cast<Buffer*>(workBuffer);

    time::PosixTime updatedTime;
    NN_RESULT_DO(fs::GetSaveDataTimeStamp(&updatedTime, saveDataInfo.saveDataId));

    size_t saveDataSize = static_cast<size_t>(saveDataInfo.saveDataSize);

    SaveDataArchiveInfo sda;
    NN_RESULT_TRY(transfer::StartSaveDataArchiveUpload(&sda, uid, appId, seriesInfo, saveDataSize, updatedTime, nsaIdToken, curlHandle, buffer->ioBuffer.data(), buffer->ioBuffer.size(), &cancelable))
        // 「別に UL が実行中」エラー
        NN_RESULT_CATCH_CONVERT(http::ResultHttpStatusConflict, olsc::ResultServerDataUploadingByOtherDevice())
    NN_RESULT_END_TRY;
    *outSaveDataArchiveInfo = sda;
    NN_RESULT_SUCCESS;
}

Result TransferHelper::ExportSaveData(
    SaveDataArchiveId sdaId,
    std::unique_ptr<fs::ISaveDataDivisionExporter>& exporter,
    std::unique_ptr<fs::ISaveDataChunkIterator>& iter,
    NsaIdToken& nsaIdToken,
    CURL* curlHandle,
    const ContextUpdater& contextUpdater,
    const account::Uid& uid,
    void* workBuffer, size_t workBufferSize, nn::util::Cancelable& cancelable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(Buffer));
    NN_SDK_REQUIRES_EQUAL(reinterpret_cast<uintptr_t>(workBuffer) % static_cast<uint64_t>(std::alignment_of<Buffer>::value), 0);

    auto buffer = reinterpret_cast<Buffer*>(workBuffer);

    size_t offsetNumber = 0;
    for (; !iter->IsEnd(); iter->Next())
    {
        // UL タイムアウトの延長
        SaveDataArchiveInfo sda;
        int cfCount = 0;
        NN_RESULT_DO(ExtendSaveDataArchiveUploadTimeout(&sda, &cfCount,
            buffer->componentFileListBuffer.data(),
            buffer->componentFileListBuffer.size(),
            sdaId, nsaIdToken, curlHandle, buffer->ioBuffer.data(), buffer->ioBuffer.size(), &cancelable));

        auto index = iter->GetId();

        auto endCf = buffer->componentFileListBuffer.begin() + cfCount;
        auto targetCf = std::find_if(buffer->componentFileListBuffer.begin(), endCf, [&index](ComponentFileInfo i) {
            return i.clientArgument.chunkId == index;
        });

        ComponentFileInfo cf;
        size_t size = 0;

        if(targetCf != endCf)// index の CF が既存
        {
            NN_RESULT_THROW_UNLESS(targetCf->status != ComponentFileStatus::Fixed, olsc::ResultServerDataChunkUnexpectedStatus());
            if(targetCf->status == ComponentFileStatus::Uploading)
            {
                NN_RESULT_DO(GetComponentFileSignedUrlForUpload(&cf, targetCf->id, nsaIdToken, curlHandle, buffer->ioBuffer.data(), buffer->ioBuffer.size(), &cancelable));
            }
            else if(targetCf->status == ComponentFileStatus::HandOver)
            {
                NN_RESULT_DO(UpdateComponentFile(&cf, targetCf->id, nsaIdToken, curlHandle, buffer->ioBuffer.data(), buffer->ioBuffer.size(), &cancelable));
            }
        }
        else // index の CF がない
        {
            ComponentFileType type = index == nn::fs::SaveDataChunkIdForInitialData ? ComponentFileType::Meta : ComponentFileType::Save;
            ClientArgument clientArgument = {index};
            NN_RESULT_DO(CreateComponentFile(&cf, sdaId, clientArgument, type, nsaIdToken, curlHandle, buffer->ioBuffer.data(), buffer->ioBuffer.size(), &cancelable));
        }

        SaveDataChunkDigest digest;
        NN_RESULT_DO(ExportComponentFile(&size, &digest, cf.url, exporter.get(), index, nsaIdToken, curlHandle, buffer->ioBuffer.data(), buffer->ioBuffer.size(), &cancelable));

        // ExportComponentFile で時間がかかって expire している可能性があるため再取得
        size_t nsaIdTokenSize;
        NN_RESULT_DO(util::GetNsaIdToken(&nsaIdTokenSize, nsaIdToken.data, sizeof(nsaIdToken.data), uid, cancelable));

        NN_RESULT_DO(CompleteComponentFile(cf.id, size, digest, nsaIdToken, curlHandle, buffer->ioBuffer.data(), buffer->ioBuffer.size(), &cancelable));
        contextUpdater(size, ++offsetNumber);
    }
    NN_RESULT_SUCCESS;
}

Result TransferHelper::FinishWholeUpload(
    SaveDataArchiveId sdaId,
    std::unique_ptr<fs::ISaveDataDivisionExporter>& exporter,
    const NsaIdToken& nsaIdToken,
    CURL* curlHandle,
    void* workBuffer, size_t workBufferSize, nn::util::Cancelable& cancelable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(Buffer));
    NN_SDK_REQUIRES_EQUAL(reinterpret_cast<uintptr_t>(workBuffer) % static_cast<uint64_t>(std::alignment_of<Buffer>::value), 0);

    auto buffer = reinterpret_cast<Buffer*>(workBuffer);

    // TODO : fs から正式な値取得 API を提供されたら呼び出すように修正
    nn::fs::detail::KeySeed keySeedFs;
    nn::fs::detail::InitialDataMac macFs;
    NN_RESULT_DO(exporter->FinalizeFullExport(&keySeedFs, &macFs));
    nn::olsc::srv::KeySeed keySeed;
    memcpy(keySeed.data, keySeedFs.data, nn::olsc::srv::KeySeed::Size);
    nn::olsc::srv::InitialDataMac mac;
    memcpy(mac.data, macFs.data, nn::olsc::srv::InitialDataMac::Size);

    NN_RESULT_DO(transfer::FinishSaveDataArchiveUpload(sdaId, keySeed, mac, nsaIdToken, curlHandle, buffer->ioBuffer.data(), buffer->ioBuffer.size(), &cancelable));

    NN_RESULT_SUCCESS;
}

Result TransferHelper::GetSuspendedSaveDataArchiveInfo(
    SaveDataArchiveInfo* outSda,
    int* pOutCfCount,
    SaveDataArchiveId sdaId,
    const NsaIdToken& nsaIdToken,
    CURL* curlHandle,
    void* workBuffer, size_t workBufferSize, nn::util::Cancelable& cancelable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(Buffer));
    NN_SDK_REQUIRES_EQUAL(reinterpret_cast<uintptr_t>(workBuffer) % static_cast<uint64_t>(std::alignment_of<Buffer>::value), 0);

    auto buffer = reinterpret_cast<Buffer*>(workBuffer);
    NN_RESULT_TRY(transfer::ExtendSaveDataArchiveUploadTimeout(outSda, pOutCfCount,
        buffer->componentFileListBuffer.data(),
        buffer->componentFileListBuffer.size(),
        sdaId, nsaIdToken, curlHandle, buffer->ioBuffer.data(), buffer->ioBuffer.size(), &cancelable))
        // 再開しようとしたけどタイムアウトして SDA が削除されていた
        NN_RESULT_CATCH_CONVERT(http::ResultHttpStatusNotFound, olsc::ResultServerDataDeletedWhileSuspended())
        // SDA が uploading ではなかった
        NN_RESULT_CATCH(http::ResultHttpStatusForbidden)
        {
            nn::util::optional<SaveDataArchiveInfo> serverSda;
            NN_RESULT_DO(transfer::RequestSaveDataArchiveInfo(
                &serverSda, nsaIdToken, sdaId, curlHandle, buffer->ioBuffer.data(), buffer->ioBuffer.size(), &cancelable));

            // SDA がない場合はタイミング良く消されたケースなので、 ExtendUploadTimeout で NotFound が返った場合と同様の result にする
            NN_RESULT_THROW_UNLESS(serverSda, olsc::ResultServerDataDeletedWhileSuspended());

            // SDA が fixed ではない場合は「呼び出したデバイスが、指定した ID の SDA の所有元デバイスでない」ため、そのまま返す
            NN_RESULT_THROW_UNLESS(serverSda->status == SaveDataArchiveStatus::Fixed, NN_RESULT_CURRENT_RESULT);

            *outSda = *serverSda;
        }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}


// TODO : SCSI-534 の結論次第でインポート方法を変更する必要あり
// とりあえず「差分データを DL しながらオンザフライに適用する」
Result TransferHelper::ImportSaveDataCore(
    const NsaIdToken& nsaIdToken,
    CURL* curlHandle,
    fs::ISaveDataDivisionImporter& importer,
    TransferTaskBase::ComponentFileInfoListBuffer& cfList,
    size_t listCount,
    const ContextUpdater& contextUpdater,
    void* workBuffer,
    size_t workBufferSize,
    nn::util::Cancelable& cancelable) NN_NOEXCEPT
{

    // TODO: WEB API: start_download に該当する関数を呼び出す。

    // ここからセーブデータを破壊的に拡張・更新、または作成
    int64_t requiredSize;
    NN_RESULT_DO(importer.InitializeImport(&requiredSize));

    // TODO: セーブデータ新規作成・拡張のための空き容量不足のハンドリング
    NN_UNUSED(requiredSize);

    {
        std::unique_ptr<fs::ISaveDataChunkIterator> iter;
        NN_RESULT_DO(importer.OpenSaveDataDiffChunkIterator(&iter));

        // 差分のある sf について（新規なら全 sf について）ループ
        for (int index = 0; !iter->IsEnd(); iter->Next(), ++index)
        {
            auto chunkId = iter->GetId();
            std::unique_ptr<fs::ISaveDataChunkImporter> chunkImporter;
            NN_RESULT_DO(importer.OpenSaveDataChunkImporter(&chunkImporter, chunkId));

            auto endCf = cfList.begin() + listCount;
            auto targetCf = std::find_if(cfList.begin(), endCf, [&chunkId](olsc::srv::ComponentFileInfo i) {
                return i.clientArgument.chunkId == chunkId;
            });
            NN_RESULT_THROW_UNLESS(targetCf != endCf, olsc::ResultServerDataChunkNotExist());

            NN_RESULT_DO(transfer::RequestImportComponentFile(
                chunkImporter.get(),
                targetCf->url,
                targetCf->saveDataChunkSize,
                curlHandle,
                workBuffer,
                workBufferSize, &cancelable));

            contextUpdater(targetCf->saveDataChunkSize, index + 1);
        }
    }

    // TODO: WEB API: finish_download に該当する関数を呼び出す。

    NN_RESULT_DO(importer.FinalizeImport());
    NN_RESULT_SUCCESS;

}

Result TransferHelper::Download(
    const account::Uid& uid,
    const SaveDataArchiveInfo& sda,
    const NsaIdToken& nsaIdToken,
    CURL* curlHandle,
    const ContextInitializer& contextInitializer, const ContextUpdater& contextUpdater,
    void* workBuffer, size_t workBufferSize, nn::util::Cancelable& cancelable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(Buffer));
    NN_SDK_REQUIRES_EQUAL(reinterpret_cast<uintptr_t>(workBuffer) % static_cast<uint64_t>(std::alignment_of<Buffer>::value), 0);

    auto buffer = reinterpret_cast<Buffer*>(workBuffer);

    contextInitializer(sda.dataSize, sda.numOfPartitions);

    int cfNum = 0;

    fs::SaveDataTransferManagerForCloudBackUp manager;
    fs::SaveDataTransferManagerForCloudBackUp::Challenge challenge;
    NN_RESULT_DO(manager.GetChallenge(&challenge));

    fs::SaveDataTransferManagerForCloudBackUp::KeySeedPackage keySeedPackage;
    NN_RESULT_DO(olsc::srv::transfer::GetKeySeedPackage(
        &keySeedPackage, sda.id, challenge, nsaIdToken,
        curlHandle, buffer->ioBuffer.data(), buffer->ioBuffer.size(), &cancelable));

    NN_RESULT_DO(manager.SetKeySeedPackage(keySeedPackage));

    NN_RESULT_DO(RequestStartDownloadSaveDataArchive(
        &cfNum,
        buffer->componentFileListBuffer.data(),
        buffer->componentFileListBuffer.size(),
        sda.id, nsaIdToken,
        curlHandle, buffer->ioBuffer.data(), buffer->ioBuffer.size(), &cancelable));

    // InitialData を探す
    std::memset(&buffer->initialData, 0, sizeof(buffer->initialData));
    size_t initialDataSize = 0;
    auto endCf = buffer->componentFileListBuffer.begin() + cfNum;
    auto targetCf = std::find_if(buffer->componentFileListBuffer.begin(), endCf, [](olsc::srv::ComponentFileInfo i) {
        return i.clientArgument.chunkId == nn::fs::SaveDataChunkIdForInitialData;
    });

    NN_RESULT_THROW_UNLESS(targetCf != endCf, ResultServerDataChunkNotExist());

    // CF のサイズエラー
    NN_RESULT_THROW_UNLESS(targetCf->saveDataChunkSize == sizeof(fs::InitialDataForCloudBackUp), ResultComponentFileUnexpectedContentSize());
    // InitialData を単独で DL
    NN_RESULT_DO(transfer::RequestDownloadComponentFile(
        &initialDataSize,
        &buffer->initialData,
        sizeof(fs::InitialDataForCloudBackUp),
        targetCf->url,
        targetCf->saveDataChunkSize,
        curlHandle,
        &cancelable));

    std::unique_ptr<fs::ISaveDataDivisionImporter> importer;
    NN_RESULT_DO(manager.OpenSaveDataImporter(&importer, buffer->initialData, uid, fs::SaveDataSpaceId::User));

    NN_RESULT_DO(ImportSaveDataCore(
        nsaIdToken,
        curlHandle,
        *(importer.get()),
        buffer->componentFileListBuffer,
        cfNum, contextUpdater, buffer->ioBuffer.data(), buffer->ioBuffer.size(), cancelable));

    NN_RESULT_DO(RequestFinishDownloadSaveDataArchive(
        sda.id,nsaIdToken, curlHandle,
        buffer->ioBuffer.data(), buffer->ioBuffer.size(), &cancelable));

    NN_RESULT_SUCCESS;
}

}}}} //namespace nn::olsc::srv::transfer

