﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/olsc/sfdl/olsc_IOlscService.sfdl.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabaseManager.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/olsc_SystemEventManager.h>
#include <nn/olsc/srv/olsc_TransferTaskListControllerImpl.h>
#include <nn/olsc/srv/util/olsc_CancelListener.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_ObjectImplFactory.h>

namespace nn { namespace olsc { namespace srv {

namespace
{
    void DetailInfoToTransferTaskInfo(TransferTaskInfo* out, const TransferTaskDetailInfo& detailInfo, const TransferTaskBase* currentTask) NN_NOEXCEPT
    {
        using State = TransferTaskDetailInfo::State;
        TransferTaskInfo ti = {};
        ti.applicationId = detailInfo.appId;
        ti.id = detailInfo.id;
        ti.type = detailInfo.config.kind;
        ti.uid = detailInfo.uid;
        ti.rank = detailInfo.rank;

        switch (detailInfo.state)
        {
        case State::NotFinished:
            if (currentTask && currentTask->GetDetailInfo().id == detailInfo.id)
            {
                ti.status = TransferTaskStatus::Running;
            }
            else
            {
                ti.status = TransferTaskStatus::Runnable;
            }
            break;
        case State::Error:
            ti.status = TransferTaskStatus::Error;
            break;
        case State::Success:
            ti.status = TransferTaskStatus::Completed;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        *out = ti;
    }

    void TransferTaskContextToProgress(TransferTaskProgressForIpc* out, const TransferTaskBase::TransferTaskContext& context) NN_NOEXCEPT
    {
        TransferTaskProgressForIpc progress;
        progress.totalSize = context.totalSize;
        progress.completedSize = context.currentSize;
        progress.throughput = 0; // TODO

        *out = progress;
    }

    class ExecutionStopper
    {
    public:
        NN_IMPLICIT ExecutionStopper(TransferTaskAgent& agent) NN_NOEXCEPT;
        ~ExecutionStopper() NN_NOEXCEPT;
    private:
        TransferTaskAgent& m_Agent;
    };
}

Result TransferTaskListControllerImpl::GetTransferTaskCount(sf::Out<std::uint32_t> outValue) const NN_NOEXCEPT
{
    uint32_t count;
    NN_RESULT_DO(m_Ttdm.GetTransferTaskCount(&count));
    outValue.Set(count);

    NN_RESULT_SUCCESS;
}

Result TransferTaskListControllerImpl::GetTransferTaskInfo(sf::Out<TransferTaskInfo> outValue, TransferTaskId id) const NN_NOEXCEPT
{
    auto detailInfo = m_Ttdm.GetTransferTaskDetailInfo(id);
    NN_RESULT_THROW_UNLESS(detailInfo, olsc::ResultTransferTaskNotFoundInDatabase());

    auto taskLocker = m_Agent.LockTask();
    auto currentTask = m_Agent.GetCurrentTask();

    DetailInfoToTransferTaskInfo(outValue.GetPointer(), *detailInfo, currentTask);
    NN_RESULT_SUCCESS;
}

Result TransferTaskListControllerImpl::ListTransferTaskInfo(const sf::OutArray<TransferTaskInfo>& outValue, sf::Out<std::int32_t> outCount, std::int32_t offset) const NN_NOEXCEPT
{
    // TODO: 現状はとりあえず API とつなぎこむための仮実装。
    // Ttdm List 単位でしかロックされないので、整合性が崩れることがあるので正しい形にしなければならない。
    //   Ttdm::ForEachWithIndex 的なのを作って、λを渡して DetailInfo -> TransferTaskInfo とするのが良い気がしている。
    auto restCount = outValue.GetLength();
    auto currentOffset = offset;
    int currentOutOffset = 0;

    auto taskLocker = m_Agent.LockTask();
    auto currentTask = m_Agent.GetCurrentTask();

    while (restCount > 0)
    {
        auto listCount = std::min(static_cast<size_t>(m_ListBuffer.size()), restCount);
        auto listedCount = m_Ttdm.ListDetailInfo(&m_ListBuffer[0], static_cast<int>(listCount), currentOffset);
        if (listedCount == 0)
        {
            break;
        }

        for (int i = 0; i < listedCount; ++i)
        {
            auto& di = m_ListBuffer[i];
            DetailInfoToTransferTaskInfo(&outValue[currentOutOffset + i], di, currentTask);
        }
        currentOffset += listedCount;
        currentOutOffset += listedCount;
        restCount -= listedCount;
    }
    *outCount = currentOutOffset;

    NN_RESULT_SUCCESS;
}

Result TransferTaskListControllerImpl::DeleteTransferTask(TransferTaskId id) NN_NOEXCEPT
{
    SuspendTransferTask(id);
    NN_RESULT_DO(m_Ttdm.RemoveTransferTask(id));
    NN_RESULT_SUCCESS;
}

Result TransferTaskListControllerImpl::RaiseTransferTaskPriority(TransferTaskId id) NN_NOEXCEPT
{
    NN_RESULT_DO(m_Ttdm.MoveTransferTaskToHead(id));
    NN_RESULT_SUCCESS;
}

Result TransferTaskListControllerImpl::GetTransferTaskProgress(sf::Out<TransferTaskProgressForIpc> outValue, TransferTaskId id) const NN_NOEXCEPT
{
    // 実行中のタスクならメモリからコンテキストをひっぱる
    {
        auto taskLocker = m_Agent.LockTask();
        auto task = m_Agent.GetCurrentTask();
        if (task)
        {
            auto& context = task->GetContext();

            if (context.id == id)
            {
                TransferTaskContextToProgress(outValue.GetPointer(), context);
                NN_RESULT_SUCCESS;
            }
        }
    }

    // 実行中のタスクでない場合は　DB からコンテキストをひっぱって進捗を出す。
    // タスクが成功状態なら 100%
    // タスクが未完了状態なら context どおり
    // リトライ不可能エラー状態なら 0%

    auto detailInfo = m_Ttdm.GetTransferTaskDetailInfo(id);
    NN_RESULT_THROW_UNLESS(detailInfo, olsc::ResultTransferTaskNotFoundInDatabase());

    if (detailInfo->state == TransferTaskDetailInfo::State::Success)
    {
        outValue.Set(TransferTaskProgressForIpc::MakeCompletedProgress());
    }
    else if (detailInfo->state == TransferTaskDetailInfo::State::Error)
    {
        outValue.Set(TransferTaskProgressForIpc::MakeInitialProgress());
    }
    else if (detailInfo->state == TransferTaskDetailInfo::State::NotFinished)
    {
        auto context = m_Ttdm.GetTransferTaskContext(id);
        if (context)
        {
            TransferTaskContextToProgress(outValue.GetPointer(), *context);
        }
        else
        {
            outValue.Set(TransferTaskProgressForIpc::MakeInitialProgress());
        }
    }
    else
    {
        NN_ABORT("Not come here.\n");
    }

    NN_RESULT_SUCCESS;
}

Result TransferTaskListControllerImpl::GetTransferTaskLastResult(TransferTaskId id) const NN_NOEXCEPT
{
    // 実行中のタスクならメモリからコンテキストをひっぱる
    {
        auto taskLocker = m_Agent.LockTask();
        auto task = m_Agent.GetCurrentTask();
        if (task)
        {
            auto& context = task->GetContext();

            if (context.id == id)
            {
                NN_RESULT_THROW_UNLESS(context.lastResult.IsSuccess(), context.lastResult);
                NN_RESULT_SUCCESS;
            }
        }
    }

    // 実行中でないものは DB から取得する
    auto context = m_Ttdm.GetTransferTaskContext(id);
    if (context)
    {
        NN_RESULT_THROW_UNLESS(context->lastResult.IsSuccess(), context->lastResult);
    }

    NN_RESULT_SUCCESS;
}

Result TransferTaskListControllerImpl::DeleteAllTransferTask() NN_NOEXCEPT
{
    NN_RESULT_DO(m_Ttdm.RemoveAllTransferTask());

    NN_RESULT_SUCCESS;
}

Result TransferTaskListControllerImpl::StopNextTransferTaskExecution(sf::Out<sf::SharedPointer<IStopperObject>> outValue) NN_NOEXCEPT
{
    outValue.Set(sf::CreateSharedObjectEmplaced<IStopperObject, ExecutionStopper>(&m_ObjectMemoryResource, m_Agent));
    NN_RESULT_SUCCESS;
}

Result TransferTaskListControllerImpl::GetTransferTaskEndEventNativeHandleHolder(sf::Out<sf::SharedPointer<srv::INativeHandleHolder>> outValue) NN_NOEXCEPT
{
    auto holder = sf::CreateSharedObjectEmplaced<INativeHandleHolder, TransferTaskCompleteEventManager::NativeHandleHolder>(&m_ObjectMemoryResource, m_TaskCompleteEventManager);
    NN_RESULT_DO(holder.GetImpl().Initialize());
    outValue.Set(holder);
    NN_RESULT_SUCCESS;
}

Result TransferTaskListControllerImpl::GetTransferTaskStartEventNativeHandleHolder(sf::Out<sf::SharedPointer<srv::INativeHandleHolder>> outValue) NN_NOEXCEPT
{
    auto holder = sf::CreateSharedObjectEmplaced<INativeHandleHolder, TransferTaskCompleteEventManager::NativeHandleHolder>(&m_ObjectMemoryResource, m_TaskStartEventManager);
    NN_RESULT_DO(holder.GetImpl().Initialize());
    outValue.Set(holder);
    NN_RESULT_SUCCESS;
}

Result TransferTaskListControllerImpl::SuspendTransferTask(TransferTaskId id) NN_NOEXCEPT
{
    auto taskLocker = m_Agent.LockTask();
    auto currentTask = m_Agent.GetCurrentTask();
    if (currentTask && currentTask->GetDetailInfo().id == id)
    {
        currentTask->Cancel();
    }

    NN_RESULT_SUCCESS;
}

ExecutionStopper::ExecutionStopper(TransferTaskAgent& agent) NN_NOEXCEPT : m_Agent(agent)
{
    NN_DETAIL_OLSC_TRACE("StopTaskExecution\n");
    m_Agent.IncrementStopExecutionCount();
}

ExecutionStopper::~ExecutionStopper() NN_NOEXCEPT
{
    NN_DETAIL_OLSC_TRACE("UnstopTaskExecution\n");
    m_Agent.DecrementStopExecutionCount();
}

}}} //namespace nn::olsc::srv

