﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabase.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/util/util_TFormatString.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace olsc { namespace srv { namespace database {
namespace {

const int MaxPathLen = 64;

} // namespace

//------------------------------------------------------------------------------
// Private
//------------------------------------------------------------------------------

const char* TransferTaskDatabase::DetailInfoStore::GetMetadataFileRelativePath() const NN_NOEXCEPT
{
    return m_MetadataPath;
}

const char* TransferTaskDatabase::DetailInfoStore::GetEntryFileRelativePath() const NN_NOEXCEPT
{
    return m_EntryPath;
}

util::ReadMount TransferTaskDatabase::DetailInfoStore::AcquireReadMount(util::DefaultMountManager& mountManager) const NN_NOEXCEPT
{
    return mountManager.AcquireDeviceSaveForRead();
}

util::WriteMount TransferTaskDatabase::DetailInfoStore::AcquireWriteMount(util::DefaultMountManager& mountManager) NN_NOEXCEPT
{
    return mountManager.AcquireDeviceSaveForWrite();
}

const char* TransferTaskDatabase::MakeMetadataFilePath(char* out, size_t maxOutSize, const char* rootPath) NN_NOEXCEPT
{
    nn::util::TSNPrintf(out, maxOutSize, "%sttdb_meta", rootPath);
    return out;
}

Result TransferTaskDatabase::WriteMetadataFile() NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);

    auto writeMount = m_MountManager.AcquireDeviceSaveForWrite();

    char path[MaxPathLen];
    NN_RESULT_DO(util::WriteFile(MakeMetadataFilePath(path, sizeof(path), writeMount.GetRootPath()), &m_Metadata, sizeof(m_Metadata)));
    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabase::ReadMetadataFile() NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);

    auto readMount = m_MountManager.AcquireDeviceSaveForRead();

    char path[MaxPathLen];
    NN_RESULT_TRY(util::ReadFile(MakeMetadataFilePath(path, sizeof(path), readMount.GetRootPath()), &m_Metadata, sizeof(m_Metadata), 0))
        NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            m_Metadata = {};
        }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabase::GenerateId(TransferTaskId* out) NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    auto id = m_Metadata.currentId++;

    NN_RESULT_DO(WriteMetadataFile());

    *out = id;
    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabase::CreateDetailInfoEntry(DetailInfoEntry* out, const account::Uid& uid, ApplicationId appId, const TransferTaskConfig& config) NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);

    NN_RESULT_DO(GenerateId(&out->id));
    out->uid = uid;
    out->appId = appId;
    out->state = TransferTaskDetailInfo::State::NotFinished;
    out->config = config;

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabase::AddDetailInfoEntry(const DetailInfoEntry& detailInfoEntry) NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);

    auto holder = m_DetailInfoStoreManager.Acquire(detailInfoEntry.uid);
    NN_UTIL_LOCK_GUARD(holder.Get());

    NN_RESULT_DO(holder->PushBack(detailInfoEntry));

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabase::RemoveDetailInfoEntry(const account::Uid& uid, TransferTaskId id) NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);

    auto holder = m_DetailInfoStoreManager.Acquire(uid);
    NN_UTIL_LOCK_GUARD(holder.Get());

    return holder->RemoveIf([&id](const DetailInfoEntry& di) {
        return di.id == id;
    });
}

nn::util::optional<ExecutionList::ExecutionInfo> TransferTaskDatabase::FindExecutionInfo(TransferTaskId id) const NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<const ExecutionList> executionListLock(m_ExecutionInfoList);

    return m_ExecutionInfoList.FindIf([&id](const ExecutionList::ExecutionInfo& ei) {
        return ei.id == id;
    });
}

nn::util::optional<TransferTaskDatabase::DetailInfoEntry> TransferTaskDatabase::FindDetailInfoEntry(const account::Uid& uid, ApplicationId appId) const NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);

    auto holder = m_DetailInfoStoreManager.Acquire(uid);
    NN_UTIL_LOCK_GUARD(holder.Get());

    return holder->FindIf([&appId](const DetailInfoEntry& di) {
        return di.appId == appId;
    });
}

nn::util::optional<TransferTaskDatabase::DetailInfoEntry> TransferTaskDatabase::FindDetailInfoEntry(const account::Uid& uid, TransferTaskId id) const NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);

    auto holder = m_DetailInfoStoreManager.Acquire(uid);
    NN_UTIL_LOCK_GUARD(holder.Get());

    return holder->FindIf([&id](const DetailInfoEntry& di) {
        return di.id == id;
    });
}

//------------------------------------------------------------------------------
// Public
//------------------------------------------------------------------------------

Result TransferTaskDatabase::Initialize() NN_NOEXCEPT
{
    NN_RESULT_DO(ReadMetadataFile());
    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabase::RemoveTask(TransferTaskId id) NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<ExecutionList> executionListLock(m_ExecutionInfoList);

    auto removed = m_ExecutionInfoList.Remove(id);
    if (removed)
    {
        auto r = RemoveDetailInfoEntry(removed->uid, id);
        NN_RESULT_THROW_UNLESS(r.IsSuccess() || olsc::ResultDatabaseEntryNotFound::Includes(r), r);
        SignalExecutionListUpdatedEvent();
    }

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabase::MoveTaskToHead(TransferTaskId id) NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<ExecutionList> executionListLock(m_ExecutionInfoList);


    auto removed = m_ExecutionInfoList.Remove(id);
    NN_RESULT_THROW_UNLESS(removed, olsc::ResultTransferTaskNotFoundInExecutionList());

    // 削除したものを Basic リストの先頭に積みなおし
    removed->rank = TransferTaskRank::Basic;
    NN_RESULT_DO(m_ExecutionInfoList.PushFront(*removed));

    SignalExecutionListUpdatedEvent();

    NN_RESULT_SUCCESS;
}

int TransferTaskDatabase::ListTasks(TransferTaskDetailInfo out[], int maxOutCount, int offset) const NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<const ExecutionList> executionListLock(m_ExecutionInfoList);

    auto executionInfoCount = m_ExecutionInfoList.GetCount();

    int outCount = 0;
    for (int i = 0; i < maxOutCount && i + offset < executionInfoCount; ++i, outCount++)
    {
        ExecutionList::ExecutionInfo executionInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_ExecutionInfoList.At(&executionInfo, i + offset));

        auto die = FindDetailInfoEntry(executionInfo.uid, executionInfo.id);
        NN_ABORT_UNLESS(die);
        out[i] = TransferTaskDetailInfo::Make(die->id, die->uid, die->appId, die->config, die->state, executionInfo.rank);
    }

    return outCount;
}

nn::util::optional<TransferTaskDetailInfo> TransferTaskDatabase::FindTaskIf(const FindTaskPredicate& pred) const NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<const ExecutionList> executionListLock(m_ExecutionInfoList);

    auto count = m_ExecutionInfoList.GetCount();
    for (int i = 0; i < count; ++i)
    {
        ExecutionList::ExecutionInfo executionInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_ExecutionInfoList.At(&executionInfo, i));

        auto die = FindDetailInfoEntry(executionInfo.uid, executionInfo.id);
        NN_ABORT_UNLESS(die);
        auto detailInfo = TransferTaskDetailInfo::Make(die->id, die->uid, die->appId, die->config, die->state, executionInfo.rank);
        if (pred(detailInfo))
        {
            return detailInfo;
        }
    }
    return nn::util::nullopt;
}

int TransferTaskDatabase::GetTaskCount() const NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<const ExecutionList> executionListLock(m_ExecutionInfoList);

    return m_ExecutionInfoList.GetCount();
}

Result TransferTaskDatabase::GetTask(TransferTaskDetailInfo* out, TransferTaskId id) const NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);

    auto executionInfo = FindExecutionInfo(id);
    NN_RESULT_THROW_UNLESS(executionInfo, olsc::ResultTransferTaskNotFoundInExecutionList());

    auto die = FindDetailInfoEntry(executionInfo->uid, executionInfo->id);
    NN_RESULT_THROW_UNLESS(die, olsc::ResultTransferTaskNotFoundInDatabase());
    *out = TransferTaskDetailInfo::Make(die->id, die->uid, die->appId, die->config, die->state, executionInfo->rank);

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabase::CleanupUserData(const account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<ExecutionList> executionListLock(m_ExecutionInfoList);

    m_ExecutionInfoList.RemoveAll([&uid](const ExecutionList::ExecutionInfo& info)
    {
        return info.uid == uid;
    });

    auto holder = m_DetailInfoStoreManager.Acquire(uid);
    NN_UTIL_LOCK_GUARD(holder.Get());

    NN_RESULT_DO(holder->Cleanup());

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabase::RemoveTaskIfMatch(const account::Uid& uid, const std::function<bool(const TransferTaskDetailInfo& detailInfo)>& pred) NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<ExecutionList> executionListLock(m_ExecutionInfoList);

    auto holder = m_DetailInfoStoreManager.Acquire(uid);
    NN_UTIL_LOCK_GUARD(holder.Get());

    m_ExecutionInfoList.RemoveAll([&uid, &holder, &pred](const ExecutionList::ExecutionInfo& ei) -> bool {
        if (ei.uid != uid)
        {
            return false;
        }

        auto die = holder->FindIf([&ei](const DetailInfoEntry& die) -> bool {
            return ei.id == die.id;
        });
        NN_ABORT_UNLESS(die);
        auto detailInfo = TransferTaskDetailInfo::Make(die->id, die->uid, die->appId, die->config, die->state, ei.rank);
        if (pred(detailInfo))
        {
            holder->Remove(*die);
            return true;
        }
        return false;
    });

    NN_RESULT_SUCCESS;
}


Result TransferTaskDatabase::GetExecutionInfoByIndex(ExecutionList::ExecutionInfo* out, int index) const NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<const ExecutionList> executionListLock(m_ExecutionInfoList);

    NN_RESULT_DO(m_ExecutionInfoList.At(out, index));

    NN_RESULT_SUCCESS;
}

void TransferTaskDatabase::SignalExecutionListUpdatedEvent() NN_NOEXCEPT
{
    m_ExecutionListUpdatedEvent.Signal();
}

os::SystemEvent& TransferTaskDatabase::GetExecutionListUpdatedEvent() NN_NOEXCEPT
{
    return m_ExecutionListUpdatedEvent;
}

Result TransferTaskDatabase::UpdateTaskState(TransferTaskId id, TransferTaskDetailInfo::State state) NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);

    auto executionInfo = FindExecutionInfo(id);
    NN_RESULT_THROW_UNLESS(executionInfo, olsc::ResultTransferTaskNotFoundInExecutionList());

    auto uid = executionInfo->uid;

    auto detailInfo = FindDetailInfoEntry(uid, id);
    NN_ABORT_UNLESS(detailInfo);

    auto holder = m_DetailInfoStoreManager.Acquire(uid);
    NN_UTIL_LOCK_GUARD(holder.Get());

    detailInfo->state = state;
    NN_RESULT_DO(holder->ReplaceIf([&](const DetailInfoEntry& di) -> bool {
        return di.id == id;
    }, *detailInfo));

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabase::RemoveAllTask() NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<decltype(m_ExecutionInfoList)> executionInfoLock(m_ExecutionInfoList);

    // TODO: 1つのユーザに対する detailInfoStore を複数回削除してしまっている。最適化の余地あり。
    for (int i = 0; i < m_ExecutionInfoList.GetCount(); ++i)
    {
        ExecutionList::ExecutionInfo ei;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_ExecutionInfoList.At(&ei, i));

        auto uid = ei.uid;

        auto holder = m_DetailInfoStoreManager.Acquire(uid);
        NN_UTIL_LOCK_GUARD(holder.Get());

        holder->Cleanup();
    }

    NN_RESULT_DO(m_ExecutionInfoList.Cleanup());
    NN_RESULT_SUCCESS;
}

void TransferTaskDatabase::ForEachDetailInfo(const account::Uid& uid, const std::function<bool(const TransferTaskDetailInfo& detailInfo)>& pred) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    NN_UTIL_LOCK_GUARD(m_ExecutionInfoList);

    auto holder = m_DetailInfoStoreManager.Acquire(uid);
    NN_UTIL_LOCK_GUARD(holder.Get());

    m_ExecutionInfoList.ForEach([&holder, &uid, &pred](const ExecutionList::ExecutionInfo& ei) -> bool {
        if (ei.uid != uid)
        {
            return true;
        }

        auto die = holder->FindIf([&ei](const DetailInfoEntry& die) -> bool {
            return ei.id == die.id;
        });
        NN_ABORT_UNLESS(die);
        auto detailInfo = TransferTaskDetailInfo::Make(die->id, die->uid, die->appId, die->config, die->state, ei.rank);

        return pred(detailInfo);
    });
}

Result TransferTaskDatabase::AppendTask(TransferTaskId* out, const account::Uid& uid, ApplicationId appId, const TransferTaskConfig& config, TransferTaskRank rank) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    NN_UTIL_LOCK_GUARD(m_ExecutionInfoList);

    NN_ABORT_UNLESS(m_ExecutionInfoList.HasSpace(rank));
    NN_RESULT_THROW_UNLESS(!FindDetailInfoEntry(uid, appId), olsc::ResultTransferTaskDuplicated());

    DetailInfoEntry detailInfo;
    NN_RESULT_DO(CreateDetailInfoEntry(&detailInfo, uid, appId, config));

    ExecutionList::ExecutionInfo executionInfo =
    {
        detailInfo.id,
        detailInfo.uid,
        rank
    };
    NN_RESULT_DO(m_ExecutionInfoList.PushBack(executionInfo));

    NN_RESULT_DO(AddDetailInfoEntry(detailInfo));
    NN_RESULT_DO(WriteMetadataFile());

    *out = detailInfo.id;

    SignalExecutionListUpdatedEvent();

    NN_RESULT_SUCCESS;
}


}}}} //namespace nn::olsc::srv::database

