﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "olsc_SaveDataArchiveInfoAdaptor.h"

#include <nn/olsc/detail/olsc_DetailLog.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/olsc/srv/olsc_Util.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Base64.h>

namespace nn { namespace olsc { namespace srv { namespace adaptor {

namespace {
    const int MaxPropertyCount = SaveDataArchiveInfoBuilder::MaxPropertyCount;
    enum FlagIndex : int
    {
        FlagIndex_Id = 0,
        FlagIndex_NsaId,
        FlagIndex_ApplicationId,
        FlagIndex_UserId,
        FlagIndex_DeviceId,
        FlagIndex_DataSize,
        FlagIndex_AutoBackup,
        FlagIndex_HasThumbnail,
        FlagIndex_Digest,
        FlagIndex_Status,
        FlagIndex_LaunchRequiredVersion,
        FlagIndex_NumOfPartitions,
        FlagIndex_SeriesId,
        FlagIndex_SavedAt,
        FlagIndex_TimeoutAt,
        FlagIndex_FinishedAt,
        FlagIndex_CreatedAt,
        FlagIndex_UpdatedAt,
        FlagIndex_PropertyCount // Property 数
    };

    const char FieldString[][SaveDataArchiveInfoAdaptor::MaxPathLength] =
    {
        ".id",
        ".nsa_id",
        ".application_id",
        ".ua_id",
        ".device_id",
        ".data_size",
        ".auto_backup",
        ".has_thumbnail",
        ".encoded_digest",
        ".status",
        ".launch_required_version",
        ".num_of_partitions",
        ".series_id",
        ".saved_at_as_unixtime",
        ".timeout_at_as_unixtime",
        ".finished_at_as_unixtime",
        ".created_at_as_unixtime",
        ".updated_at_as_unixtime"
    };

    struct Flags
    {
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_Id> Id;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_NsaId> NsaId;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_ApplicationId> ApplicationId;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_UserId> UserId;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_DeviceId> DeviceId;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_DataSize> DataSize;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_AutoBackup> AutoBackup;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_HasThumbnail> HasThumbnail;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_Digest> Digest;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_Status> Status;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_LaunchRequiredVersion> LaunchRequiredVersion;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_NumOfPartitions> NumOfPartitions;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_SeriesId> SeriesId;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_SavedAt> SavedAt;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_TimeoutAt> TimeoutAt;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_FinishedAt> FinishedAt;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_CreatedAt> CreatedAt;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_UpdatedAt> UpdatedAt;
    };
} // namespace

SaveDataArchiveInfoBuilder::SaveDataArchiveInfoBuilder() NN_NOEXCEPT
{
    Reset();
}

void SaveDataArchiveInfoBuilder::Reset() NN_NOEXCEPT
{
    m_Imported.Reset();
    memset(&m_SaveDataArchiveInfo, 0x00, sizeof(SaveDataArchiveInfo));
}

bool SaveDataArchiveInfoBuilder::Validate() NN_NOEXCEPT
{
    // 必要な要素が全て設定されているかどうかの判定
    if(!(
        m_Imported[FlagIndex_Id] &&
        m_Imported[FlagIndex_NsaId] &&
        m_Imported[FlagIndex_ApplicationId] &&
        m_Imported[FlagIndex_Status] &&
        m_Imported[FlagIndex_DeviceId] &&
        m_Imported[FlagIndex_DataSize] &&
        m_Imported[FlagIndex_SeriesId] &&
        m_Imported[FlagIndex_Digest] &&
        m_Imported[FlagIndex_LaunchRequiredVersion] &&
        m_Imported[FlagIndex_NumOfPartitions] &&
        m_Imported[FlagIndex_SavedAt] ))
    {
        NN_DETAIL_OLSC_WARN("[Error] Sda Validate Error. below fields are not set.\n");
        for(int i = 0; i < FlagIndex_PropertyCount; i++)
        {
            if(!m_Imported[i])
            {
                NN_DETAIL_OLSC_WARN(" - %s\n", FieldString[i] + 1);
            }
        }
        return false;
    }
    return true;
}

const SaveDataArchiveInfo& SaveDataArchiveInfoBuilder::GetSaveDataArchiveInfo() const NN_NOEXCEPT
{
    return m_SaveDataArchiveInfo;
}

template<typename FlagType>
void SaveDataArchiveInfoBuilder::SetFlag() NN_NOEXCEPT
{
    m_Imported.template Set<FlagType>();
}

template<typename FlagType>
bool SaveDataArchiveInfoBuilder::IsUnset() const NN_NOEXCEPT
{
    return !m_Imported.template Test<FlagType>();
}

void SaveDataArchiveInfoBuilder::SetId(SaveDataArchiveId saveDataArchiveId) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.id = saveDataArchiveId;
    SetFlag<Flags::Id>();
}
void SaveDataArchiveInfoBuilder::SetNsaId(account::NetworkServiceAccountId nsaId) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.nsaId = nsaId;
    SetFlag<Flags::NsaId>();
}
void SaveDataArchiveInfoBuilder::SetApplicationId(ApplicationId applicationId) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.applicationId = applicationId;
    SetFlag<Flags::ApplicationId>();
}
void SaveDataArchiveInfoBuilder::SetUaId(const account::Uid& userId) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.userId = userId;
    SetFlag<Flags::UserId>();
}
void SaveDataArchiveInfoBuilder::SetDeviceId(DeviceId deviceId) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.deviceId = deviceId;
    SetFlag<Flags::DeviceId>();
}
void SaveDataArchiveInfoBuilder::SetDataSize(uint32_t dataSize) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.dataSize = dataSize;
    SetFlag<Flags::DataSize>();
}
void SaveDataArchiveInfoBuilder::SetAutoBackup(bool autoBackup) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.autoBackup = autoBackup;
    SetFlag<Flags::AutoBackup>();
}
void SaveDataArchiveInfoBuilder::SetHasThumbnail(bool hasThumbnail) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.hasThumbnail = hasThumbnail;
    SetFlag<Flags::HasThumbnail>();
}
void SaveDataArchiveInfoBuilder::SetEncodedDigest(const char digest[], size_t digestSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(digestSize == SaveDataArchiveInfo::DigestSize, "%s %s SizeError", __FILE__, __FUNCTION__);
    NN_UNUSED(digestSize);
    std::memcpy(&m_SaveDataArchiveInfo.seriesInfo.commitId, digest, sizeof(fs::SaveDataCommitId));
    SetFlag<Flags::Digest>();
}
void SaveDataArchiveInfoBuilder::SetStatus(SaveDataArchiveStatus status) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.status = status;
    SetFlag<Flags::Status>();
}
void SaveDataArchiveInfoBuilder::SetLaunchRequiredVersion(uint32_t launchRequiredVersion) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.launchRequiredVersion = launchRequiredVersion;
    SetFlag<Flags::LaunchRequiredVersion>();
}
void SaveDataArchiveInfoBuilder::SetNumOfPartitions(uint32_t numOfPartitions) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.numOfPartitions = numOfPartitions;
    SetFlag<Flags::NumOfPartitions>();
}
void SaveDataArchiveInfoBuilder::SetSeriesId(Bit64 seriesId) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.seriesInfo.seriesId = seriesId;
    SetFlag<Flags::SeriesId>();
}

void SaveDataArchiveInfoBuilder::SetSavedAtAsUnixTime(const time::PosixTime& time) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.savedAt = time;
    SetFlag<Flags::SavedAt>();
}
void SaveDataArchiveInfoBuilder::SetTimeoutAtAsUnixTime(const time::PosixTime& time) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.timeoutAt = time;
    SetFlag<Flags::TimeoutAt>();
}
void SaveDataArchiveInfoBuilder::SetFinishedAtAsUnixTime(const time::PosixTime& time) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.finishedAt = time;
    SetFlag<Flags::FinishedAt>();
}
void SaveDataArchiveInfoBuilder::SetCreatedAtAsUnixTime(const time::PosixTime& time) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.createdAt = time;
    SetFlag<Flags::CreatedAt>();
}
void SaveDataArchiveInfoBuilder::SetUpdatedAtAsUnixTime(const time::PosixTime& time) NN_NOEXCEPT
{
    m_SaveDataArchiveInfo.updatedAt = time;
    SetFlag<Flags::UpdatedAt>();
}


// ----------------------

bool SaveDataArchiveInfoAdaptor::UpdateImpl(const JsonPathType& jsonPath, bool value) NN_NOEXCEPT
{
    if(!m_IsAcceptable)
    {
        return true;
    }

    size_t fullPathLength = 0;
    const char* FullPath = jsonPath.ToString(&fullPathLength);
    NN_SDK_ASSERT(fullPathLength > static_cast<size_t>(m_PathBufferLength), "%s %s pathLength Error", __FILE__, __FUNCTION__);

    if(m_Builder.IsUnset<Flags::AutoBackup>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_AutoBackup], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetAutoBackup(value);
            return true;
        }
    }

    if(m_Builder.IsUnset<Flags::HasThumbnail>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_HasThumbnail], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetHasThumbnail(value);
            return true;
        }
    }
    return true;
}

bool SaveDataArchiveInfoAdaptor::UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    if(!m_IsAcceptable)
    {
        return true;
    }

    size_t fullPathLength = 0;
    const char* FullPath = jsonPath.ToString(&fullPathLength);
    NN_SDK_ASSERT(fullPathLength > static_cast<size_t>(m_PathBufferLength), "%s %s pathLength Error", __FILE__, __FUNCTION__);
    NN_SDK_ASSERT(nn::util::Strncmp(FullPath, m_PathBuffer, m_PathBufferLength) == 0, "JsonRootPath is wrong.");
    // integer 系
    if(m_Builder.IsUnset<Flags::Id>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_Id], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetId(static_cast<SaveDataArchiveId>(value));
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::DataSize>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_DataSize], MaxPathLength - m_PathBufferLength) == 0)
        {
            // size チェックは Validate でできないのでここで落とす
            m_IsAcceptable = (value <= std::numeric_limits<uint32_t>::max() && value > 0);
            if(m_IsAcceptable)
            {
                m_Builder.SetDataSize(static_cast<uint32_t>(value));
            }
            else
            {
                NN_DETAIL_OLSC_WARN("Sda parse error. '%s' is invalid value\n", FieldString[FlagIndex_DataSize]);
            }
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::LaunchRequiredVersion>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_LaunchRequiredVersion], MaxPathLength - m_PathBufferLength) == 0)
        {
            // size チェックは Validate でできないのでここで落とす
            m_IsAcceptable = value <= std::numeric_limits<uint32_t>::max();
            if(m_IsAcceptable)
            {
                m_Builder.SetLaunchRequiredVersion(static_cast<uint32_t>(value));
            }
            else
            {
                NN_DETAIL_OLSC_WARN("Sda parse error. '%s' is invalid value\n", FieldString[FlagIndex_LaunchRequiredVersion]);
            }
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::NumOfPartitions>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_NumOfPartitions], MaxPathLength - m_PathBufferLength) == 0)
        {
            // size チェックは Validate でできないのでここで落とす
            m_IsAcceptable = (value <= std::numeric_limits<uint32_t>::max() && value > 0);
            if(m_IsAcceptable)
            {
                m_Builder.SetNumOfPartitions(static_cast<uint32_t>(value));
            }
            else
            {
                NN_DETAIL_OLSC_WARN("Sda parse error. '%s' is invalid value\n", FieldString[FlagIndex_NumOfPartitions]);
            }
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::SeriesId>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_SeriesId], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetSeriesId(static_cast<Bit64>(value));
            return true;
        }
    }

    // datatime 系

    if(m_Builder.IsUnset<Flags::SavedAt>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_SavedAt], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetSavedAtAsUnixTime({value});
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::TimeoutAt>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_TimeoutAt], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetTimeoutAtAsUnixTime({value});
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::FinishedAt>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_FinishedAt], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetFinishedAtAsUnixTime({value});
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::CreatedAt>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_CreatedAt], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetCreatedAtAsUnixTime({value});
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::UpdatedAt>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_UpdatedAt], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetUpdatedAtAsUnixTime({value});
            return true;
        }
    }
    return true;
} // NOLINT(impl/function_size)

bool SaveDataArchiveInfoAdaptor::UpdateImpl(const JsonPathType& jsonPath, uint64_t value) NN_NOEXCEPT
{
    if(!m_IsAcceptable)
    {
        return true;
    }

    size_t fullPathLength = 0;
    const char* FullPath = jsonPath.ToString(&fullPathLength);
    NN_SDK_ASSERT(fullPathLength > static_cast<size_t>(m_PathBufferLength), "%s %s pathLength Error", __FILE__, __FUNCTION__);
    NN_SDK_ASSERT(nn::util::Strncmp(FullPath, m_PathBuffer, m_PathBufferLength) == 0, "JsonRootPath is wrong.");

    if(m_Builder.IsUnset<Flags::SeriesId>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_SeriesId], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetSeriesId(static_cast<Bit64>(value));
            return true;
        }
    }
    return true;
}

bool SaveDataArchiveInfoAdaptor::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if(!m_IsAcceptable)
    {
        return true;
    }

    size_t fullPathLength = 0;
    const char* FullPath = jsonPath.ToString(&fullPathLength);
    NN_SDK_ASSERT(fullPathLength > static_cast<size_t>(m_PathBufferLength), "%s %s pathLength Error", __FILE__, __FUNCTION__);

    if(m_Builder.IsUnset<Flags::NsaId>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_NsaId], MaxPathLength - m_PathBufferLength) == 0)
        {
            account::NetworkServiceAccountId nsaId = {static_cast<uint64_t>(std::strtoull(value, nullptr, 16))};
            m_Builder.SetNsaId(nsaId);
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::ApplicationId>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_ApplicationId], MaxPathLength - m_PathBufferLength) == 0)
        {
            ApplicationId applicationId = {std::strtoull(value, nullptr, 16)};
            // ApplicationId の値チェック
            m_IsAcceptable = (applicationId != nn::ApplicationId::GetInvalidId());
            if(m_IsAcceptable)
            {
                m_Builder.SetApplicationId(applicationId);
            }
            else
            {
                NN_DETAIL_OLSC_WARN("Sda parse error. '%s' is invalid value\n", FieldString[FlagIndex_ApplicationId]);
            }
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::DeviceId>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_DeviceId], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetDeviceId(std::strtoull(value, nullptr, 16));
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::Digest>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_Digest], MaxPathLength - m_PathBufferLength) == 0)
        {
            char hash[SaveDataArchiveInfo::DigestSize];
            size_t decodedLength;
            auto b64Error = nn::util::Base64::FromBase64String(&decodedLength, hash, sizeof(hash), value, nn::util::Base64::Mode_UrlSafe);

            // digest チェックは Validate でできないのでここで落とす
            m_IsAcceptable = (b64Error == nn::util::Base64::Status_Success) && (decodedLength == SaveDataArchiveInfo::DigestSize);

            if(m_IsAcceptable)
            {
                m_Builder.SetEncodedDigest(hash, sizeof(hash));
            }
            else
            {
                NN_DETAIL_OLSC_WARN("Sda parse error. '%s' is invalid value\n", FieldString[FlagIndex_Digest]);
            }
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::Status>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_Status], MaxPathLength - m_PathBufferLength) == 0)
        {
            if (nn::util::Strncmp(value, "fixed", valueLength + 1) == 0)
            {
                m_Builder.SetStatus(SaveDataArchiveStatus::Fixed);
            }
            else if(nn::util::Strncmp(value, "uploading", valueLength + 1) == 0)
            {
                m_Builder.SetStatus(SaveDataArchiveStatus::Uploading);
            }
            // 無効な文字列はパース失敗
            else
            {
                NN_DETAIL_OLSC_WARN("Sda parse error. '%s' is invalid value\n", FieldString[FlagIndex_Status]);
                m_IsAcceptable = false;
            }
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::UserId>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_UserId], MaxPathLength - m_PathBufferLength) == 0)
        {
            // TODO: 現状 64ビットの値が渡されることになっているが、正しくは 128ビット。
            //       サーバ側のフォーマット修正待ち
            NN_DETAIL_OLSC_WARN_DEV("ua_id parse is tentative. \n");
            account::Uid userId = { { 0, std::strtoull(value, nullptr, 16) } };
            // UserId の値チェック
            m_IsAcceptable = (userId != nn::account::InvalidUid);
            if(m_IsAcceptable)
            {
                m_Builder.SetUaId(userId);
            }
            else
            {
                NN_DETAIL_OLSC_WARN("Sda parse error. '%s' is invalid value\n", FieldString[FlagIndex_UserId]);
            }
            return true;
        }
    }
    return true;
} // NOLINT(impl/function_size)

bool SaveDataArchiveInfoAdaptor::NotifyObjectBeginImpl(const JsonPathType& jsonPath) NN_NOEXCEPT
{
    if(MakeAndCheckObjectPath(m_PathBuffer, sizeof(m_PathBuffer), jsonPath))
    {
        m_Builder.Reset();
        m_PathBufferLength = nn::util::Strnlen(m_PathBuffer, MaxPathLength);
        // パス長さが許容範囲外なら、Validate できないだけで処理は続行
        m_IsAcceptable = m_PathBufferLength < MaxPathLength;
        if(!m_IsAcceptable)
        {
            NN_DETAIL_OLSC_WARN("Sda parse error. pathLength is invalid\n");
        }
    }
    return true;
}

bool SaveDataArchiveInfoAdaptor::NotifyObjectEndImpl(const JsonPathType& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Match(m_PathBuffer))
    {
        m_PathBufferLength = 0;
        m_CurrentIndex++;

        if(m_IsAcceptable && m_Builder.Validate())
        {
            // Output に失敗した場合は続行不可と判断する
            auto result = m_OutputStream->Output(&m_Builder.GetSaveDataArchiveInfo(), 1);
            if(result.IsFailure())
            {
                SetResult(result);
                return false;
            }
        }
        // パース or Validate に失敗して、かつ単一 Sda の場合はエラー
        else if(m_IsSingleSda)
        {
            SetResult(ResultSaveDataArchiveUnacceptableContent());
            return false;
        }
        m_IsAcceptable = false;
    }
    return true;
}

bool SaveDataArchiveInfoAdaptor::MakeAndCheckObjectPath(char* buffer, size_t bufferSize, const JsonPathType& jsonPath) NN_NOEXCEPT
{
    // 複数 Sda が送られてくる場合は、 1 つ変な SDA があっても残りは取得できるようにエラーはスルー
    nn::util::TSNPrintf(buffer, bufferSize, "$.save_data_archives[%d]", m_CurrentIndex);
    if (jsonPath.Match(buffer))
    {
        m_IsSingleSda = false;
        return true;
    }

    // 1 つしか Sda が送られてこない場合は、 その SDA が変だったらエラーにする
    nn::util::TSNPrintf(buffer, bufferSize, "$.save_data_archive");
    if (jsonPath.Match(buffer))
    {
        m_IsSingleSda = true;
        return true;
    }
    return false;
}

}}}} //namespace nn::olsc::srv
