﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "olsc_ComponentFileInfoAdaptor.h"

#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/olsc/srv/olsc_Util.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Base64.h>

namespace nn { namespace olsc { namespace srv { namespace adaptor {

namespace {
    const int MaxPropertyCount = ComponentFileInfoBuilder::MaxPropertyCount;
    enum FlagIndex : int
    {
        FlagIndex_Id = 0,
        FlagIndex_SdaId,
        FlagIndex_ComponentFileSize,
        FlagIndex_SaveDataChunkSize,
        FlagIndex_Index,
        FlagIndex_Type,
        FlagIndex_Status,
        FlagIndex_ComponentFileDigest,
        FlagIndex_SaveDataChunkDigest,
        FlagIndex_CreatedAt,
        FlagIndex_UpdatedAt,
        FlagIndex_GetUrl,
        FlagIndex_PutUrl,
        FlagIndex_PropertyCount // Property 数
    };

    struct Flags
    {
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_Id> Id;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_SdaId> SdaId;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_ComponentFileSize> ComponentFileSize;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_SaveDataChunkSize> SaveDataChunkSize;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_Index> Index;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_Type> Type;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_Status> Status;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_ComponentFileDigest> ComponentFileDigest;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_SaveDataChunkDigest> SaveDataChunkDigest;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_CreatedAt> CreatedAt;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_UpdatedAt> UpdatedAt;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_GetUrl> GetUrl;
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_PutUrl> PutUrl;
    };

    const char FieldString[][ComponentFileInfoAdaptor::MaxPathLength] =
    {
        ".id",
        ".save_data_archive_id",
        ".raw_size",
        ".archive_size",
        ".index",
        ".datatype",
        ".status",
        ".encoded_raw_digest",
        ".encoded_archive_digest",
        ".created_at_as_unixtime",
        ".updated_at_as_unixtime",
        ".get_url",
        ".put_url",
    };

} // namespace

ComponentFileInfoBuilder::ComponentFileInfoBuilder() NN_NOEXCEPT
{
    Reset();
}

void ComponentFileInfoBuilder::Reset() NN_NOEXCEPT
{
    m_Imported.Reset();
    memset(&m_ComponentFileInfo, 0x00, sizeof(ComponentFileInfo));
}

bool ComponentFileInfoBuilder::Validate() NN_NOEXCEPT
{
    // 必要な要素が全て設定されているかどうかの判定
    if(!(
        m_Imported[FlagIndex_Id] &&
        m_Imported[FlagIndex_Index] &&
        m_Imported[FlagIndex_Type] &&
        m_Imported[FlagIndex_Status]))
    {
        NN_DETAIL_OLSC_WARN("[Error] Cf Validate Error. below fields are not set.\n");
        for(int i = 0; i < FlagIndex_PropertyCount; i++)
        {
            if(!m_Imported[i])
            {
                NN_DETAIL_OLSC_WARN(" - %s\n", FieldString[i] + 1);
            }
        }
        srv::DumpComponentFileInfo(m_ComponentFileInfo);
        return false;
    }
    return true;
}

const ComponentFileInfo& ComponentFileInfoBuilder::GetComponentFileInfo() const NN_NOEXCEPT
{
    return m_ComponentFileInfo;
}

template<typename FlagType>
void ComponentFileInfoBuilder::SetFlag() NN_NOEXCEPT
{
    m_Imported.template Set<FlagType>();
}

template<typename FlagType>
bool ComponentFileInfoBuilder::IsUnset() const NN_NOEXCEPT
{
    return !m_Imported.template Test<FlagType>();
}

void ComponentFileInfoBuilder::SetId(ComponentFileId componentFileId) NN_NOEXCEPT
{
    m_ComponentFileInfo.id = componentFileId;
    SetFlag<Flags::Id>();
}
void ComponentFileInfoBuilder::SetSaveDataArchiveId(SaveDataArchiveId saveDataArchiveId) NN_NOEXCEPT
{
    m_ComponentFileInfo.sdaId = saveDataArchiveId;
    SetFlag<Flags::SdaId>();
}
void ComponentFileInfoBuilder::SetComponentFileSize(size_t dataSize) NN_NOEXCEPT
{
    m_ComponentFileInfo.componentFileSize = dataSize;
    SetFlag<Flags::ComponentFileSize>();
}
void ComponentFileInfoBuilder::SetSaveDataChunkSize(size_t dataSize) NN_NOEXCEPT
{
    m_ComponentFileInfo.saveDataChunkSize = dataSize;
    SetFlag<Flags::SaveDataChunkSize>();
}
void ComponentFileInfoBuilder::SetEncodedComponentFileDigest(const ComponentFileDigest& digest) NN_NOEXCEPT
{
    m_ComponentFileInfo.componentFileDigest = digest;
    SetFlag<Flags::ComponentFileDigest>();
}
void ComponentFileInfoBuilder::SetEncodedSaveDataChunkDigest(const SaveDataChunkDigest& digest) NN_NOEXCEPT
{
    m_ComponentFileInfo.saveDataChunkDigest = digest;
    SetFlag<Flags::SaveDataChunkDigest>();
}
void ComponentFileInfoBuilder::SetIndex(nn::fs::SaveDataChunkId chunkId) NN_NOEXCEPT
{
    m_ComponentFileInfo.clientArgument.chunkId = chunkId;
    SetFlag<Flags::Index>();
}
void ComponentFileInfoBuilder::SetCreatedAtAsUnixTime(const time::PosixTime& time) NN_NOEXCEPT
{
    m_ComponentFileInfo.createdAt = time;
    SetFlag<Flags::CreatedAt>();
}
void ComponentFileInfoBuilder::SetUpdatedAtAsUnixTime(const time::PosixTime& time) NN_NOEXCEPT
{
    m_ComponentFileInfo.updatedAt = time;
    SetFlag<Flags::UpdatedAt>();
}
void ComponentFileInfoBuilder::SetDataType(ComponentFileType type) NN_NOEXCEPT
{
    m_ComponentFileInfo.type = type;
    SetFlag<Flags::Type>();
}
void ComponentFileInfoBuilder::SetStatus(ComponentFileStatus status) NN_NOEXCEPT
{
    m_ComponentFileInfo.status = status;
    SetFlag<Flags::Status>();
}
void ComponentFileInfoBuilder::SetGetUrl(const void* url, size_t urlLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT(sizeof(m_ComponentFileInfo.url) > urlLength + 1);
    NN_UNUSED(urlLength);
    nn::util::Strlcpy(m_ComponentFileInfo.url, reinterpret_cast<const char*>(url), sizeof(m_ComponentFileInfo.url));
    SetFlag<Flags::GetUrl>();
}
void ComponentFileInfoBuilder::SetPutUrl(const void* url, size_t urlLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT(sizeof(m_ComponentFileInfo.url) > urlLength + 1);
    NN_UNUSED(urlLength);
    nn::util::Strlcpy(m_ComponentFileInfo.url, reinterpret_cast<const char*>(url), sizeof(m_ComponentFileInfo.url));
    SetFlag<Flags::PutUrl>();
}


// ----------------------

bool ComponentFileInfoAdaptor::UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    if(!m_IsAcceptable)
    {
        return true;
    }

    size_t fullPathLength = 0;
    const char* FullPath = jsonPath.ToString(&fullPathLength);
    NN_SDK_ASSERT(fullPathLength > static_cast<size_t>(m_PathBufferLength), "%s %s pathLength Error", __FILE__, __FUNCTION__);
    NN_SDK_ASSERT(nn::util::Strncmp(FullPath, m_PathBuffer, m_PathBufferLength) == 0, "JsonRootPath is wrong.", __FILE__, __FUNCTION__);

    if(m_Builder.IsUnset<Flags::Id>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_Id], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetId(static_cast<ComponentFileId>(value));
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::SdaId>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_SdaId], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetSaveDataArchiveId(static_cast<SaveDataArchiveId>(value));
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::Index>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_Index], MaxPathLength - m_PathBufferLength) == 0)
        {
            // size チェックは Validate でできないのでここで落とす
            m_IsAcceptable = (value <= std::numeric_limits<uint32_t>::max() && value >= 0);
            if(m_IsAcceptable)
            {
                m_Builder.SetIndex(static_cast<nn::fs::SaveDataChunkId>(value));
            }
            else
            {
                NN_DETAIL_OLSC_WARN("Cf parse error. '%s' is invalid value\n", FieldString[FlagIndex_Index]);
            }
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::ComponentFileSize>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_ComponentFileSize], MaxPathLength - m_PathBufferLength) == 0)
        {
            // size チェックは Validate でできないのでここで落とす
            m_IsAcceptable = (value <= std::numeric_limits<uint32_t>::max() && value > 0);
            if(m_IsAcceptable)
            {
                m_Builder.SetComponentFileSize(static_cast<size_t>(value));
            }
            else
            {
                NN_DETAIL_OLSC_WARN("Cf parse error. '%s' is invalid value\n", FieldString[FlagIndex_ComponentFileSize]);
            }
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::SaveDataChunkSize>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_SaveDataChunkSize], MaxPathLength - m_PathBufferLength) == 0)
        {
            // size チェックは Validate でできないのでここで落とす
            m_IsAcceptable = (value <= std::numeric_limits<uint32_t>::max() && value > 0);
            if(m_IsAcceptable)
            {
                m_Builder.SetSaveDataChunkSize(static_cast<size_t>(value));
            }
            else
            {
                NN_DETAIL_OLSC_WARN("Cf parse error. '%s' is invalid value\n", FieldString[FlagIndex_SaveDataChunkSize]);
            }
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::CreatedAt>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_CreatedAt], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetCreatedAtAsUnixTime({value});
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::UpdatedAt>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_UpdatedAt], MaxPathLength - m_PathBufferLength) == 0)
        {
            m_Builder.SetUpdatedAtAsUnixTime({value});
            return true;
        }
    }
    return true;
} // NOLINT(impl/function_size)

bool ComponentFileInfoAdaptor::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if(!m_IsAcceptable)
    {
        return true;
    }
    size_t fullPathLength = 0;
    const char* FullPath = jsonPath.ToString(&fullPathLength);
    NN_SDK_ASSERT(fullPathLength > static_cast<size_t>(m_PathBufferLength), "%s %s pathLength Error", __FILE__, __FUNCTION__);
    NN_SDK_ASSERT(nn::util::Strncmp(FullPath, m_PathBuffer, m_PathBufferLength) == 0, "JsonRootPath is wrong.");

    if(m_Builder.IsUnset<Flags::ComponentFileDigest>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_ComponentFileDigest], MaxPathLength - m_PathBufferLength) == 0)
        {
            ComponentFileDigest digest;
            memset(digest.data, 0x00, sizeof(digest.data));
            size_t decodedLength;
            auto b64Error = nn::util::Base64::FromBase64String(&decodedLength, digest.data, sizeof(digest.data), value, nn::util::Base64::Mode_UrlSafe);
            // digest チェックは Validate でできないのでここで落とす
            m_IsAcceptable = (b64Error == nn::util::Base64::Status_Success) && (decodedLength == ComponentFileDigest::Size);

            if(m_IsAcceptable)
            {
                m_Builder.SetEncodedComponentFileDigest(digest);
            }
            else
            {
                NN_DETAIL_OLSC_WARN("Cf parse error. '%s' is invalid value\n", FieldString[FlagIndex_ComponentFileDigest]);
            }
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::SaveDataChunkDigest>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_SaveDataChunkDigest], MaxPathLength - m_PathBufferLength) == 0)
        {
            SaveDataChunkDigest digest;
            size_t decodedLength;
            auto b64Error = nn::util::Base64::FromBase64String(&decodedLength, digest.data, sizeof(digest.data), value, nn::util::Base64::Mode_UrlSafe);
            // digest チェックは Validate でできないのでここで落とす
            m_IsAcceptable = (b64Error == nn::util::Base64::Status_Success) && (decodedLength == SaveDataChunkDigest::Size);
            if(m_IsAcceptable)
            {
                m_Builder.SetEncodedSaveDataChunkDigest(digest);
            }
            else
            {
                NN_DETAIL_OLSC_WARN("Cf parse error. '%s' is invalid value\n", FieldString[FlagIndex_SaveDataChunkDigest]);
            }
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::Type>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_Type], MaxPathLength - m_PathBufferLength) == 0)
        {
            if (nn::util::Strncmp(value, "meta", valueLength + 1) == 0)
            {
                m_Builder.SetDataType(ComponentFileType::Meta);
            }
            else if(nn::util::Strncmp(value, "save", valueLength + 1) == 0)
            {
                m_Builder.SetDataType(ComponentFileType::Save);
            }
            // 無効な文字列はパース失敗
            else
            {
                NN_DETAIL_OLSC_WARN("Cf parse error. '%s' is invalid value\n", FieldString[FlagIndex_Type]);
                m_IsAcceptable = false;
            }
            return true;
        }
    }
    if(m_Builder.IsUnset<Flags::Status>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_Status], MaxPathLength - m_PathBufferLength) == 0)
        {
            if (nn::util::Strncmp(value, "fixed", valueLength + 1) == 0)
            {
                m_Builder.SetStatus(ComponentFileStatus::Fixed);
            }
            else if(nn::util::Strncmp(value, "uploading", valueLength + 1) == 0)
            {
                m_Builder.SetStatus(ComponentFileStatus::Uploading);
            }
            else if(nn::util::Strncmp(value, "hand_over", valueLength + 1) == 0)
            {
                m_Builder.SetStatus(ComponentFileStatus::HandOver);
            }
            // 無効な文字列はパース失敗
            else
            {
                NN_DETAIL_OLSC_WARN("Cf parse error. '%s' is invalid value\n", FieldString[FlagIndex_Status]);
                m_IsAcceptable = false;
            }
            return true;
        }
    }

    if(m_Builder.IsUnset<Flags::GetUrl>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_GetUrl], MaxPathLength - m_PathBufferLength) == 0)
        {
            const char CheckHead[] = "https://";
            // TODO : url の簡易チェック方法検討
            if(nn::util::Strncmp(CheckHead, value, sizeof(CheckHead) - 1) == 0)
            {
                m_Builder.SetGetUrl(value, valueLength);
            }
            return true;
        }
    }

    if(m_Builder.IsUnset<Flags::PutUrl>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_PutUrl], MaxPathLength - m_PathBufferLength) == 0)
        {
            const char CheckHead[] = "https://";
            // TODO : url の簡易チェック方法検討
            if(nn::util::Strncmp(CheckHead, value, sizeof(CheckHead) - 1) == 0)
            {
                m_Builder.SetPutUrl(value, valueLength);
            }
            return true;
        }
    }

    return true;
}// NOLINT(impl/function_size)

bool ComponentFileInfoAdaptor::NotifyObjectBeginImpl(const JsonPathType& jsonPath) NN_NOEXCEPT
{
    if(MakeAndCheckObjectPath(m_PathBuffer, sizeof(m_PathBuffer), jsonPath))
    {
        m_Builder.Reset();
        m_PathBufferLength = nn::util::Strnlen(m_PathBuffer, MaxPathLength);
        m_IsAcceptable = m_PathBufferLength < MaxPathLength;

        // パス長さが許容範囲外なら、以降の CF のパースは無意味なので止める
        if(!m_IsAcceptable)
        {
            SetResult(ResultComponentFileUnacceptableContent());
            return false;
        }
    }
    return true;
}

bool ComponentFileInfoAdaptor::NotifyObjectEndImpl(const JsonPathType& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Match(m_PathBuffer))
    {
        m_PathBufferLength = 0;
        m_CurrentIndex++;

        // パース or Validate に失敗したら、それ以降の CF のパースは無意味なので止める
        if(m_IsAcceptable && m_Builder.Validate())
        {
            // Output に失敗した場合は続行不可と判断する
            auto result = m_OutputStream->Output(&m_Builder.GetComponentFileInfo(), 1);
            if(result.IsFailure())
            {
                SetResult(result);
                return false;
            }
        }
        else
        {
            SetResult(ResultComponentFileUnacceptableContent());
            return false;
        }
        m_IsAcceptable = false;
    }
    return true;
}

bool ComponentFileInfoAdaptor::MakeAndCheckObjectPath(char* buffer, size_t bufferSize, const JsonPathType& jsonPath) const NN_NOEXCEPT
{
    nn::util::TSNPrintf(buffer, bufferSize, "$.save_data_archive.component_files[%d]", m_CurrentIndex);
    if (jsonPath.Match(buffer))
    {
        return true;
    }

    nn::util::TSNPrintf(buffer, bufferSize, "$.component_file");
    if (jsonPath.Match(buffer))
    {
        return true;
    }
    return false;
}


}}}} //namespace nn::olsc::srv
