﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "detail/olsc_Initialization.h"

#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_ApiForSystemService.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/olsc/olsc_TransferTaskListController.h>
#include <nn/olsc/olsc_TransferTaskTypesForPrivate.h>
#include <nn/olsc/sfdl/olsc_IOlscService.sfdl.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>

namespace nn { namespace olsc {

namespace {
    void ConvertTransferTaskProgress(TransferTaskProgress* out, const TransferTaskProgressForIpc& progressForIpc) NN_NOEXCEPT
    {
        out->completedSize = static_cast<size_t>(progressForIpc.completedSize);
        out->throughput = static_cast<size_t>(progressForIpc.throughput);
        out->totalSize = static_cast<size_t>(progressForIpc.totalSize);
    }

}

void swap(TransferTaskListController& lhs, TransferTaskListController& rhs) NN_NOEXCEPT
{
    std::swap(lhs.m_Interface, rhs.m_Interface);
}

TransferTaskListController::TransferTaskListController() NN_NOEXCEPT
    : TransferTaskListController(nullptr)
{
}

TransferTaskListController::TransferTaskListController(srv::ITransferTaskListController* interface) NN_NOEXCEPT
    : m_Interface(interface)
{
}

TransferTaskListController::TransferTaskListController(TransferTaskListController&& rhs) NN_NOEXCEPT
    : m_Interface(rhs.m_Interface)
{
    rhs.m_Interface = nullptr;
}

TransferTaskListController& TransferTaskListController::operator=(TransferTaskListController&& rhs) NN_NOEXCEPT
{
    TransferTaskListController t(std::move(rhs));
    swap(*this, t);
    return *this;
}


TransferTaskListController::~TransferTaskListController() NN_NOEXCEPT
{
    if (m_Interface)
    {
        sf::ReleaseSharedObject(m_Interface);
        m_Interface = nullptr;
    }
}

int TransferTaskListController::GetTransferTaskCount() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);

    uint32_t count;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetTransferTaskCount(&count));
    return count;
}

bool TransferTaskListController::GetTransferTask(TransferTaskInfo* out, TransferTaskId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);

    NN_RESULT_TRY(m_Interface->GetTransferTaskInfo(out, id))
        NN_RESULT_CATCH(olsc::ResultTransferTaskNotFound)
        {
            return false;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
        }
    NN_RESULT_END_TRY;

    return true;
}

int TransferTaskListController::ListTransferTask(TransferTaskInfo outValue[], int count, int offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    sf::OutArray<TransferTaskInfo> out(outValue, count);
    int32_t outCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->ListTransferTaskInfo(out, &outCount, offset));

    return outCount;
}

Result TransferTaskListController::DeleteTransferTask(TransferTaskId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    NN_RESULT_DO(m_Interface->DeleteTransferTask(id));

    NN_RESULT_SUCCESS;
}

Result TransferTaskListController::RaiseTransferTaskPriority(TransferTaskId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    NN_RESULT_DO(m_Interface->RaiseTransferTaskPriority(id));

    NN_RESULT_SUCCESS;
}

Result TransferTaskListController::GetTransferTaskEvent(SystemEventHolder* out) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    sf::SharedPointer<srv::INativeHandleHolder> handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetTransferTaskEndEventNativeHandleHolder(&handle));

    SystemEventHolder holder(handle.Detach());
    swap(*out, holder);

    NN_RESULT_SUCCESS;
}

SystemEventHolder TransferTaskListController::GetTransferTaskStartEvent() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    sf::SharedPointer<srv::INativeHandleHolder> handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetTransferTaskStartEventNativeHandleHolder(&handle));

    return SystemEventHolder(handle.Detach());
}

SystemEventHolder TransferTaskListController::GetTransferTaskEndEvent() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    sf::SharedPointer<srv::INativeHandleHolder> handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetTransferTaskEndEventNativeHandleHolder(&handle));

    return SystemEventHolder(handle.Detach());
}

Result TransferTaskListController::GetTransferTaskProgress(TransferTaskProgress* out, TransferTaskId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    TransferTaskProgressForIpc progress;
    NN_RESULT_DO(m_Interface->GetTransferTaskProgress(&progress, id));

    ConvertTransferTaskProgress(out, progress);
    NN_RESULT_SUCCESS;
}

Result TransferTaskListController::GetTransferTaskLastResult(TransferTaskId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);

    return m_Interface->GetTransferTaskLastResult(id);
}

void TransferTaskListController::SuspendTransferTask(TransferTaskId tid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->SuspendTransferTask(tid));
}

Stopper TransferTaskListController::StopNextTransferTaskExecution() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    sf::SharedPointer<srv::IStopperObject> s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->StopNextTransferTaskExecution(&s));

    return Stopper(s.Detach());
}

Stopper TransferTaskListController::StopTransferTaskExecution() NN_NOEXCEPT
{
    return StopNextTransferTaskExecution();
}


}} // namespace nn::olsc

