﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>

#include <nn/apm/apm.h>
#include <nn/oe.h>
#include <nn/oe/oe_BatteryDrainingPerformance.h>
#include <nn/oe/oe_HiddenPerformance.h>

namespace nn { namespace oe {

namespace {

struct ModeEntry
{
    PerformanceMode oeMode;
    nn::apm::PerformanceMode apmMode;
};

struct ConfigurationEntry
{
    PerformanceConfiguration oeConfig;
    nn::apm::PerformanceConfiguration apmConfig;
};

struct PriviledgedEntry
{
    nn::apm::PerformanceMode apmMode;
    nn::apm::PerformanceConfiguration apmConfig;
};

const ModeEntry ModeTable[] =
{
    { PerformanceMode_Normal, nn::apm::PerformanceMode_Normal },
    { PerformanceMode_Boost,  nn::apm::PerformanceMode_Boost  },
};

NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

const ConfigurationEntry ConfigurationTable[] =
{
    { PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1600Mhz, nn::apm::PerformanceConfiguration_Cpu1020MhzGpu307Mhz },
    { PerformanceConfiguration_Cpu1020MhzGpu384MhzEmc1600Mhz, nn::apm::PerformanceConfiguration_Cpu1020MhzGpu384Mhz },
    { PerformanceConfiguration_Cpu1020MhzGpu768MhzEmc1600Mhz, nn::apm::PerformanceConfiguration_Cpu1020MhzGpu768Mhz },
    { PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1331Mhz, nn::apm::PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1331Mhz },
    { PerformanceConfiguration_Cpu1020MhzGpu384MhzEmc1331Mhz, nn::apm::PerformanceConfiguration_Cpu1020MhzGpu384MhzEmc1331Mhz },
    { PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1065Mhz, nn::apm::PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1065Mhz },
    { PerformanceConfiguration_Cpu1020MhzGpu384MhzEmc1065Mhz, nn::apm::PerformanceConfiguration_Cpu1020MhzGpu384MhzEmc1065Mhz },
    { PerformanceConfiguration_Cpu1020MhzGpu460MhzEmc1600Mhz, nn::apm::PerformanceConfiguration_Cpu1020MhzGpu460MhzEmc1600Mhz },
    { PerformanceConfiguration_Cpu1020MhzGpu460MhzEmc1331Mhz, nn::apm::PerformanceConfiguration_Cpu1020MhzGpu460MhzEmc1331Mhz },
};

} // namespace

void SetPerformanceConfiguration(PerformanceConfiguration config) NN_NOEXCEPT
{
    SetPerformanceConfiguration(PerformanceMode_Normal, config);
    SetPerformanceConfiguration(PerformanceMode_Boost, config);
}

// 不正な引数を指定した場合、関数内でアボートする。
void SetPerformanceConfiguration(PerformanceMode mode, PerformanceConfiguration config) NN_NOEXCEPT
{
    nn::apm::PerformanceMode apmMode = nn::apm::PerformanceMode_Invalid;

    for ( const ModeEntry& entry : ModeTable )
    {
        if ( entry.oeMode == mode )
        {
            apmMode = entry.apmMode;
            break;
        }
    }

    // 無効な PerformanceMode を指定したときに確実に停止する。
    NN_ABORT_UNLESS(apmMode != nn::apm::PerformanceMode_Invalid, "Invalid performance mode!");

    nn::apm::PerformanceConfiguration apmConfig = nn::apm::PerformanceConfiguration_Invalid;

    for ( const ConfigurationEntry& entry : ConfigurationTable )
    {
        if ( entry.oeConfig == config )
        {
            apmConfig = entry.apmConfig;
            break;
        }
    }

    // 無効な PerformanceConfiguration を指定したときに確実に停止する。
    NN_ABORT_UNLESS(apmConfig != nn::apm::PerformanceConfiguration_Invalid, "Invalid performance configuration!");

    nn::apm::SetPerformanceConfiguration(apmMode, apmConfig);
}

// 不正な引数を指定した場合、関数内でアボートする。
PerformanceConfiguration GetPerformanceConfiguration(PerformanceMode mode) NN_NOEXCEPT
{
    nn::apm::PerformanceMode apmMode = nn::apm::PerformanceMode_Invalid;

    for ( const ModeEntry& entry : ModeTable )
    {
        if ( entry.oeMode == mode )
        {
            apmMode = entry.apmMode;
            break;
        }
    }

    // 無効な PerformanceMode を指定したときに確実に停止する。
    NN_ABORT_UNLESS(apmMode != nn::apm::PerformanceMode_Invalid, "Invalid performance mode!");

    nn::apm::PerformanceConfiguration apmConfig = nn::apm::GetPerformanceConfiguration(apmMode);

    for ( const ConfigurationEntry& entry : ConfigurationTable )
    {
        if ( entry.apmConfig == apmConfig )
        {
            return entry.oeConfig;
        }
    }

    // SetConfiguration を一度も実行していない場合 PerformanceConfiguration_Invalid を返す。
    return PerformanceConfiguration_Invalid;
}

}} // namespace nn::oe
