﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ntc/detail/service/ntc_Common.h>
#include <nn/ntc/detail/service/ntc_SystemReport.h>
#include <nn/srepo/srepo_SystemReport.h>

#include <nn/time/time_StandardSteadyClock.h>
#include <nn/time/time_ApiForRepair.h>
#include <nn/time/time_ApiForSystem.h>

namespace nn { namespace ntc { namespace detail { namespace service {

namespace
{
#if !defined(NN_BUILD_CONFIG_OS_WIN)

    const size_t ReportValueCountMax = 9;

    Result SetupSystemReport(nn::srepo::SystemReport* pOut) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);

        const size_t BufferSize = nn::srepo::SystemReport::BufferSizeMin + (nn::srepo::KeyValueSizeMax * ReportValueCountMax);
        NN_FUNCTION_LOCAL_STATIC(char, s_Buffer[BufferSize]);

        // 参照:https://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=111809427
        const nn::ApplicationId appId = {0x010000000000101F};
        NN_RESULT_DO(pOut->SetApplicationId(appId));

        pOut->SetBuffer(s_Buffer, BufferSize);

        NN_RESULT_SUCCESS;
    }

    Result SendCorrectionSystemReportImpl(
        const nn::time::SystemClockContext& oldContext,
        const nn::time::PosixTime& serverTime,
        const nn::TimeSpan& timeOfGettingServerPosixTime) NN_NOEXCEPT
    {
        nn::time::SystemClockContext currentContext;
        NN_RESULT_DO(nn::time::StandardNetworkSystemClock::GetSystemClockContext(&currentContext));

        // 前回補正からの経過時間を計算
        int64_t elapsedSeconds;
        auto result = nn::time::GetSpanBetween(&elapsedSeconds, oldContext.timeStamp, currentContext.timeStamp);
        if(result.IsFailure())
        {
            NN_RESULT_SUCCESS; // ネットワーク時計が未補正状態からの補正イベントはここではじかれる
        }

        // (過去の値-最新値) で、補正前にどれだけ進んでいたか(正ならRTCが実時間より進んでたことになる)
        int64_t diffSeconds = oldContext.offset - currentContext.offset;

        // RTC値に足しこむ内部オフセット
        nn::TimeSpan internalOffset = nn::time::GetStandardSteadyClockInternalOffset();

        // RTC値に足しこむテスト用オフセット
        nn::TimeSpan testOffset = nn::time::GetStandardSteadyClockTestOffset();

        // 現在の TimePoint から RTC 生値を計算してレポートする
        nn::time::SteadyClockTimePoint currentTimePoint;
        NN_RESULT_DO(nn::time::StandardSteadyClock::GetCurrentTimePoint(&currentTimePoint));
        int64_t rtcRawValue = currentTimePoint.value - internalOffset.GetSeconds() - testOffset.GetSeconds();

        // RTCセットアップ時のResult
        uint32_t setupResultValue = nn::time::GetStandardSteadyClockSetupResultValue();

        // レポート保存
        nn::srepo::SystemReport report("network_clock_correction");
        {
            NN_RESULT_DO(SetupSystemReport(&report));

            NN_RESULT_DO(report.Add("diff_seconds", diffSeconds));
            NN_RESULT_DO(report.Add("elapsed_seconds", elapsedSeconds));
            NN_RESULT_DO(report.Add("server_posix_time", serverTime.value));
            NN_RESULT_DO(report.Add("download_msec", timeOfGettingServerPosixTime.GetMilliSeconds()));
            NN_RESULT_DO(report.Add("context_offset_seconds", currentContext.offset));
            NN_RESULT_DO(report.Add("rtc_raw_seconds", rtcRawValue));
            NN_RESULT_DO(report.Add("rtc_internal_offset_seconds", internalOffset.GetSeconds()));
            NN_RESULT_DO(report.Add("rtc_test_offset_seconds", testOffset.GetSeconds()));
            NN_RESULT_DO(report.Add("rtc_setup_result", static_cast<int64_t>(setupResultValue)));

            NN_SDK_ASSERT_LESS_EQUAL(report.GetCount(), static_cast<int>(ReportValueCountMax));
        }
        return report.Save();
    }

    Result SendRtcResetSystemReportImpl() NN_NOEXCEPT
    {
        nn::srepo::SystemReport report("rtc_reset");

        NN_RESULT_DO(SetupSystemReport(&report));

        const int64_t count = 1;
        NN_RESULT_DO(report.Add("reset", count));

        NN_SDK_ASSERT_LESS_EQUAL(report.GetCount(), static_cast<int>(ReportValueCountMax));

        NN_DETAIL_NTC_SERVER_LOG("SendRtcResetSystemReportImpl\n");

        return report.Save();
    }
#endif
}

void SendCorrectionSystemReport(
    const nn::time::SystemClockContext& oldContext,
    const nn::time::PosixTime& serverTime,
    const nn::TimeSpan& timeOfGettingServerPosixTime) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    NN_UNUSED(oldContext);
    NN_UNUSED(serverTime);
    NN_UNUSED(timeOfGettingServerPosixTime);
    // do-nothing
#else
    auto result = SendCorrectionSystemReportImpl(oldContext, serverTime, timeOfGettingServerPosixTime);
    if(result.IsFailure())
    {
        NN_DETAIL_NTC_SERVER_LOG("SendCorrectionSystemReportImpl failed. (0x%08x)\n", result.GetInnerValueForDebug());
    }
#endif
}

void SendRtcResetSystemReport() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // do-nothing
#else
    auto result = SendRtcResetSystemReportImpl();
    if(result.IsFailure())
    {
        NN_DETAIL_NTC_SERVER_LOG("SendRtcResetSystemReportImpl failed. (0x%08x)\n", result.GetInnerValueForDebug());
    }
#endif
}

}}}} // nn::ntc::detail::service
