﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ntc/detail/service/ntc_HipcServerManager.h>
#include <nn/ntc/detail/service/ntc_SystemReport.h>
#include <nn/ntc/detail/service/core/ntc_PmModuleManager.h>
#include <nn/ntc/service/ntc_Service.h>
#include <nn/ntc/service/ntc_ServiceName.h>
#include <nn/time/time_StandardNetworkSystemClock.h>
#include <nn/time/time_ApiForSystem.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SystemThreadDefinition.h>

namespace nn { namespace ntc { namespace detail { namespace service {

namespace
{
    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_ThreadStackBuffer[1024 * 8];

    void DispatchFunction(void* pUserData) NN_NOEXCEPT
    {
        reinterpret_cast<HipcServerManager*>(pUserData)->DispatchFunctionImpl();
    }

    class MyPmModuleManager : public core::PmModuleManager
    {
    public:
        NN_IMPLICIT MyPmModuleManager(
            AutonomicEnsureNetworkClockAvailabilityThread* pAutonomicEnsureNetworkClockAvailabilityThread) NN_NOEXCEPT:
                m_pAutonomicEnsureNetworkClockAvailabilityThread(pAutonomicEnsureNetworkClockAvailabilityThread)
        {
        }

        virtual void FullAwakeCallback() NN_NOEXCEPT
        {
            if (m_pAutonomicEnsureNetworkClockAvailabilityThread)
            {
                // スリープ復帰時,ネットワーク時計が補正されてなければ補正処理をトリガー
                nn::time::PosixTime current;
                auto result = nn::time::StandardNetworkSystemClock::GetCurrentTime(&current);
                if (result.IsFailure())
                {
                    NN_DETAIL_NTC_SERVER_LOG("Execute immediate ensuring network clock.\n");
                    m_pAutonomicEnsureNetworkClockAvailabilityThread->ImmediatelyExecute();
                }
            }
        }
    private:
        AutonomicEnsureNetworkClockAvailabilityThread* m_pAutonomicEnsureNetworkClockAvailabilityThread;
    };
};

void HipcServerManager::StartThread() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        RegisterObjectForPort(
            m_StaticService.GetShared(), SessionCountMax, nn::ntc::service::ServiceName));

    this->Start();

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread(&g_Thread,
                                   DispatchFunction, this,
                                   g_ThreadStackBuffer, sizeof(g_ThreadStackBuffer),
                                   NN_SYSTEM_THREAD_PRIORITY(ntc, IpcServer)));

    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(ntc, IpcServer));

    nn::os::StartThread(&g_Thread);
}

void HipcServerManager::StopThread() NN_NOEXCEPT
{
    this->RequestStop();
    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);
}

void HipcServerManager::DispatchFunctionImpl() NN_NOEXCEPT
{
    // システム起動時、RTCリセット検知したら1度だけシステムレポートを送る
    if(nn::time::IsRtcResetDetected())
    {
        SendRtcResetSystemReport();
    }

    nn::os::MultiWaitHolderType pmModuleEventHolder;
    MyPmModuleManager myPmModuleManager(m_pAutonomicEnsureNetworkClockAvailabilityThread);


    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(myPmModuleManager.Initialize());

        nn::os::InitializeMultiWaitHolder(
            &pmModuleEventHolder, myPmModuleManager.GetPmModuleSystemEvent()->GetBase());

        nn::os::SetMultiWaitHolderUserData(
            &pmModuleEventHolder, nn::psc::PmModuleId_Ntc);

        this->AddUserWaitHolder(&pmModuleEventHolder);
    }

    while (NN_STATIC_CONDITION(true))
    {
        auto p = this->Wait();
        if (p == nullptr)
        {
            break;
        }
        switch (nn::os::GetMultiWaitHolderUserData(p))
        {
        case HipcServerManager::HipcSimpleAllInOneServerManager::AcceptTag:
        case HipcServerManager::HipcSimpleAllInOneServerManager::InvokeTag:
            {
                this->ProcessAuto(p);
            }
            break;
        case nn::psc::PmModuleId_Ntc:
            {
                myPmModuleManager.GetPmModuleSystemEvent()->Clear();
                this->AddUserWaitHolder(&pmModuleEventHolder);
                NN_ABORT_UNLESS_RESULT_SUCCESS(myPmModuleManager.HandlePmModule());
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    myPmModuleManager.Finalize();
}

}}}} // nn::ntc::detail::service
