﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ntc/detail/service/ntc_AutonomicEnsureConfig.h>

#include <nn/ntc/detail/service/ntc_Common.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/system/settings_Boot.h>

namespace nn { namespace ntc { namespace detail { namespace service {

namespace
{
#if !defined (NN_BUILD_CONFIG_OS_WIN)
    template <typename T>
    void GetNtcFwdbgSettings(T* pOut, const char* key) NN_NOEXCEPT
    {
        auto outBytes = nn::settings::fwdbg::GetSettingsItemValue(
            pOut,
            sizeof(T),
            "ntc",
            key);
        NN_ABORT_UNLESS_EQUAL(outBytes, sizeof(T));
    }
#endif
}

const bool IsAutonomicCorrectionEnabled() NN_NOEXCEPT
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    return true;
#else
    // http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-74412 参照
    // 修理現場で時計補正ツールの実行が(手違いで)行われなかった場合、FATAL プロセスがそれを検知してfatalエラーを発生させる。
    // (TIMEは別プロセスで動いており)このとき自動的な時刻補正が行われると、その後の時計補正ツールを実行する意味がなくなくなるのでここではじく。
    // 時計補正ツールが実行された後、このフラグは落ちそれ以降は正常に動作する。
    if(nn::settings::system::RequiresRunRepairTimeReviser())
    {
        NN_DETAIL_NTC_SERVER_LOG("Autonomic correction is disabled.(RequiresRunRepairTimeReviser:true)\n");
        return false;
    }

    bool enabled;
    GetNtcFwdbgSettings(&enabled, "is_autonomic_correction_enabled");
    return enabled;
#endif
}

const nn::TimeSpan GetAutonomicCorrectionIntervalTimeSpan() NN_NOEXCEPT
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    return nn::TimeSpan::FromDays(10);
#else
    uint32_t seconds;
    GetNtcFwdbgSettings(&seconds, "autonomic_correction_interval_seconds");
    return nn::TimeSpan::FromSeconds(static_cast<int64_t>(seconds));
#endif
}

const nn::TimeSpan GetAutonomicCorrectionRetryTimeSpan() NN_NOEXCEPT
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    return nn::TimeSpan::FromHours(1);
#else
    uint32_t seconds;
    GetNtcFwdbgSettings(&seconds, "autonomic_correction_failed_retry_interval_seconds");
    return nn::TimeSpan::FromSeconds(static_cast<int64_t>(seconds));
#endif
}

const int GetAutonomicCorrectionImmediateTryCountMax() NN_NOEXCEPT
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    return 4;
#else
    uint32_t countMax;
    GetNtcFwdbgSettings(&countMax, "autonomic_correction_immediate_try_count_max");
    return static_cast<int>(countMax);
#endif
}

const nn::TimeSpan GetAutonomicCorrectionImmediateTryTimeSpan() NN_NOEXCEPT
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    return nn::TimeSpan::FromMilliSeconds(5000);
#else
    uint32_t milliSeconds;
    GetNtcFwdbgSettings(&milliSeconds, "autonomic_correction_immediate_try_interval_milliseconds");
    return nn::TimeSpan::FromMilliSeconds(static_cast<int64_t>(milliSeconds));
#endif
}

}}}} // nn::ntc::detail::service
